/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package com.bsiag.scout.rt.ui.swing.rayo.window.desktop.toolbar;

import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.util.ArrayList;
import java.util.Collection;

import javax.swing.JDesktopPane;
import javax.swing.JInternalFrame;
import javax.swing.UIManager;
import javax.swing.event.InternalFrameAdapter;
import javax.swing.event.InternalFrameEvent;

import org.eclipse.scout.commons.NumberUtility;

import com.bsiag.scout.rt.ui.swing.rayo.window.desktop.RayoColumnSplitStrategy;

/**
 * Provides a single access point to the tools view, which is in fact not really a view but
 * a collection of JInternalFrames placed in the right most column of the MultiSplitLayout
 */
public class ToolsViewController {
  private JDesktopPane m_swingDesktop;
  private RayoColumnSplitStrategy m_columnSplitStrategy;

  private final Collection<ComponentListener> m_listeners = new ArrayList<ComponentListener>();
  private final ArrayList<JInternalFrame> m_frames = new ArrayList<JInternalFrame>();
  private final P_FrameListener m_frameListener = new P_FrameListener();
  private int m_minWidth;
  private int m_oldWidth;

  public ToolsViewController(JDesktopPane swingDesktop, RayoColumnSplitStrategy columnSplitStrategy) {
    m_swingDesktop = swingDesktop;
    m_columnSplitStrategy = columnSplitStrategy;
    m_minWidth = ensureMinWidth(0);
  }

  private void fireComponentResized(ComponentEvent e) {
    int newWidth = getWidth();
    if (m_oldWidth != newWidth) {
      m_oldWidth = newWidth;
      for (ComponentListener l : m_listeners) {
        l.componentResized(e);
      }
    }
  }

  public void addComponentListener(ComponentListener l) {
    m_listeners.add(l);
  }

  public void removeComponentListener(ComponentListener l) {
    m_listeners.remove(l);
  }

  public int getWidth() {
    if (m_frames.size() > 0) {
      // ensure to not return a width that does not meet the minWidth condition
      return ensureMinWidth(m_frames.get(m_frames.size() - 1).getWidth());
    }
    else {
      return 0;
    }
  }

  public void addFrame(JInternalFrame frame) {
    m_frames.add(frame);
    frame.addComponentListener(m_frameListener);
    frame.addInternalFrameListener(m_frameListener);
  }

  public void removeFrame(JInternalFrame frame) {
    if (m_frames.contains(frame)) {
      m_frames.remove(frame);
      frame.removeComponentListener(m_frameListener);
      frame.removeInternalFrameListener(m_frameListener);
    }
  }

  public void setMinimumWidth(int minWidth) {
    m_minWidth = ensureMinWidth(minWidth);
    // prevent view from being resized to a smaller width than the given value
    m_columnSplitStrategy.setToolsViewMinWidth(m_minWidth);
    m_swingDesktop.revalidate();
    m_swingDesktop.repaint();
  }

  public void expandView() {
    //set split position to minimal location
    if (m_minWidth > 0) {
      m_columnSplitStrategy.setToolsViewWidth(m_minWidth);
    }
    m_swingDesktop.revalidate();
    m_swingDesktop.repaint();
  }

  public void collapseView() {
    m_swingDesktop.revalidate();
    m_swingDesktop.repaint();
  }

  private int ensureMinWidth(int width) {
    return (int) NumberUtility.max(UIManager.getInt("Synth.ToolButtonView.minWidth"), m_minWidth, width);
  }

  private class P_FrameListener extends InternalFrameAdapter implements ComponentListener {
    @Override
    public void internalFrameOpened(InternalFrameEvent e) {
      if (e.getInternalFrame().isVisible()) {
        fireComponentResized(new ComponentEvent(e.getInternalFrame(), ComponentEvent.COMPONENT_RESIZED));
      }
    }

    @Override
    public void internalFrameClosed(InternalFrameEvent e) {
      removeFrame(e.getInternalFrame());
    }

    @Override
    public void componentResized(ComponentEvent e) {
      if (e.getComponent().isVisible()) {
        fireComponentResized(e);
      }
    }

    @Override
    public void componentMoved(ComponentEvent e) {
      if (e.getComponent().isVisible()) {
        fireComponentResized(e);
      }
    }

    @Override
    public void componentShown(ComponentEvent e) {
      if (e.getComponent().isVisible()) {
        fireComponentResized(e);
      }
    }

    @Override
    public void componentHidden(ComponentEvent e) {
    }
  }
}
