/*
 * Copyright (c) 2002, 2006, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.bsiag.javax.swing.plaf.synth;

import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.Rectangle;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.AbstractButton;
import javax.swing.ButtonModel;
import javax.swing.Icon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JToggleButton;
import javax.swing.LookAndFeel;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.UIResource;
import javax.swing.plaf.basic.BasicButtonUI;
import javax.swing.plaf.basic.BasicHTML;
import javax.swing.text.View;

import com.bsiag.sun.swing.plaf.synth.SynthUI;

/**
 * Synth's ButtonUI implementation.
 * 
 * @author Scott Violet
 */
class SynthButtonUI extends BasicButtonUI implements
    PropertyChangeListener, SynthUI {
  private SynthStyle style;

  public static ComponentUI createUI(JComponent c) {
    return new SynthButtonUI();
  }

  protected void installDefaults(AbstractButton b) {
    updateStyle(b);

    LookAndFeel.installProperty(b, "rolloverEnabled", Boolean.TRUE);
  }

  protected void installListeners(AbstractButton b) {
    super.installListeners(b);
    b.addPropertyChangeListener(this);
  }

  void updateStyle(AbstractButton b) {
    SynthContext context = getContext(b, SynthConstants.ENABLED);
    SynthStyle oldStyle = style;
    style = SynthLookAndFeel.updateStyle(context, this);
    if (style != oldStyle) {
      if (b.getMargin() == null || (b.getMargin() instanceof UIResource)) {
        Insets margin = (Insets) style.get(context, getPropertyPrefix() + "margin");

        if (margin == null) {
          // Some places assume margins are non-null.
          margin = SynthLookAndFeel.EMPTY_UIRESOURCE_INSETS;
        }
        b.setMargin(margin);
      }

      Object value = style.get(context, getPropertyPrefix() + "iconTextGap");
      if (value != null) {
        LookAndFeel.installProperty(b, "iconTextGap", value);
      }

      value = style.get(context, getPropertyPrefix() + "contentAreaFilled");
      LookAndFeel.installProperty(b, "contentAreaFilled", value != null ? value : Boolean.TRUE);

      if (oldStyle != null) {
        uninstallKeyboardActions(b);
        installKeyboardActions(b);
      }

    }
    context.dispose();
  }

  protected void uninstallListeners(AbstractButton b) {
    super.uninstallListeners(b);
    b.removePropertyChangeListener(this);
  }

  protected void uninstallDefaults(AbstractButton b) {
    SynthContext context = getContext(b, ENABLED);

    style.uninstallDefaults(context);
    context.dispose();
    style = null;
  }

  public SynthContext getContext(JComponent c) {
    return getContext(c, getComponentState(c));
  }

  SynthContext getContext(JComponent c, int state) {
    Region region = getRegion(c);
    return SynthContext.getContext(SynthContext.class, c, region,
        style, state);
  }

  private Region getRegion(JComponent c) {
    return SynthLookAndFeel.getRegion(c);
  }

  /**
   * Returns the current state of the passed in <code>AbstractButton</code>.
   */
  private int getComponentState(JComponent c) {
    int state = ENABLED;

    if (!c.isEnabled()) {
      state = DISABLED;
    }
    if (SynthLookAndFeel.selectedUI == this) {
      return SynthLookAndFeel.selectedUIState | SynthConstants.ENABLED;
    }
    AbstractButton button = (AbstractButton) c;
    ButtonModel model = button.getModel();

    if (model.isPressed()) {
      if (model.isArmed()) {
        state = PRESSED;
      }
      else {
        state = MOUSE_OVER;
      }
    }
    if (model.isRollover()) {
      state |= MOUSE_OVER;
    }
    if (model.isSelected()) {
      state |= SELECTED;
    }
    if (c.isFocusOwner() && button.isFocusPainted()) {
      state |= FOCUSED;
    }
    if ((c instanceof JButton) && ((JButton) c).isDefaultButton()) {
      state |= DEFAULT;
    }
    return state;
  }

  public int getBaseline(JComponent c, int width, int height) {
    if (c == null) {
      throw new NullPointerException("Component must be non-null");
    }
    if (width < 0 || height < 0) {
      throw new IllegalArgumentException(
           "Width and height must be >= 0");
    }
    AbstractButton b = (AbstractButton) c;
    String text = b.getText();
    if (text == null || "".equals(text)) {
      return -1;
    }
    Insets i = b.getInsets();
    Rectangle viewRect = new Rectangle();
    Rectangle textRect = new Rectangle();
    Rectangle iconRect = new Rectangle();
    viewRect.x = i.left;
    viewRect.y = i.top;
    viewRect.width = width - (i.right + viewRect.x);
    viewRect.height = height - (i.bottom + viewRect.y);

    // layout the text and icon
    SynthContext context = getContext(b);
    FontMetrics fm = context.getComponent().getFontMetrics(
         context.getStyle().getFont(context));
    context.getStyle().getGraphicsUtils(context).layoutText(
         context, fm, b.getText(), b.getIcon(),
         b.getHorizontalAlignment(), b.getVerticalAlignment(),
         b.getHorizontalTextPosition(), b.getVerticalTextPosition(),
         viewRect, iconRect, textRect, b.getIconTextGap());
    View view = (View) b.getClientProperty(BasicHTML.propertyKey);
    int baseline;
    if (view != null) {
      baseline = BasicHTML.getHTMLBaseline(view, textRect.width,
           textRect.height);
      if (baseline >= 0) {
        baseline += textRect.y;
      }
    }
    else {
      baseline = textRect.y + fm.getAscent();
    }
    context.dispose();
    return baseline;
  }

  // ********************************
  //          Paint Methods
  // ********************************

  public void update(Graphics g, JComponent c) {
    SynthContext context = getContext(c);

    SynthLookAndFeel.update(context, g);
    paintBackground(context, g, c);
    paint(context, g);
    context.dispose();
  }

  public void paint(Graphics g, JComponent c) {
    SynthContext context = getContext(c);

    paint(context, g);
    context.dispose();
  }

  protected void paint(SynthContext context, Graphics g) {
    AbstractButton b = (AbstractButton) context.getComponent();

    g.setColor(context.getStyle().getColor(context,
         ColorType.TEXT_FOREGROUND));
    g.setFont(style.getFont(context));
    context.getStyle().getGraphicsUtils(context).paintText(
         context, g, b.getText(), getIcon(b),
         b.getHorizontalAlignment(), b.getVerticalAlignment(),
         b.getHorizontalTextPosition(), b.getVerticalTextPosition(),
         b.getIconTextGap(), b.getDisplayedMnemonicIndex(),
         getTextShiftOffset(context));
  }

  void paintBackground(SynthContext context, Graphics g, JComponent c) {
    if (((AbstractButton) c).isContentAreaFilled()) {
      context.getPainter().paintButtonBackground(context, g, 0, 0,
           c.getWidth(),
           c.getHeight());
    }
  }

  public void paintBorder(SynthContext context, Graphics g, int x,
       int y, int w, int h) {
    context.getPainter().paintButtonBorder(context, g, x, y, w, h);
  }

  /**
   * Returns the default icon. This should NOT callback
   * to the JComponent.
   * 
   * @param b
   *          AbstractButton the icon is associated with
   * @return default icon
   */

  protected Icon getDefaultIcon(AbstractButton b) {
    SynthContext context = getContext(b);
    Icon icon = context.getStyle().getIcon(context, getPropertyPrefix() + "icon");
    context.dispose();
    return icon;
  }

  /**
   * Returns the Icon to use in painting the button.
   */
  protected Icon getIcon(AbstractButton b) {
    Icon icon = null;
    Icon defaultUserIcon = b.getIcon();
    ButtonModel model = b.getModel();

    if (!model.isEnabled()) {
      icon = getDisabledIcon(b, defaultUserIcon, defaultUserIcon);
    }
    else if (model.isPressed() && model.isArmed()) {
      icon = getPressedIcon(b, getSelectedIcon(b, defaultUserIcon));
    }
    else if (model.isRollover() && b.isRolloverEnabled()) {
      icon = getRolloverIcon(b, getSelectedIcon(b, defaultUserIcon), defaultUserIcon);
    }
    else if (model.isSelected()) {
      icon = getSelectedIcon(b, defaultUserIcon);
    }
    else {
      if (b instanceof JToggleButton) {
        // For JToggleButtons (e.g. Checkboxes), we need to respect the configured icons for the different Synth states (e.g. FOCUSED).
        icon = getEnabledIcon(b, null); // defaultIcon is null by intention to return immediately if no user icon is provided.
        if (icon == null) {
          icon = getDefaultIcon(b, defaultUserIcon); // resolve Rayo L&F icon in respect to diffent Synth states such as FOCUSED.
        }
      }
      else {
        icon = getEnabledIcon(b, defaultUserIcon);
      }
    }
    if (icon == null) {
      return getDefaultIcon(b);
    }
    return icon;
  }

  /**
   * (awe, ticket #94968) the original call to getDefaultIcon() for JToggleButtons ignores the icon set directly on the
   * button instance. This causes the icon to vanish in the UI.
   * 
   * @param b
   * @param defaultIcon
   * @return
   */
  private Icon getDefaultIcon(AbstractButton b, Icon defaultIcon) {
    Icon icon = getDefaultIcon(b);
    if (icon == null) {
      icon = defaultIcon;
    }
    return icon;
  }

  /**
   * <p>
   * This method will return the icon that should be used for a button. We only want to use the synth icon defined by
   * the style if the specific icon has not been defined for the button state and the backup icon is a UIResource (we
   * set it, not the developer).
   * </p>
   * <b>Resolution:</b>
   * <ul>
   * <li>1. priority: user icon if not null</li>
   * <li>2. priority: if defaultIcon == null, null is returned</li>
   * <li>3. priority: if defaultIcon originates from a L&F provider (instance of {@link UIResource}), not the default
   * icon is considered, but the icon of Rayo L&F. That is to overwrite a L&F default. If null, go to priorty 4.</li>
   * <li>4. priority: default icon is returned</li>
   * </ul>
   * 
   * @param b
   *          button
   * @param userIcon
   *          icon directly set on the button for the specified state
   * @param defaultIcon
   *          fallback icon in specificIcon is null. If null, the icon configured in the style is considered
   * @param state
   *          the synth state of the button
   */
  private Icon getIcon(AbstractButton b, Icon userIcon, Icon defaultIcon, int state) {
    if (userIcon != null) {
      return userIcon;
    }
    if (defaultIcon == null) {
      return null;
    }
    if (defaultIcon instanceof UIResource) {
      // in case the default icon is provided by a L&F provider, consider the R&F icon in first priority
      Icon synthIcon = getSynthIcon(b, state);
      if (synthIcon != null) {
        return synthIcon;
      }
    }

    return defaultIcon;
  }

  private Icon getSynthIcon(AbstractButton b, int synthConstant) {
    return style.getIcon(getContext(b, synthConstant), getPropertyPrefix() + "icon");
  }

  /**
   * See {@link SynthButtonUI#getIcon(AbstractButton, Icon, Icon, int)} for resolution details.
   * 
   * @param b
   * @param defaultIcon
   * @return
   */
  private Icon getEnabledIcon(AbstractButton b, Icon defaultIcon) {
    return getIcon(b, b.getIcon(), defaultIcon, SynthConstants.ENABLED);
  }

  /**
   * See {@link SynthButtonUI#getIcon(AbstractButton, Icon, Icon, int)} for resolution details.
   * 
   * @param b
   * @param defaultIcon
   * @return
   */
  private Icon getSelectedIcon(AbstractButton b, Icon defaultIcon) {
    return getIcon(b, b.getSelectedIcon(), defaultIcon,
          SynthConstants.SELECTED);
  }

  /**
   * See {@link SynthButtonUI#getIcon(AbstractButton, Icon, Icon, int)} for resolution details.
   * 
   * @param b
   * @param defaultSelectedIcon
   * @param defaultUnselectedIcon
   * @return
   */
  private Icon getRolloverIcon(AbstractButton b, Icon defaultSelectedIcon, Icon defaultUnselectedIcon) {
    ButtonModel model = b.getModel();
    Icon icon;
    if (model.isSelected()) {
      icon = getIcon(b, b.getRolloverSelectedIcon(), defaultSelectedIcon,
            SynthConstants.MOUSE_OVER | SynthConstants.SELECTED);
    }
    else {
      icon = getIcon(b, b.getRolloverIcon(), defaultUnselectedIcon,
            SynthConstants.MOUSE_OVER);
    }
    return icon;
  }

  /**
   * See {@link SynthButtonUI#getIcon(AbstractButton, Icon, Icon, int)} for resolution details.
   * 
   * @param b
   * @param defaultIcon
   * @return
   */
  private Icon getPressedIcon(AbstractButton b, Icon defaultIcon) {
    return getIcon(b, b.getPressedIcon(), defaultIcon,
          SynthConstants.PRESSED);
  }

  /**
   * See {@link SynthButtonUI#getIcon(AbstractButton, Icon, Icon, int)} for resolution details.
   * 
   * @param b
   * @param defaultSelectedIcon
   * @param defaultUnselectedIcon
   * @return
   */
  private Icon getDisabledIcon(AbstractButton b, Icon defaultSelectedIcon, Icon defaultUnselectedIcon) {
    ButtonModel model = b.getModel();
    Icon icon;
    if (model.isSelected()) {
      icon = getIcon(b, b.getDisabledSelectedIcon(), defaultSelectedIcon,
            SynthConstants.DISABLED | SynthConstants.SELECTED);
    }
    else {
      icon = getIcon(b, b.getDisabledIcon(), defaultUnselectedIcon,
            SynthConstants.DISABLED);
    }
    return icon;
  }

  /**
   * Returns the amount to shift the text/icon when painting.
   */
  protected int getTextShiftOffset(SynthContext state) {
    AbstractButton button = (AbstractButton) state.getComponent();
    ButtonModel model = button.getModel();

    if (model.isArmed() && model.isPressed() &&
          button.getPressedIcon() == null) {
      return state.getStyle().getInt(state, getPropertyPrefix() +
            "textShiftOffset", 0);
    }
    return 0;
  }

  // ********************************
  //          Layout Methods
  // ********************************
  public Dimension getMinimumSize(JComponent c) {
    if (c.getComponentCount() > 0 && c.getLayout() != null) {
      return null;
    }
    AbstractButton b = (AbstractButton) c;
    SynthContext ss = getContext(c);
    Dimension size = ss.getStyle().getGraphicsUtils(ss).getMinimumSize(
          ss, ss.getStyle().getFont(ss), b.getText(), getSizingIcon(b),
          b.getHorizontalAlignment(), b.getVerticalAlignment(),
          b.getHorizontalTextPosition(),
          b.getVerticalTextPosition(), b.getIconTextGap(),
          b.getDisplayedMnemonicIndex());

    ss.dispose();
    return size;
  }

  public Dimension getPreferredSize(JComponent c) {
    if (c.getComponentCount() > 0 && c.getLayout() != null) {
      return null;
    }
    AbstractButton b = (AbstractButton) c;
    SynthContext ss = getContext(c);
    Dimension size = ss.getStyle().getGraphicsUtils(ss).getPreferredSize(
          ss, ss.getStyle().getFont(ss), b.getText(), getSizingIcon(b),
          b.getHorizontalAlignment(), b.getVerticalAlignment(),
          b.getHorizontalTextPosition(),
          b.getVerticalTextPosition(), b.getIconTextGap(),
          b.getDisplayedMnemonicIndex());

    ss.dispose();
    return size;
  }

  public Dimension getMaximumSize(JComponent c) {
    if (c.getComponentCount() > 0 && c.getLayout() != null) {
      return null;
    }

    AbstractButton b = (AbstractButton) c;
    SynthContext ss = getContext(c);
    Dimension size = ss.getStyle().getGraphicsUtils(ss).getMaximumSize(
          ss, ss.getStyle().getFont(ss), b.getText(), getSizingIcon(b),
          b.getHorizontalAlignment(), b.getVerticalAlignment(),
          b.getHorizontalTextPosition(),
          b.getVerticalTextPosition(), b.getIconTextGap(),
          b.getDisplayedMnemonicIndex());

    ss.dispose();
    return size;
  }

  /**
   * Returns the Icon used in calculating the pref/min/max size.
   */
  protected Icon getSizingIcon(AbstractButton b) {
    Icon icon = getEnabledIcon(b, b.getIcon());
    if (icon == null) {
      icon = getDefaultIcon(b);
    }
    return icon;
  }

  public void propertyChange(PropertyChangeEvent e) {
    if (SynthLookAndFeel.shouldUpdateStyle(e)) {
      updateStyle((AbstractButton) e.getSource());
    }
  }
}
