/*
 * Copyright (c) 2011, BSI Business Systems Integration AG. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. BSI Business Systems Integration AG
 * designates this particular file as subject to the "Classpath" exception as provided
 * by BSI Business Systems Integration AG in the LICENSE_BSI file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact BSI Business Systems Integration AG, Taefernstrasse 16a,
 * CH-5405 Baden, Switzerland or visit www.bsiag.com if you need additional
 * information or have any questions.
 */
package com.bsiag.javax.swing.plaf.synth;

import java.awt.Component;
import java.awt.Cursor;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Frame;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.InputEvent;
import java.awt.event.MouseEvent;

import javax.swing.JComponent;
import javax.swing.JRootPane;
import javax.swing.SwingUtilities;
import javax.swing.event.MouseInputAdapter;

/**
 * MouseInputHandler is responsible for handling resize/moving of the Window. It
 * sets the cursor directly on the Window when then mouse moves over a hot spot.
 */
class MouseInputHandler extends MouseInputAdapter {

  /**
   * The amount of space (in pixels) that the cursor is changed on.
   */
  private static final int CORNER_DRAG_WIDTH = 4;

  /**
   * Region from edges that dragging is active from.
   */
  private static final int BORDER_DRAG_THICKNESS = 4; // bsh: original value was 2

  /**
   * Maps from positions to cursor type. Refer to calculateCorner and
   * calculatePosition for details of this.
   */
  private static final int[] cursorMapping = new int[]{
      Cursor.NW_RESIZE_CURSOR, Cursor.NW_RESIZE_CURSOR,
      Cursor.N_RESIZE_CURSOR, Cursor.NE_RESIZE_CURSOR,
      Cursor.NE_RESIZE_CURSOR, Cursor.NW_RESIZE_CURSOR, 0, 0, 0,
      Cursor.NE_RESIZE_CURSOR, Cursor.W_RESIZE_CURSOR, 0, 0, 0,
      Cursor.E_RESIZE_CURSOR, Cursor.SW_RESIZE_CURSOR, 0, 0, 0,
      Cursor.SE_RESIZE_CURSOR, Cursor.SW_RESIZE_CURSOR,
      Cursor.SW_RESIZE_CURSOR, Cursor.S_RESIZE_CURSOR,
      Cursor.SE_RESIZE_CURSOR, Cursor.SE_RESIZE_CURSOR};

  private final SynthRootPaneUI ui;
  private boolean isPressed;

  /**
   * Set to true if the drag operation is moving the window.
   */
  private boolean isMovingWindow;

  /**
   * Used to determine the corner the resize is occuring from.
   */
  private int dragCursor;

  /**
   * X location the mouse went down on for a drag operation.
   */
  private int dragOffsetX;

  /**
   * Y location the mouse went down on for a drag operation.
   */
  private int dragOffsetY;

  /**
   * Width of the window when the drag started.
   */
  private int dragWidth;

  /**
   * Height of the window when the drag started.
   */
  private int dragHeight;

  MouseInputHandler(SynthRootPaneUI ui) {
    this.ui = ui;
  }

  @Override
  public void mouseMoved(MouseEvent e) {
    Window window = ui.getWindow();
    JRootPane rootPane = ui.getRootPane();
    if (e.getSource() != window) {
      return;
    }
    isPressed = false;
    checkCursor(e);
  }

  @Override
  public void mouseEntered(MouseEvent e) {
    checkCursor(e);
  }

  @Override
  public void mouseExited(MouseEvent e) {
    checkCursor(e);
  }

  @Override
  public void mousePressed(MouseEvent ev) {
    Window window = ui.getWindow();
    JComponent titlePane = ui.getTitlePane();
    if (ev.getSource() != window) {
      return;
    }
    isPressed = true;
    Point dragWindowOffset = ev.getPoint();
    Window w = (Window) ev.getSource();

    Frame f = null;
    Dialog d = null;

    if (w instanceof Frame) {
      f = (Frame) w;
    }
    else if (w instanceof Dialog) {
      d = (Dialog) w;
    }

    int frameState = (f != null) ? f.getExtendedState() : 0;

    if (dragCursor != 0) {
      if ((f != null && f.isResizable() && (frameState & Frame.MAXIMIZED_BOTH) == 0)
          || (d != null && d.isResizable())) {
        dragOffsetX = dragWindowOffset.x;
        dragOffsetY = dragWindowOffset.y;
        dragWidth = w.getWidth();
        dragHeight = w.getHeight();
      }
    }
    else if (titlePane != null) {
      Point convertedDragWindowOffset = SwingUtilities.convertPoint(w, dragWindowOffset, titlePane);
      if (titlePane.contains(convertedDragWindowOffset)) {
        if ((f != null && ((frameState & Frame.MAXIMIZED_BOTH) == 0) || (d != null))
            && dragWindowOffset.y >= BORDER_DRAG_THICKNESS
            && dragWindowOffset.x >= BORDER_DRAG_THICKNESS
            && dragWindowOffset.x < w.getWidth()
                - BORDER_DRAG_THICKNESS) {
          isMovingWindow = true;
          dragOffsetX = dragWindowOffset.x;
          dragOffsetY = dragWindowOffset.y;
        }
      }
    }
  }

  @Override
  public void mouseDragged(MouseEvent ev) {
    Window w = (Window) ev.getSource();
    JRootPane rootPane = ui.getRootPane();
    Point pt = ev.getPoint();

    if (isMovingWindow) {
      Point windowPt = w.getLocationOnScreen();

      windowPt.x += pt.x - dragOffsetX;
      windowPt.y += pt.y - dragOffsetY;
      w.setLocation(windowPt);
    }
    else if (dragCursor != 0) {
      Rectangle r = w.getBounds();
      Rectangle startBounds = new Rectangle(r);
      Dimension min = w.getMinimumSize();
      switch (dragCursor) {
        case Cursor.E_RESIZE_CURSOR:
          adjust(r, min, 0, 0,
              pt.x + (dragWidth - dragOffsetX) - r.width, 0);
          break;
        case Cursor.S_RESIZE_CURSOR:
          adjust(r, min, 0, 0, 0, pt.y + (dragHeight - dragOffsetY)
              - r.height);
          break;
        case Cursor.N_RESIZE_CURSOR:
          adjust(r, min, 0, pt.y - dragOffsetY, 0, -(pt.y - dragOffsetY));
          break;
        case Cursor.W_RESIZE_CURSOR:
          adjust(r, min, pt.x - dragOffsetX, 0, -(pt.x - dragOffsetX), 0);
          break;
        case Cursor.NE_RESIZE_CURSOR:
          adjust(r, min, 0, pt.y - dragOffsetY, pt.x
              + (dragWidth - dragOffsetX) - r.width,
              -(pt.y - dragOffsetY));
          break;
        case Cursor.SE_RESIZE_CURSOR:
          adjust(r, min, 0, 0,
              pt.x + (dragWidth - dragOffsetX) - r.width, pt.y
                  + (dragHeight - dragOffsetY) - r.height);
          break;
        case Cursor.NW_RESIZE_CURSOR:
          adjust(r, min, pt.x - dragOffsetX, pt.y - dragOffsetY,
              -(pt.x - dragOffsetX), -(pt.y - dragOffsetY));
          break;
        case Cursor.SW_RESIZE_CURSOR:
          adjust(r, min, pt.x - dragOffsetX, 0, -(pt.x - dragOffsetX),
              pt.y + (dragHeight - dragOffsetY) - r.height);
          break;
        default:
          break;
      }
      if (!r.equals(startBounds)) {
        w.setBounds(r);
        // Defer repaint/validate on mouseReleased unless dynamic
        // layout is active.
        if (/* Toolkit.getDefaultToolkit().isDynamicLayoutActive() */true) {
          w.validate();
          rootPane.repaint();
        }
      }
    }
  }

  @Override
  public void mouseReleased(MouseEvent ev) {
    isPressed = false;
    Window window = ui.getWindow();
    JRootPane rootPane = ui.getRootPane();
    if (dragCursor != 0 && window != null && !window.isValid()) {
      // Some Window systems validate as you resize, others won't,
      // thus the check for validity before repainting.
      window.validate();
      rootPane.repaint();
    }
    isMovingWindow = false;
    checkCursor(ev);
  }

  @Override
  public void mouseClicked(MouseEvent ev) {
    Window w = (Window) ev.getSource();
    JComponent titlePane = ui.getTitlePane();
    Frame f = null;

    if (w instanceof Frame) {
      f = (Frame) w;
    }
    else {
      return;
    }

    int state = f.getExtendedState();
    if (titlePane != null) {
      Point convertedPoint = SwingUtilities.convertPoint(w, ev.getPoint(), titlePane);
      if (titlePane.contains(convertedPoint)) {
        if ((ev.getClickCount() % 2) == 0
            && ((ev.getModifiers() & InputEvent.BUTTON1_MASK) != 0)) {
          if (f.isResizable()) {
            if ((state & Frame.MAXIMIZED_BOTH) != 0) {
              f.setExtendedState(state & ~Frame.MAXIMIZED_BOTH);
            }
            else {
              f.setExtendedState(state | Frame.MAXIMIZED_BOTH);
            }
            return;
          }
        }
      }
    }
  }

  private void checkCursor(MouseEvent e) {
    Window window = ui.getWindow();
    if (!isPressed) {
      Point dragWindowOffset = e.getPoint();
      dragCursor = getCursor(calculateCorner(window, dragWindowOffset.x, dragWindowOffset.y));
      if (window instanceof Frame && (((Frame) window).getExtendedState() & Frame.MAXIMIZED_BOTH) == Frame.MAXIMIZED_BOTH) {
        // bsh 2010-11-11: Do not allow resizing of maximized windows
        dragCursor = 0;
      }
    }
    if (dragCursor != 0) {
      window.setCursor(Cursor.getPredefinedCursor(dragCursor));
    }
    else {
      window.setCursor(Cursor.getDefaultCursor());
    }
  }

  private void adjust(Rectangle bounds, Dimension min, int deltaX,
      int deltaY, int deltaWidth, int deltaHeight) {
    bounds.x += deltaX;
    bounds.y += deltaY;
    bounds.width += deltaWidth;
    bounds.height += deltaHeight;
    if (min != null) {
      if (bounds.width < min.width) {
        int correction = min.width - bounds.width;
        if (deltaX != 0) {
          bounds.x -= correction;
        }
        bounds.width = min.width;
      }
      if (bounds.height < min.height) {
        int correction = min.height - bounds.height;
        if (deltaY != 0) {
          bounds.y -= correction;
        }
        bounds.height = min.height;
      }
    }
  }

  /**
   * Returns the corner that contains the point <code>x</code>, <code>y</code> , or -1 if the position doesn't match a
   * corner.
   */
  private int calculateCorner(Component c, int x, int y) {
    int xPosition = calculatePosition(x, c.getWidth());
    int yPosition = calculatePosition(y, c.getHeight());

    if (xPosition == -1 || yPosition == -1) {
      return -1;
    }
    return yPosition * 5 + xPosition;
  }

  /**
   * Returns the Cursor to render for the specified corner. This returns 0 if
   * the corner doesn't map to a valid Cursor
   */
  private int getCursor(int corner) {
    if (corner == -1) {
      return 0;
    }
    return cursorMapping[corner];
  }

  /**
   * Returns an integer indicating the position of <code>spot</code> in <code>width</code>. The return value will be: 0
   * if <
   * BORDER_DRAG_THICKNESS 1 if < CORNER_DRAG_WIDTH 2 if >= CORNER_DRAG_WIDTH
   * && < width - BORDER_DRAG_THICKNESS 3 if >= width - CORNER_DRAG_WIDTH 4 if
   * >= width - BORDER_DRAG_THICKNESS 5 otherwise
   */
  private int calculatePosition(int spot, int width) {
    if (spot < BORDER_DRAG_THICKNESS) {
      return 0;
    }
    if (spot < CORNER_DRAG_WIDTH) {
      return 1;
    }
    if (spot >= (width - BORDER_DRAG_THICKNESS)) {
      return 4;
    }
    if (spot >= (width - CORNER_DRAG_WIDTH)) {
      return 3;
    }
    return 2;
  }
}
