/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package com.bsiag.scout.rt.ui.swing.rayo.window.desktop.toolbar;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Shape;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.List;

import javax.swing.AbstractButton;
import javax.swing.Icon;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.synth.SynthLookAndFeel;

import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.rt.client.ui.action.ActionUtility;
import org.eclipse.scout.rt.client.ui.action.IAction;
import org.eclipse.scout.rt.client.ui.desktop.IDesktop;
import org.eclipse.scout.rt.client.ui.desktop.outline.AbstractFormToolButton;
import org.eclipse.scout.rt.ui.swing.Activator;
import org.eclipse.scout.rt.ui.swing.ISwingEnvironment;
import org.eclipse.scout.rt.ui.swing.SwingUtility;
import org.eclipse.scout.rt.ui.swing.action.ISwingScoutAction;
import org.eclipse.scout.rt.ui.swing.ext.FlowLayoutEx;
import org.eclipse.scout.rt.ui.swing.window.desktop.toolbar.AbstractJToolTabsBar;
import org.eclipse.scout.rt.ui.swing.window.desktop.toolbar.SwingScoutHeaderPanel;

import com.bsiag.scout.rt.ui.swing.rayo.RayoIcons;

public class RayoToolTabsBar extends AbstractJToolTabsBar {
  private static final long serialVersionUID = 1L;
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(RayoToolTabsBar.class);

  public static final String PROP_COLLAPSED = "collapsed";
  public static final String PROP_MINIMUM_SIZE = "minimumSize";

  private final ISwingEnvironment m_env;

  private Icon m_activeTabIcon;
  private boolean m_collapsed = true;
  private RayoCollapseButton m_collapseButton;
  private SwingScoutHeaderPanel m_swingScoutHeaderPanel;

  public RayoToolTabsBar(ISwingEnvironment env) {
    m_env = env;
    m_activeTabIcon = Activator.getIcon(RayoIcons.ToolTab);
    setLayout(new Layout());
    setName("Synth.ToolTabsBar");
    if (!(UIManager.getLookAndFeel() instanceof SynthLookAndFeel)) {
      setOpaque(true);
      setBackground(new Color(0x67a8ce));
    }
  }

  @Override
  public void rebuild(IDesktop desktop) {
    removeAll();
    addCollapseButton();
    add(SwingUtility.createGlue(0, 0, true, false));
    List<String> textList = new ArrayList<String>();
    for (IAction a : desktop.getToolButtons()) {
      if (!(a instanceof AbstractFormToolButton)) {
        LOG.warn("Tool button of the type '" + a.getClass().getName() + "' is not supported by Rayo L&F");
        continue; // only render @{link AbstractFormToolButton}'s
      }
      ISwingScoutAction button = m_env.createAction(this, a, ActionUtility.createVisibleFilter());
      if (button != null) {
        addToolTab((AbstractButton) button.getSwingField());
        textList.add(a.getText());
      }
    }
    m_collapseButton.setPotentialTexts(textList);
  }

  @Override
  protected void tabActivated(AbstractButton tab) {
    if (m_collapsed) {
      expandBar();
    }
    m_collapseButton.setText(tab.getText());
    firePropertyChange(PROP_MINIMUM_SIZE, -1, getMinimumSize());
    revalidateAndRepaint();
  }

  private void expandBar() {
    m_collapseButton.setVisible(true);
    revalidateAndRepaint();
    setCollapsed(false);
  }

  private void collapseBar() {
    if (isCollapsed()) {
      return;
    }
    if (getActiveTab() != null) {
      getActiveTab().doClick();

    }
    m_collapseButton.setVisible(false);
    revalidateAndRepaint();
    setCollapsed(true);
  }

  public boolean isCollapsed() {
    return m_collapsed;
  }

  private void setCollapsed(boolean collapsed) {
    boolean oldCollapsed = m_collapsed;
    m_collapsed = collapsed;
    firePropertyChange(PROP_COLLAPSED, oldCollapsed, m_collapsed);
    m_swingScoutHeaderPanel.adjustToolButtonPanelWidth((int) getMinimumSize().getWidth(), false);
  }

  private void revalidateAndRepaint() {
    revalidate();
    repaint();
  }

  private void addCollapseButton() {
    m_collapseButton = new RayoCollapseButton(m_env) {
      private static final long serialVersionUID = 1L;

      /**
       * WORKAROUND swing is inconsistent in handling gui events: all events are sync except component events!
       */
      @SuppressWarnings("deprecation")
      @Override
      public void hide() {
        super.hide();
        //
        collapseBar();
      }
    };
    if (isCollapsed()) {
      m_collapseButton.setVisible(false);
    }
    add(m_collapseButton);
  }

  private void addToolTab(AbstractButton tab) {
    tab.setBorder(new EmptyBorder(0, 10, 0, 10));
    add(tab);
    addActiveTabListener(tab);
    addToolBarChangedListener(tab);
  }

  private void addToolBarChangedListener(AbstractButton tab) {
    // bsh 2010-10-15: This listener is needed in order for the "on/off semantics"
    // to work properly. The listener acts as a bridge between the tab and
    // the tool bar (so each tab can inform the toolbar that it should hide
    // the collapse button).
    tab.addPropertyChangeListener(RayoToolTab.PROP_CHANGED_TOOL_TAB_STATE, new PropertyChangeListener() {
      @Override
      public void propertyChange(PropertyChangeEvent evt) {
        // If the tool tab state was changed, but now tabs are selected anymore,
        // we interpret this as a request to collapse the toolbar.
        boolean allTabsAreInactive = true;
        for (Component c : getComponents()) {
          if (c instanceof AbstractButton) {
            if (((AbstractButton) c).isSelected()) {
              allTabsAreInactive = false;
              break;
            }
          }
        }
        if (allTabsAreInactive) {
          collapseBar();
        }
      }
    });
  }

  @Override
  public void paintComponent(Graphics g) {
    super.paintComponent(g);
    paintActiveTabOnBackground(g);
  }

  private void paintActiveTabOnBackground(Graphics g) {
    AbstractButton tab = getActiveTab();
    Icon icon = m_activeTabIcon;
    if (tab != null && icon != null) {
      int x = tab.getX();
      int y = getHeight() - icon.getIconHeight();
      int w = tab.getWidth();
      int h = icon.getIconHeight();
      Shape oldClip = g.getClip();
      g.setClip(x, y, 6, h);
      icon.paintIcon(this, g, x, y);
      g.setClip(x + 6, y, w - 12, h);
      for (int k = 0; k < w; k = k + icon.getIconWidth()) {
        icon.paintIcon(this, g, x + k, y);
      }
      g.setClip(x + w - 6, y, 6, h);
      icon.paintIcon(this, g, x, y);
      g.setClip(oldClip);
    }
  }

  private class Layout extends FlowLayoutEx {

    Layout() {
      super(FlowLayoutEx.LEFT, FlowLayoutEx.HORIZONTAL, 0, 0);
      setFillVertical(true);
      setFillHorizontal(true);
    }

    @Override
    protected Dimension getLayoutSize(Container parent, int sizeflag) {
      Dimension d;
      if (isCollapsed()) {
        d = super.getLayoutSize(parent, PREF_SIZE);
      }
      else {
        d = super.getLayoutSize(parent, PREF_SIZE);
        switch (sizeflag) {
          case MIN_SIZE: {
            break;
          }
          case MAX_SIZE: {
            d.width = 10240;
            break;
          }
        }
      }

      // ensure min width if expanded
      if (!isCollapsed()) {
        d.width = Math.max(d.width, UIManager.getInt("Synth.ToolButtonView.minWidth"));
      }
      d.height = RayoViewTabsBar.TABS_BAR_HEIGHT;
      return d;
    }
  }

  @Override
  public void setSwingScoutHeaderPanel(SwingScoutHeaderPanel swingScoutHeaderPanel) {
    m_swingScoutHeaderPanel = swingScoutHeaderPanel;
  }
}
