/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package com.bsiag.scout.rt.ui.swing.rayo.window.desktop.toolbar;

import java.awt.Component;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;

import javax.swing.AbstractAction;
import javax.swing.Icon;
import javax.swing.JComponent;

import org.eclipse.scout.rt.ui.swing.basic.IconGroup;
import org.eclipse.scout.rt.ui.swing.basic.IconGroup.IconState;

import com.bsiag.scout.rt.ui.swing.rayo.RayoIcons;

/**
 * Base class of all buttons of the navigation widget.
 *
 * @author awe
 */
public class RayoNavigationWidgetButton {

  enum ActionType {
    PrimaryAction,
    SecondaryAction,
    None;
  }

  private ActionType m_activeActionType;

  private AbstractAction m_primaryAction;
  private AbstractAction m_secondaryAction;

  private IconGroup m_widgetIconGroup;
  private IconGroup m_primaryIconGroup;
  private IconGroup m_secondaryIconGroup;
  private P_ActionEnableStateChangedListener m_actionEnableStateChangedListener;

  protected static final String PROP_ENABLED = "enabled";

  private PropertyChangeSupport m_support = new PropertyChangeSupport(this);

  private JComponent m_parent;

  private Point m_buttonPos;

  private Point m_iconPos;

  private boolean m_enabled = true;

  private boolean m_mouseOver = false;

  private boolean m_pressed = false;

  private ButtonName m_name;

  RayoNavigationWidgetButton(ButtonName name, JComponent parent) {
    m_name = name;
    m_parent = parent;
    m_activeActionType = ActionType.None;
    m_actionEnableStateChangedListener = new P_ActionEnableStateChangedListener();
  }

  protected Icon getPrimaryIconForCurrentState() {
    Icon icon = m_primaryIconGroup.getIcon(IconState.NORMAL);
    if (!isEnabled() && m_primaryIconGroup.hasIcon(IconState.DISABLED)) {
      icon = m_primaryIconGroup.getIcon(IconState.DISABLED);
    }
    else if (isMouseOver() && m_primaryIconGroup.hasIcon(IconState.ROLLOVER)) {
      icon = m_primaryIconGroup.getIcon(IconState.ROLLOVER);
    }
    return icon;
  }

  protected Icon getSecondaryIconForCurrentState() {
    Icon icon = m_secondaryIconGroup.getIcon(IconState.NORMAL);
    if (!isEnabled() && m_secondaryIconGroup.hasIcon(IconState.DISABLED)) {
      icon = m_secondaryIconGroup.getIcon(IconState.DISABLED);
    }
    else if (isMouseOver() && m_secondaryIconGroup.hasIcon(IconState.ROLLOVER)) {
      icon = m_secondaryIconGroup.getIcon(IconState.ROLLOVER);
    }
    return icon;
  }

  private Icon getWidgetIconForCurrentState() {
    Icon icon = m_widgetIconGroup.getIcon(IconState.NORMAL);
    if (isMouseOver() && m_widgetIconGroup.hasIcon(IconState.ROLLOVER)) {
      icon = m_widgetIconGroup.getIcon(IconState.ROLLOVER);
    }
    return icon;
  }

  public boolean isPressed() {
    return m_pressed;
  }

  public void setPressed(boolean pressed) {
    if (isEnabled()) {
      boolean oldPressed = m_pressed;
      m_pressed = pressed;
      if (oldPressed != m_pressed) {
        repaintParent();
      }
    }
  }

  protected void performAction(AbstractAction action) {
    action.actionPerformed(new ActionEvent(m_parent, ActionEvent.ACTION_PERFORMED, null));
  }

  /**
   * Called when a button has been clicked. The default impl. performs the primary action.
   *
   * @param button
   *          A button constant from MouseEvent.
   */
  public void buttonClicked(int button) {
    performAction(m_primaryAction);
  }

  public boolean isMouseOver() {
    return m_mouseOver;
  }

  public void setMouseOver(boolean mouseOver) {
    if (isEnabled()) {
      boolean oldMouseOver = m_mouseOver;
      m_mouseOver = mouseOver;
      if (oldMouseOver != m_mouseOver) {
        repaintParent();
      }
    }
  }

  public boolean isEnabled() {
    return m_enabled;
  }

  public void setEnabled(boolean enabled) {
    boolean oldEnabled = m_enabled;
    m_enabled = enabled;
    m_support.firePropertyChange(PROP_ENABLED, oldEnabled, m_enabled);
    if (oldEnabled != m_enabled) {
      if (!enabled) {
        m_mouseOver = false;
        m_pressed = false;
      }
      repaintParent();
    }
  }

  public final void paintComponent(Component c, Graphics g) {
    paintButton(c, g);
    paintIcon(c, g);
  }

  protected void paintButton(Component c, Graphics g) {
    paintPressedImage(getWidgetIconForCurrentState(), m_buttonPos, m_pressed, c, g);
  }

  protected void paintIcon(Component c, Graphics g) {
    if (getActiveActionType() == ActionType.PrimaryAction) {
      paintPressedImage(getPrimaryIconForCurrentState(), m_iconPos, m_pressed, c, g);
    }
    else if (getActiveActionType() == ActionType.SecondaryAction) {
      paintPressedImage(getSecondaryIconForCurrentState(), m_iconPos, m_pressed, c, g);
    }
  }

  protected void paintPressedImage(Icon img, Point p, boolean pressed, Component c, Graphics g) {
    int x = p.x;
    int y = p.y;
    if (pressed) {
      x += 1;
      y += 1;
    }
    img.paintIcon(c, g, x, y);
  }

  protected void repaintParent() {
    m_parent.repaint();
  }

  void addPropertyChangeListener(PropertyChangeListener l) {
    m_support.addPropertyChangeListener(l);
  }

  void removePropertyChangeListener(PropertyChangeListener l) {
    m_support.removePropertyChangeListener(l);
  }

  public ButtonName getName() {
    return m_name;
  }

  public final void setPrimaryAction(AbstractAction primaryAction) {
    if (m_primaryAction != null) {
      m_primaryAction.removePropertyChangeListener(m_actionEnableStateChangedListener);
    }
    m_primaryAction = primaryAction;
    if (m_primaryAction != null) {
      m_primaryAction.addPropertyChangeListener(new P_ActionEnableStateChangedListener());
    }
  }

  public final AbstractAction getPrimaryAction() {
    return m_primaryAction;
  }

  public final void setSecondaryAction(AbstractAction secondaryAction) {
    if (m_secondaryAction != null) {
      m_secondaryAction.removePropertyChangeListener(m_actionEnableStateChangedListener);
    }
    m_secondaryAction = secondaryAction;
    if (m_secondaryAction != null) {
      m_secondaryAction.addPropertyChangeListener(new P_ActionEnableStateChangedListener());
    }
  }

  public final AbstractAction getSecondaryAction() {
    return m_secondaryAction;
  }

  public void setIconId(ActionType actionType, String iconId) {
    if (actionType == ActionType.PrimaryAction) {
      m_primaryIconGroup = new IconGroup(iconId);
    }
    else {
      m_secondaryIconGroup = new IconGroup(iconId);
    }
  }

  public IconGroup getIconGroup(ActionType actionType) {
    if (actionType == ActionType.PrimaryAction) {
      return m_primaryIconGroup;
    }
    else {
      return m_secondaryIconGroup;
    }
  }

  public void setWidgetIconId(String iconId) {
    m_widgetIconGroup = new IconGroup(iconId);
  }

  public ActionType getActiveActionType() {
    return m_activeActionType;
  }

  public void setActiveActionType(ActionType activeActionType) {
    m_activeActionType = activeActionType;
  }

  protected AbstractAction getActiveAction() {
    switch (m_activeActionType) {
      case PrimaryAction:
        return m_primaryAction;
      case SecondaryAction:
        return m_secondaryAction;
      default:
        return null;
    }
  }

  private class P_ActionEnableStateChangedListener implements PropertyChangeListener {

    @Override
    public void propertyChange(PropertyChangeEvent evt) {
      if ("enabled".equals(evt.getPropertyName())) {
        RayoNavigationWidgetButton.this.setEnabled((Boolean) evt.getNewValue());
      }
    }
  }

  public Point getButtonPos() {
    return m_buttonPos;
  }

  public void setButtonPos(Point buttonPos) {
    m_buttonPos = buttonPos;
  }

  public Point getIconPos() {
    return m_iconPos;
  }

  public void setIconPos(Point iconPos) {
    m_iconPos = iconPos;
  }

  enum ButtonName {
    NONE(new Rectangle(), RayoIcons.NavigationShadow),
    BACK(new Rectangle(17, 17, 28, 30), RayoIcons.NavigationShadowBack),
    FORWARD(new Rectangle(49, 17, 33, 30), RayoIcons.NavigationShadowForward),
    STOP_REFRESH(new Rectangle(82, 17, 30, 30), RayoIcons.NavigationShadowStopRefresh);

    private Rectangle m_bounds;

    private String m_dropshadowIcon;

    ButtonName(Rectangle bounds, String dropshadowIcon) {
      m_bounds = bounds;
      m_dropshadowIcon = dropshadowIcon;
    }

    static ButtonName getMouseOverButton(int x, int y) {
      for (ButtonName buttonName : values()) {
        if (buttonName.m_bounds.contains(x, y)) {
          return buttonName;
        }
      }
      return NONE;
    }

    public String getDropshadowIcon() {
      return m_dropshadowIcon;
    }
  }
}
