/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package com.bsiag.scout.rt.ui.swing.rayo.window.desktop.toolbar;

import java.awt.Dimension;
import java.awt.event.ComponentAdapter;
import java.awt.event.ComponentEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import org.eclipse.scout.commons.OptimisticLock;
import org.eclipse.scout.rt.ui.swing.window.desktop.toolbar.AbstractJToolTabsBar;

import com.bsiag.scout.rt.ui.swing.rayo.window.desktop.RayoSwingScoutHeaderPanel;

/**
 * Synchronizes the ToolTabsBar widget with the ToolsView so both widget looks like a single unit.
 * 
 * @author awe
 */
public class ToolsViewAndTabsBarSynchronizer {

  private final ToolsViewController m_toolsViewController;
  private final RayoSwingScoutHeaderPanel m_swingScoutHeaderPanel;
  private final OptimisticLock m_syncLock;

  public ToolsViewAndTabsBarSynchronizer(ToolsViewController toolsViewController, RayoSwingScoutHeaderPanel swingScoutHeaderPanel) {
    m_syncLock = new OptimisticLock();
    m_toolsViewController = toolsViewController;
    m_swingScoutHeaderPanel = swingScoutHeaderPanel;
    // It is crucial to relayout the header panel with the width of the toolbar and not with the width of the view. This is because no view might be active.
    // Also, force a relayout to initially install the tool button panel.
    m_swingScoutHeaderPanel.adjustToolButtonPanelWidth((int) swingScoutHeaderPanel.getSwingToolTabsPanel().getPreferredSize().getWidth(), true);
    installListeners();
  }

  protected void reset() {
    //nop
  }

  private void installListeners() {
    AbstractJToolTabsBar toolTabs = (AbstractJToolTabsBar) m_swingScoutHeaderPanel.getSwingToolTabsPanel();

    toolTabs.addPropertyChangeListener(new PropertyChangeListener() {
      @Override
      public void propertyChange(PropertyChangeEvent evt) {
        try {
          m_syncLock.acquire();
          //
          if (RayoToolTabsBar.PROP_COLLAPSED.equals(evt.getPropertyName())) {
            boolean collapsed = (Boolean) evt.getNewValue();
            if (collapsed) {
              m_toolsViewController.collapseView();
            }
            else {
              m_toolsViewController.expandView();
            }
          }
          else if (RayoToolTabsBar.PROP_MINIMUM_SIZE.equals(evt.getPropertyName())) {
            Dimension d = (Dimension) evt.getNewValue();
            m_toolsViewController.setMinimumWidth(d.width);
          }
        }
        finally {
          m_syncLock.release();
        }
      }
    });
    m_toolsViewController.addComponentListener(new ComponentAdapter() {
      /**
       * WORKAROUND careful: swing is inconsistent in handling gui events: all events are sync except component events!
       * This event is posted after the causing runnable is executed.
       */
      @Override
      public void componentResized(ComponentEvent e) {
        try {
          if (m_syncLock.acquire()) {
            m_swingScoutHeaderPanel.adjustToolButtonPanelWidth(m_toolsViewController.getWidth(), false);
          }
        }
        finally {
          m_syncLock.release();
        }
      }
    });
  }
}
