/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package com.bsiag.scout.rt.ui.swing.rayo.window.desktop.toolbar;

import java.awt.Color;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;

import javax.swing.AbstractButton;
import javax.swing.Icon;
import javax.swing.JLabel;
import javax.swing.UIManager;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.synth.SynthLookAndFeel;

import org.eclipse.scout.commons.job.JobEx;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.rt.client.ui.action.ActionUtility;
import org.eclipse.scout.rt.client.ui.action.IAction;
import org.eclipse.scout.rt.client.ui.desktop.IDesktop;
import org.eclipse.scout.rt.ui.swing.Activator;
import org.eclipse.scout.rt.ui.swing.ISwingEnvironment;
import org.eclipse.scout.rt.ui.swing.SwingUtility;
import org.eclipse.scout.rt.ui.swing.action.ISwingScoutAction;
import org.eclipse.scout.rt.ui.swing.basic.IconGroup;
import org.eclipse.scout.rt.ui.swing.basic.IconGroup.IconState;
import org.eclipse.scout.rt.ui.swing.ext.FlowLayoutEx;
import org.eclipse.scout.rt.ui.swing.window.desktop.menubar.SwingScoutMainMenuPopup;
import org.eclipse.scout.rt.ui.swing.window.desktop.toolbar.AbstractJViewTabsBar;
import org.eclipse.scout.rt.ui.swing.window.desktop.toolbar.HandCursorAdapater;

import com.bsiag.scout.rt.ui.swing.rayo.RayoIcons;

/**
 * SwingScoutToolBar
 * <p>
 * Checks UIManager.get("MenuBar.policy") == "toolbar" (alternative: "menubar")
 * </p>
 */
public class RayoViewTabsBar extends AbstractJViewTabsBar {
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(RayoViewTabsBar.class);

  private static final long serialVersionUID = 1L;

  public static final int TABS_BAR_HEIGHT = 30;
  private Icon m_activeTabLeftIcon;
  private Icon m_activeTabRightIcon;
  private IconGroup m_menuIcon;
  private IDesktop m_desktop;
  private JLabel m_menuLabel;
  private SwingScoutMainMenuPopup m_popupMenu;
  private ISwingEnvironment m_env;

  public RayoViewTabsBar(ISwingEnvironment env) {
    m_env = env;
    m_activeTabLeftIcon = Activator.getIcon(RayoIcons.ViewTabLeft);
    m_activeTabRightIcon = Activator.getIcon(RayoIcons.ViewTabRight);
    m_menuIcon = new IconGroup(RayoIcons.IconMenu);
    setName("Synth.ViewTabsBar");
    setLayout(new Layout());
    setOpaque(true);
    if (!(UIManager.getLookAndFeel() instanceof SynthLookAndFeel)) {
      setBackground(new Color(0xddebf4));//TODO [awe] add to lookandfeel
    }
  }

  @Override
  public void rebuild(IDesktop desktop) {
    removeAll();
    if (isShowMenuBar()) {
      addMenuIcon();
      //add rigid glue
      add(SwingUtility.createGlue(5, 5, false, false));
    }
    for (IAction a : desktop.getViewButtons()) {
      ISwingScoutAction button = m_env.createAction(this, a, ActionUtility.createVisibleFilter());
      if (button != null) {
        addViewTab((AbstractButton) button.getSwingField());
      }
    }
    //add variable glue
    add(SwingUtility.createGlue(0, 0, true, false));
    if (isShowMenuBar()) {
      m_popupMenu = new SwingScoutMainMenuPopup(ActionUtility.createVisibleFilter());
      m_popupMenu.createField(desktop, m_env);
    }
  }

  @Override
  public void paintComponent(Graphics g) {
    super.paintComponent(g);
    paintActiveTabOnBackground(g);
  }

  private boolean isShowMenuBar() {
    return "toolbar".equals(UIManager.get("MenuBar.policy"));
  }

  /**
   * Adds a border on the right instead of using hgap of flowlayout, because flowlayout adds the gap also
   * in front of the first element and that's not what we want.
   * 
   * @param tab
   * @return
   */
  private AbstractButton addViewTab(AbstractButton tab) {
    addActiveTabListener(tab);
    tab.setBorder(new EmptyBorder(3, 12, 0, 14));//XXX UI?
    add(tab);
    return tab;
  }

  private void addMenuIcon() {
    m_menuLabel = new JLabel();
    m_menuLabel.setIcon(m_menuIcon.getIcon(IconState.NORMAL));
    m_menuLabel.setBorder(new EmptyBorder(3, 3, 2, 0));//XXX UI?
    new HandCursorAdapater(m_menuLabel);
    m_menuLabel.addMouseListener(new MouseAdapter() {
      @Override
      public void mouseReleased(MouseEvent e) {
        if (m_popupMenu.getScoutObject() != null) {
          Runnable t = new Runnable() {
            @Override
            public void run() {
              m_popupMenu.getScoutObject().prepareAllMenus();
            }
          };

          JobEx job = m_env.invokeScoutLater(t, 5000);
          try {
            job.join(5000);
          }
          catch (InterruptedException ex) {
            LOG.warn("Exception occured while preparing all menus.", ex);
          }
        }
        m_popupMenu.getSwingPopupMenu().show(m_menuLabel, 0, m_menuLabel.getHeight());
      }

      @Override
      public void mouseEntered(MouseEvent e) {
        m_menuLabel.setIcon(m_menuIcon.getIcon(IconState.ROLLOVER));
      }

      @Override
      public void mouseExited(MouseEvent e) {
        m_menuLabel.setIcon(m_menuIcon.getIcon(IconState.NORMAL));
      }
    });
    add(m_menuLabel);
  }

  private void paintActiveTabOnBackground(Graphics g) {
    AbstractButton tab = getActiveTab();
    if (tab == null) {
      return;
    }
    Insets i = tab.getInsets();
    int y = i.top;
    // left tab icon
    int x = tab.getX() + i.left - m_activeTabLeftIcon.getIconWidth();
    m_activeTabLeftIcon.paintIcon(this, g, x, y);
    // right tab icon
    x = tab.getX() + tab.getWidth() - i.right;
    m_activeTabRightIcon.paintIcon(this, g, x, y);
    // white fill
    x = tab.getX() + i.left;
    g.setColor(Color.WHITE);
    g.fillRect(x, y, tab.getWidth() - i.right - i.left, getHeight() - y);
  }

  private class Layout extends FlowLayoutEx {

    Layout() {
      super(FlowLayoutEx.LEFT, FlowLayoutEx.HORIZONTAL, 0, 0);
      setFillVertical(true);
      setFillHorizontal(true);
    }

    @Override
    public Dimension minimumLayoutSize(Container parent) {
      Dimension d = super.minimumLayoutSize(parent);
      d.height = TABS_BAR_HEIGHT;
      return d;
    }

    @Override
    public Dimension preferredLayoutSize(Container parent) {
      Dimension d = super.preferredLayoutSize(parent);
      d.height = TABS_BAR_HEIGHT;
      return d;
    }

    @Override
    public Dimension maximumLayoutSize(Container parent) {
      Dimension d = super.maximumLayoutSize(parent);
      d.width = 10240;
      d.height = TABS_BAR_HEIGHT;
      return d;
    }
  }
}
