/*
 * Copyright (c) 2002, 2005, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.bsiag.javax.swing.plaf.synth;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JRootPane;
import javax.swing.SwingConstants;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;

import com.bsiag.sun.swing.plaf.synth.SynthUI;

/**
 * The class that manages a synth title bar
 * 
 * @author David Kloba
 * @author Joshua Outwater
 * @author Steve Wilson
 */
class SynthRootPaneTitlePane extends BasicRootPaneTitlePane implements SynthUI, PropertyChangeListener {
  private static final long serialVersionUID = 1L;
  private static final Border handyEmptyBorder = new EmptyBorder(0, 0, 0, 0);

  private SynthStyle style;
  private int titleSpacing;
  private int buttonSpacing;
  // Alignment for the title, one of SwingConstants.(LEADING|TRAILING|CENTER)
  private int titleAlignment;

  public SynthRootPaneTitlePane(JRootPane rootPane) {
    super(rootPane);
  }

  public SynthContext getContext(JComponent c) {
    return getContext(c, getComponentState(c));
  }

  public SynthContext getContext(JComponent c, int state) {
    return SynthContext.getContext(SynthContext.class, c, SynthLookAndFeel.getRegion(c), style, state);
  }

  //    private Region getRegion(JComponent c) {
  //        return SynthLookAndFeel.getRegion(c);
  //    }

  private int getComponentState(JComponent c) {
    return SynthLookAndFeel.getComponentState(c);
  }

  protected void installListeners() {
    super.installListeners();
    addPropertyChangeListener(this);
  }

  protected void uninstallListeners() {
    removePropertyChangeListener(this);
    super.uninstallListeners();
  }

  private void updateStyle(JComponent c) {
    SynthContext context = getContext(this, ENABLED);
    SynthStyle oldStyle = style;
    style = SynthLookAndFeel.updateStyle(context, this);
    if (style != oldStyle) {
      maximizeIcon = style.getIcon(context, "InternalFrameTitlePane.maximizeIcon");
      minimizeIcon = style.getIcon(context, "InternalFrameTitlePane.minimizeIcon");
      iconifyIcon = style.getIcon(context, "InternalFrameTitlePane.iconifyIcon");
      closeIcon = style.getIcon(context, "InternalFrameTitlePane.closeIcon");
      maximizeIconRollOver = style.getIcon(context, "InternalFrameTitlePane.maximizeIconRollOver");
      minimizeIconRollOver = style.getIcon(context, "InternalFrameTitlePane.minimizeIconRollOver");
      iconifyIconRollOver = style.getIcon(context, "InternalFrameTitlePane.iconifyIconRollOver");
      closeIconRollOver = style.getIcon(context, "InternalFrameTitlePane.closeIconRollOver");
      titleSpacing = style.getInt(context, "InternalFrameTitlePane.titleSpacing", 2);
      buttonSpacing = style.getInt(context, "InternalFrameTitlePane.buttonSpacing", 2);
      String alignString = (String) style.get(context, "InternalFrameTitlePane.titleAlignment");
      titleAlignment = SwingConstants.LEADING;
      if (alignString != null) {
        alignString = alignString.toUpperCase();
        if (alignString.equals("TRAILING")) {
          titleAlignment = SwingConstants.TRAILING;
        }
        else if (alignString.equals("CENTER")) {
          titleAlignment = SwingConstants.CENTER;
        }
      }
      updateActionStyles();
    }
    context.dispose();
  }

  protected void installDefaults() {
    super.installDefaults();
    updateStyle(this);
  }

  protected void uninstallDefaults() {
    SynthContext context = getContext(this, ENABLED);
    style.uninstallDefaults(context);
    context.dispose();
    style = null;
    super.uninstallDefaults();
  }

  // SynthRootPaneTitlePane has no UI, we'll invoke paint on it.
  public void paintComponent(Graphics g) {
    // As state isn't bound, we need a convenience place to check if it has changed.
    if (getFrame() != null) {
      setState(getFrame().getExtendedState());
    }
    //
    SynthContext context = getContext(this);
    SynthLookAndFeel.update(context, g);
    context.getPainter().paintInternalFrameTitlePaneBackground(context,
        g, 0, 0, getWidth(), getHeight());
    paint(context, g);
    context.dispose();
  }

  protected void paint(SynthContext context, Graphics g) {
    String title = getTitle();

    if (title != null) {
      SynthStyle style = context.getStyle();

      g.setColor(style.getColor(context, ColorType.TEXT_FOREGROUND));
      g.setFont(style.getFont(context));

      // Center text vertically.
      FontMetrics fm = rootPane.getFontMetrics(g.getFont());
      int baseline = (getHeight() + fm.getAscent() - fm.getLeading() -
          fm.getDescent()) / 2;
      JButton lastButton = null;
      if (isFrameIconifiable()) {
        lastButton = iconifyButton;
      }
      else if (isFrameMaximizable()) {
        lastButton = toggleButton;
      }
      else if (isFrameClosable()) {
        lastButton = closeButton;
      }
      int maxX;
      int minX;
      boolean ltr = true;
      int titleAlignment = this.titleAlignment;
      if (ltr) {
        if (lastButton != null) {
          maxX = lastButton.getX() - titleSpacing;
        }
        else {
          maxX = rootPane.getWidth() - rootPane.getInsets().right - titleSpacing;
        }
        //                minX = menuButton.getX() + menuButton.getWidth() +
        //                       titleSpacing;
        minX = titleSpacing;
      }
      else {
        if (lastButton != null) {
          minX = lastButton.getX() + lastButton.getWidth() + titleSpacing;
        }
        else {
          minX = rootPane.getInsets().left + titleSpacing;
        }
        //                maxX = menuButton.getX() - titleSpacing;
        maxX = 0 - titleSpacing;
        if (titleAlignment == SwingConstants.LEADING) {
          titleAlignment = SwingConstants.TRAILING;
        }
        else if (titleAlignment == SwingConstants.TRAILING) {
          titleAlignment = SwingConstants.LEADING;
        }
      }
      String clippedTitle = getTitle(title, fm, maxX - minX);
      if (clippedTitle == title) {
        // String fit, align as necessary.
        if (titleAlignment == SwingConstants.TRAILING) {
          minX = maxX - style.getGraphicsUtils(context).
              computeStringWidth(context, g.getFont(), fm, title);
        }
        else if (titleAlignment == SwingConstants.CENTER) {
          int width = style.getGraphicsUtils(context).
              computeStringWidth(context, g.getFont(), fm, title);
          minX = Math.max(minX, (getWidth() - width) / 2);
          minX = Math.min(maxX - width, minX);
        }
      }
      style.getGraphicsUtils(context).paintText(context, g, clippedTitle,
          minX, baseline - fm.getAscent(), -1);
    }
  }

  public void paintBorder(SynthContext context, Graphics g, int x,
      int y, int w, int h) {
    context.getPainter().paintInternalFrameTitlePaneBorder(context,
        g, x, y, w, h);
  }

  @Override
  protected void createButtons() {
    super.createButtons();
    closeButton.putClientProperty("paintActive", Boolean.TRUE);
    closeButton.setBorder(handyEmptyBorder);

    iconifyButton.putClientProperty("paintActive", Boolean.TRUE);
    iconifyButton.setBorder(handyEmptyBorder);

    toggleButton.putClientProperty("paintActive", Boolean.TRUE);
    toggleButton.setBorder(handyEmptyBorder);
  }

  protected LayoutManager createLayout() {
    SynthContext context = getContext(this);
    LayoutManager lm =
        (LayoutManager) style.get(context, "InternalFrameTitlePane.titlePaneLayout");
    context.dispose();
    return (lm != null) ? lm : new SynthTitlePaneLayout();
  }

  public void propertyChange(PropertyChangeEvent evt) {
    if (evt.getSource() == this) {
      if (SynthLookAndFeel.shouldUpdateStyle(evt)) {
        updateStyle(this);
      }
    }
  }

  class SynthTitlePaneLayout implements LayoutManager {
    public void addLayoutComponent(String name, Component c) {
    }

    public void removeLayoutComponent(Component c) {
    }

    public Dimension preferredLayoutSize(Container c) {
      return minimumLayoutSize(c);
    }

    public Dimension minimumLayoutSize(Container c) {
      SynthContext context = getContext(
          SynthRootPaneTitlePane.this);
      int width = 0;
      int height = 0;

      int buttonCount = 0;
      Dimension pref;

      if (isFrameClosable()) {
        pref = closeButton.getPreferredSize();
        width += pref.width;
        height = Math.max(pref.height, height);
        buttonCount++;
      }
      if (isFrameMaximizable()) {
        pref = toggleButton.getPreferredSize();
        width += pref.width;
        height = Math.max(pref.height, height);
        buttonCount++;
      }
      if (isFrameIconifiable()) {
        pref = iconifyButton.getPreferredSize();
        width += pref.width;
        height = Math.max(pref.height, height);
        buttonCount++;
      }
      //            pref = menuButton.getPreferredSize();
      //            width += pref.width;
      //            height = Math.max(pref.height, height);

      width += Math.max(0, (buttonCount - 1) * buttonSpacing);

      FontMetrics fm = SynthRootPaneTitlePane.this.getFontMetrics(getFont());
      SynthGraphicsUtils graphicsUtils = context.getStyle().
          getGraphicsUtils(context);
      String frameTitle = getTitle();
      int title_w = frameTitle != null ? graphicsUtils.
          computeStringWidth(context, fm.getFont(), fm, frameTitle) : 0;
      int title_length = frameTitle != null ? frameTitle.length() : 0;

      // Leave room for three characters in the title.
      if (title_length > 3) {
        int subtitle_w = graphicsUtils.computeStringWidth(context,
                fm.getFont(), fm, frameTitle.substring(0, 3) + "...");
        width += (title_w < subtitle_w) ? title_w : subtitle_w;
      }
      else {
        width += title_w;
      }

      height = Math.max(fm.getHeight() + 2, height);

      width += titleSpacing + titleSpacing;

      Insets insets = getInsets();
      height += insets.top + insets.bottom;
      width += insets.left + insets.right;
      context.dispose();
      return new Dimension(width, height);
    }

    private int center(Component c, Insets insets, int x,
        boolean trailing) {
      Dimension pref = c.getPreferredSize();
      if (trailing) {
        x -= pref.width;
      }
      c.setBounds(x, insets.top +
          (getHeight() - insets.top - insets.bottom -
              pref.height) / 2, pref.width, pref.height);
      if (pref.width > 0) {
        if (trailing) {
          return x - buttonSpacing;
        }
        return x + pref.width + buttonSpacing;
      }
      return x;
    }

    public void layoutContainer(Container c) {
      Insets insets = c.getInsets();
      boolean ltr = true;
      if (ltr) {
        //                      center(menuButton, insets, insets.left, false);
        int x = getWidth() - insets.right;
        if (isFrameClosable()) {
          x = center(closeButton, insets, x, true);
        }
        if (isFrameMaximizable()) {
          x = center(toggleButton, insets, x, true);
        }
        if (isFrameIconifiable()) {
          x = center(iconifyButton, insets, x, true);
        }
      }
      else {
        //                      center(menuButton, insets, getWidth() - insets.right, true);
        int x = insets.left;
        if (isFrameClosable()) {
          x = center(closeButton, insets, x, false);
        }
        if (isFrameMaximizable()) {
          x = center(toggleButton, insets, x, false);
        }
        if (isFrameIconifiable()) {
          x = center(iconifyButton, insets, x, false);
        }
      }
    }
  }

}
