/*
 * Copyright (c) 2002, 2003, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.bsiag.javax.swing.plaf.synth;

import java.awt.Component;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.LayoutManager2;
import java.awt.Rectangle;
import java.awt.Window;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JComponent;
import javax.swing.JLayeredPane;
import javax.swing.JRootPane;
import javax.swing.SwingUtilities;
import javax.swing.event.MouseInputListener;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.basic.BasicRootPaneUI;

import com.bsiag.sun.swing.plaf.synth.CustomSynthNames;
import com.bsiag.sun.swing.plaf.synth.SynthUI;

/**
 * Synth's RootPaneUI.
 * 
 * @author Scott Violet
 */
class SynthRootPaneUI extends BasicRootPaneUI implements SynthUI {
  private SynthStyle style;

  /**
   * The <code>LayoutManager</code> that is set on the <code>JRootPane</code>.
   */
  private LayoutManager layoutManager;

  /**
   * <code>JComponent</code> providing window decorations. This will be null if
   * not providing window decorations.
   */
  private JComponent titlePane;

  /**
   * Window the <code>JRootPane</code> is in.
   */
  private Window window;

  /**
   * <code>JRootPane</code> providing the look and feel for.
   */
  private JRootPane root;

  /**
   * <code>MouseInputListener</code> that is added to the parent <code>Window</code> the <code>JRootPane</code> is
   * contained in.
   */
  private MouseInputListener mouseInputListener;

  public static ComponentUI createUI(JComponent c) {
    return new SynthRootPaneUI();
  }

  protected void installDefaults(JRootPane c) {
    updateStyle(c);
    root = (JRootPane) c;
    installClientDecorations(c);
  }

  /**
   * (awe, ticket #90090) If the window decoration stlye changes after the dialog has been created, we need to set
   * the right synth name, otherwise the dialog remains unstyled. That's exactly what happens when a Dialog opens
   * by calling the JFileChooser.createDialog() method.
   * 
   * @param root
   */
  private void installClientDecorations(JRootPane root) {
    int style = root.getWindowDecorationStyle();
    if (style != JRootPane.NONE && style != JRootPane.FRAME) {
      root.setName(CustomSynthNames.DIALOG);
    }
    JComponent titlePane = createTitlePane(root);
    setTitlePane(root, titlePane);
    if (root.getParent() != null) {
      installWindowListeners(root, root.getParent());
    }
    installLayout(root);
  }

  /**
   * Installs the appropriate LayoutManager on the <code>JRootPane</code> to
   * render the window decorations.
   */
  private void installLayout(JRootPane root) {
    if (layoutManager == null) {
      layoutManager = createLayoutManager();
      root.setLayout(layoutManager);
    }
  }

  /**
   * Returns a <code>LayoutManager</code> that will be set on the <code>JRootPane</code>.
   */
  private LayoutManager createLayoutManager() {
    return new SynthRootLayout();
  }

  /**
   * Installs the necessary Listeners on the parent <code>Window</code>, if
   * there is one.
   * <p>
   * This takes the parent so that cleanup can be done from <code>removeNotify</code>, at which point the parent hasn't
   * been reset yet.
   * 
   * @param parent
   *          The parent of the JRootPane
   */
  private void installWindowListeners(JRootPane root, Component parent) {
    if (parent instanceof Window) {
      window = (Window) parent;
    }
    else {
      window = SwingUtilities.getWindowAncestor(parent);
    }
    if (window != null) {
      if (mouseInputListener == null) {
        mouseInputListener = createWindowMouseInputListener(root);
        window.addMouseListener(mouseInputListener);
        window.addMouseMotionListener(mouseInputListener);
      }
      // <bsh 2010-09-27 #90707-99>
      // Propagate title change event to the titlePane
      window.addPropertyChangeListener("title", new PropertyChangeListener() {
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
          if (titlePane != null) {
            titlePane.repaint();
          }
        }
      });
      // </bsh>
    }
  }

  /**
   * Returns a <code>MouseListener</code> that will be added to the <code>Window</code> containing the
   * <code>JRootPane</code>.
   */
  private MouseInputListener createWindowMouseInputListener(JRootPane root) {
    return new MouseInputHandler(this);
  }

  /**
   * Sets the window title pane -- the JComponent used to provide a plaf a way
   * to override the native operating system's window title pane with one whose
   * look and feel are controlled by the plaf. The plaf creates and sets this
   * value; the default is null, implying a native operating system window title
   * pane.
   * 
   * @param content
   *          the <code>JComponent</code> to use for the window title pane.
   */
  private void setTitlePane(JRootPane root, JComponent titlePane) {
    JLayeredPane layeredPane = root.getLayeredPane();
    JComponent oldTitlePane = getTitlePane();

    if (oldTitlePane != null) {
      oldTitlePane.setVisible(false);
      layeredPane.remove(oldTitlePane);
    }
    if (titlePane != null) {
      layeredPane.add(titlePane, JLayeredPane.FRAME_CONTENT_LAYER);
      titlePane.setVisible(true);
    }
    this.titlePane = titlePane;
  }

  /**
   * Returns the <code>JComponent</code> to render the window decoration style.
   */
  private JComponent createTitlePane(JRootPane root) {
    return new SynthRootPaneTitlePane(root);
  }

  protected void uninstallDefaults(JRootPane root) {
    SynthContext context = getContext(root, ENABLED);

    style.uninstallDefaults(context);
    context.dispose();
    style = null;
  }

  public SynthContext getContext(JComponent c) {
    return getContext(c, getComponentState(c));
  }

  private SynthContext getContext(JComponent c, int state) {
    return SynthContext.getContext(SynthContext.class, c,
                    SynthLookAndFeel.getRegion(c), style, state);
  }

  private int getComponentState(JComponent c) {
    return SynthLookAndFeel.getComponentState(c);
  }

  private void updateStyle(JComponent c) {
    SynthContext context = getContext(c, ENABLED);
    SynthStyle oldStyle = style;
    style = SynthLookAndFeel.updateStyle(context, this);
    if (style != oldStyle) {
      if (oldStyle != null) {
        uninstallKeyboardActions((JRootPane) c);
        installKeyboardActions((JRootPane) c);
      }
    }
    context.dispose();
  }

  public void update(Graphics g, JComponent c) {
    SynthContext context = getContext(c);

    SynthLookAndFeel.update(context, g);
    context.getPainter().paintRootPaneBackground(context,
                          g, 0, 0, c.getWidth(), c.getHeight());
    paint(context, g);
    context.dispose();
  }

  public void paint(Graphics g, JComponent c) {
    SynthContext context = getContext(c);

    paint(context, g);
    context.dispose();
  }

  protected void paint(SynthContext context, Graphics g) {
  }

  public void paintBorder(SynthContext context, Graphics g, int x, int y, int w, int h) {
    context.getPainter().paintRootPaneBorder(context, g, x, y, w, h);
  }

  /**
   * Invoked when a property changes on the root pane. If the event
   * indicates the <code>defaultButton</code> has changed, this will
   * reinstall the keyboard actions.
   * Invoked when a property changes. <code>SynthRootPaneUI</code> is primarily
   * interested in events originating from the <code>JRootPane</code> it has
   * been installed on identifying the property <code>windowDecorationStyle</code>. If the
   * <code>windowDecorationStyle</code> has changed to a value other than <code>JRootPane.NONE</code>, this will add a
   * <code>Component</code> to the <code>JRootPane</code> to render the window decorations, as well as
   * installing a <code>Border</code> on the <code>JRootPane</code>. On the
   * other hand, if the <code>windowDecorationStyle</code> has changed to <code>JRootPane.NONE</code>, this will remove
   * the <code>Component</code> that has been added to the <code>JRootPane</code> as well resetting the
   * Border to what it was before <code>installUI</code> was invoked.
   */
  public void propertyChange(PropertyChangeEvent e) {
    if (SynthLookAndFeel.shouldUpdateStyle(e)) {
      updateStyle((JRootPane) e.getSource());
    }
    String propertyName = e.getPropertyName();
    if (propertyName == null) {
      return;
    }
    else if (propertyName.equals("windowDecorationStyle")) {
      JRootPane root = (JRootPane) e.getSource();

      // This is potentially more than needs to be done,
      // but it rarely happens and makes the install/uninstall process
      // simpler. SynthTitlePane also assumes it will be recreated if
      // the decoration style changes.
      installClientDecorations(root);
    }
    else if (propertyName.equals("ancestor")) {
      if (((JRootPane) e.getSource()).getWindowDecorationStyle() != JRootPane.NONE) {
        installWindowListeners(root, root.getParent());
      }
    }
    super.propertyChange(e);
  }

  /**
   * A custom layout manager that is responsible for the layout of layeredPane,
   * glassPane, menuBar and titlePane, if one has been installed.
   */
  // NOTE: Ideally this would extends JRootPane.RootLayout, but that
  // would force this to be non-static.
  private static class SynthRootLayout implements LayoutManager2 {
    private static final int MINSIZE = 10;
    private static final int PREFSIZE = 20;
    private static final int MAXSIZE = 30;

    public Dimension minimumLayoutSize(Container parent) {
      return layoutSize(parent, MINSIZE);
    }

    public Dimension preferredLayoutSize(Container parent) {
      return layoutSize(parent, PREFSIZE);
    }

    public Dimension maximumLayoutSize(Container parent) {
      return layoutSize(parent, MAXSIZE);
    }

    protected Dimension layoutSize(Container parent, int sizeflag) {
      Dimension cpd = null, mbd = null, tpd = null;
      int cpWidth = 0;
      int cpHeight = 0;
      int mbWidth = 0;
      int mbHeight = 0;
      int tpWidth = 0;
      int tpHeight = 0;
      Insets i = parent.getInsets();
      JRootPane root = (JRootPane) parent;

      if (root.getContentPane() != null) {
        switch (sizeflag) {
          case MINSIZE: {
            cpd = root.getContentPane().getMinimumSize();
            break;
          }
          case PREFSIZE: {
            cpd = root.getContentPane().getPreferredSize();
            break;
          }
          case MAXSIZE: {
            cpd = root.getContentPane().getMaximumSize();
            break;
          }
        }
        if (cpd != null) {
          cpWidth = limit(cpd.width);
          cpHeight = limit(cpd.height);
        }
      }

      if (root.getJMenuBar() != null) {
        switch (sizeflag) {
          case MINSIZE: {
            mbd = root.getJMenuBar().getMinimumSize();
            break;
          }
          case PREFSIZE: {
            mbd = root.getJMenuBar().getPreferredSize();
            break;
          }
          case MAXSIZE: {
            // also preferred size
            mbd = root.getJMenuBar().getPreferredSize();
            break;
          }
        }
        if (mbd != null) {
          mbWidth = limit(mbd.width);
          mbHeight = limit(mbd.height);
        }
      }

      if (root.getWindowDecorationStyle() != JRootPane.NONE && (root.getUI() instanceof SynthRootPaneUI)) {
        JComponent titlePane = ((SynthRootPaneUI) root.getUI()).getTitlePane();
        if (titlePane != null) {
          switch (sizeflag) {
            case MINSIZE: {
              tpd = titlePane.getMinimumSize();
              break;
            }
            case PREFSIZE: {
              tpd = titlePane.getPreferredSize();
              break;
            }
            case MAXSIZE: {
              // also preferred size
              tpd = titlePane.getPreferredSize();
              break;
            }
          }
          if (tpd != null) {
            tpWidth = limit(tpd.width);
            tpHeight = limit(tpd.height);
          }
        }
      }

      Dimension d = new Dimension(
            Math.max(Math.max(tpWidth, mbWidth), cpWidth) + i.left + i.right,
            tpHeight + mbHeight + cpHeight + i.top + i.bottom
            );
      return d;
    }

    private int limit(int i) {
      return Math.min(i, 10240);
    }

    /**
     * Instructs the layout manager to perform the layout for the specified
     * container.
     * 
     * @param the
     *          Container for which this layout manager is being used
     */
    public void layoutContainer(Container parent) {
      JRootPane root = (JRootPane) parent;
      Rectangle b = root.getBounds();
      Insets i = root.getInsets();
      int nextY = 0;
      int w = b.width - i.right - i.left;
      int h = b.height - i.top - i.bottom;

      if (root.getLayeredPane() != null) {
        root.getLayeredPane().setBounds(i.left, i.top, w, h);
      }
      if (root.getGlassPane() != null) {
        root.getGlassPane().setBounds(i.left, i.top, w, h);
      }
      // Note: This is laying out the children in the layeredPane,
      // technically, these are not our children.
      if (root.getWindowDecorationStyle() != JRootPane.NONE && (root.getUI() instanceof SynthRootPaneUI)) {
        JComponent titlePane = ((SynthRootPaneUI) root.getUI()).getTitlePane();
        if (titlePane != null) {
          Dimension tpd = titlePane.getPreferredSize();
          if (tpd != null) {
            int tpHeight = tpd.height;
            titlePane.setBounds(0, 0, w, tpHeight);
            nextY += tpHeight;
          }
        }
      }
      if (root.getJMenuBar() != null) {
        Dimension mbd = root.getJMenuBar().getPreferredSize();
        root.getJMenuBar().setBounds(0, nextY, w, mbd.height);
        nextY += mbd.height;
      }
      if (root.getContentPane() != null) {
        // Dimension cpd = root.getContentPane().getPreferredSize();
        root.getContentPane().setBounds(0, nextY, w, h < nextY ? 0 : h - nextY);
      }
    }

    public void addLayoutComponent(String name, Component comp) {
    }

    public void removeLayoutComponent(Component comp) {
    }

    public void addLayoutComponent(Component comp, Object constraints) {
    }

    public float getLayoutAlignmentX(Container target) {
      return 0.0f;
    }

    public float getLayoutAlignmentY(Container target) {
      return 0.0f;
    }

    public void invalidateLayout(Container target) {
    }
  }

  /**
   * Returns the <code>JComponent</code> rendering the title pane. If this
   * returns null, it implies there is no need to render window decorations.
   * 
   * @return the current window title pane, or null
   * @see #setTitlePane
   */
  protected JComponent getTitlePane() {
    return titlePane;
  }

  protected Window getWindow() {
    return window;
  }

  protected JRootPane getRootPane() {
    return root;
  }
}
