/*
 * Copyright (c) 2002, 2008, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.bsiag.javax.swing.plaf.synth;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Insets;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.Icon;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuItem;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.UIResource;
import javax.swing.plaf.basic.BasicMenuUI;

import com.bsiag.sun.swing.plaf.synth.SynthUI;

/**
 * Synth's MenuUI.
 * 
 * @author Georges Saab
 * @author David Karlton
 * @author Arnaud Weber
 */
class SynthMenuUI extends BasicMenuUI implements PropertyChangeListener,
                               SynthUI {
  private SynthStyle style;
  private SynthStyle accStyle;

  private String acceleratorDelimiter;

  public static ComponentUI createUI(JComponent x) {
    return new SynthMenuUI();
  }

  protected void installDefaults() {
    updateStyle(menuItem);
  }

  protected void installListeners() {
    super.installListeners();
    menuItem.addPropertyChangeListener(this);
  }

  private void updateStyle(JMenuItem mi) {
    SynthStyle oldStyle = style;
    SynthContext context = getContext(mi, ENABLED);

    style = SynthLookAndFeel.updateStyle(context, this);
    if (oldStyle != style) {
      String prefix = getPropertyPrefix();
      defaultTextIconGap = style.getInt(
                           context, prefix + ".textIconGap", 4);
      if (menuItem.getMargin() == null ||
                         (menuItem.getMargin() instanceof UIResource)) {
        Insets insets = (Insets) style.get(context, prefix + ".margin");

        if (insets == null) {
          // Some places assume margins are non-null.
          insets = SynthLookAndFeel.EMPTY_UIRESOURCE_INSETS;
        }
        menuItem.setMargin(insets);
      }
      acceleratorDelimiter = style.getString(context, prefix +
                                            ".acceleratorDelimiter", "+");

      arrowIcon = style.getIcon(context, prefix + ".arrowIcon");

      checkIcon = style.getIcon(context, prefix + ".checkIcon");

      ((JMenu) menuItem).setDelay(style.getInt(context, prefix +
                                                    ".delay", 200));
      if (oldStyle != null) {
        uninstallKeyboardActions();
        installKeyboardActions();
      }
    }
    context.dispose();

    SynthContext accContext = getContext(mi, Region.MENU_ITEM_ACCELERATOR,
                                             ENABLED);

    accStyle = SynthLookAndFeel.updateStyle(accContext, this);
    accContext.dispose();
  }

  protected void uninstallDefaults() {
    SynthContext context = getContext(menuItem, ENABLED);
    style.uninstallDefaults(context);
    context.dispose();
    style = null;

    SynthContext accContext = getContext(menuItem,
                                     Region.MENU_ITEM_ACCELERATOR, ENABLED);
    accStyle.uninstallDefaults(accContext);
    accContext.dispose();
    accStyle = null;

    super.uninstallDefaults();
  }

  protected void uninstallListeners() {
    super.uninstallListeners();
    menuItem.removePropertyChangeListener(this);
  }

  public SynthContext getContext(JComponent c) {
    return getContext(c, getComponentState(c));
  }

  SynthContext getContext(JComponent c, int state) {
    Region region = getRegion(c);
    return SynthContext.getContext(SynthContext.class, c, region,
                                       style, state);
  }

  public SynthContext getContext(JComponent c, Region region) {
    return getContext(c, region, getComponentState(c, region));
  }

  private SynthContext getContext(JComponent c, Region region, int state) {
    return SynthContext.getContext(SynthContext.class, c,
                                       region, accStyle, state);
  }

  private Region getRegion(JComponent c) {
    return SynthLookAndFeel.getRegion(c);
  }

  private int getComponentState(JComponent c) {
    int state;

    if (!c.isEnabled()) {
      return DISABLED;
    }
    if (menuItem.isArmed()) {
      state = MOUSE_OVER;
    }
    else {
      state = SynthLookAndFeel.getComponentState(c);
    }
    if (menuItem.isSelected()) {
      state |= SELECTED;
    }
    return state;
  }

  private int getComponentState(JComponent c, Region region) {
    return getComponentState(c);
  }

  protected Dimension getPreferredMenuItemSize(JComponent c,
                                                     Icon checkIcon,
                                                     Icon arrowIcon,
                                                     int defaultTextIconGap) {
    SynthContext context = getContext(c);
    SynthContext accContext = getContext(c, Region.MENU_ITEM_ACCELERATOR);
    Dimension value = SynthMenuItemUI.getPreferredMenuItemSize(
                context, accContext, useCheckAndArrow(), c, checkIcon,
                arrowIcon, defaultTextIconGap, acceleratorDelimiter);
    context.dispose();
    accContext.dispose();
    return value;
  }

  public void update(Graphics g, JComponent c) {
    SynthContext context = getContext(c);

    SynthLookAndFeel.update(context, g);
    context.getPainter().paintMenuBackground(context,
                          g, 0, 0, c.getWidth(), c.getHeight());
    paint(context, g);
    context.dispose();
  }

  public void paint(Graphics g, JComponent c) {
    SynthContext context = getContext(c);

    paint(context, g);
    context.dispose();
  }

  protected void paint(SynthContext context, Graphics g) {
    SynthContext accContext = getContext(menuItem,
                                             Region.MENU_ITEM_ACCELERATOR);
    SynthStyle style = context.getStyle();
    // Refetch the appropriate check indicator for the current state
    String prefix = getPropertyPrefix();
    Icon arrowIcon = style.getIcon(context, prefix + ".arrowIcon");
    Icon checkIcon = style.getIcon(context, prefix + ".checkIcon");
    SynthMenuItemUI.paint(context, accContext, g, checkIcon, arrowIcon,
                useCheckAndArrow(), acceleratorDelimiter, defaultTextIconGap);
    accContext.dispose();
  }

  public void paintBorder(SynthContext context, Graphics g, int x,
                            int y, int w, int h) {
    context.getPainter().paintMenuBorder(context, g, x, y, w, h);
  }

  public void propertyChange(PropertyChangeEvent e) {
    if (SynthLookAndFeel.shouldUpdateStyle(e)) {
      updateStyle((JMenu) e.getSource());
    }
  }

  private boolean useCheckAndArrow() {
    return !((JMenu) menuItem).isTopLevelMenu();
  }
}
