/*
 * Copyright (c) 2002, 2006, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package com.bsiag.javax.swing.plaf.synth;

import java.awt.Component;
import java.awt.Graphics;
import java.awt.Insets;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JTable;
import javax.swing.RowSorter;
import javax.swing.border.Border;
import javax.swing.border.EmptyBorder;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.UIResource;
import javax.swing.plaf.basic.BasicTableHeaderUI;
import javax.swing.table.JTableHeader;
import javax.swing.table.TableCellRenderer;

import sun.swing.table.DefaultTableCellHeaderRenderer;

import com.bsiag.sun.swing.plaf.synth.SynthUI;

/**
 * SynthTableHeaderUI implementation
 * 
 * @author Alan Chung
 * @author Philip Milne
 */
class SynthTableHeaderUI extends BasicTableHeaderUI implements PropertyChangeListener, SynthUI {

  //
  // Instance Variables
  //

  private TableCellRenderer prevRenderer = null;

  private SynthStyle style;

  public static ComponentUI createUI(JComponent h) {
    return new SynthTableHeaderUI();
  }

  protected void installDefaults() {
    prevRenderer = header.getDefaultRenderer();
    if (prevRenderer instanceof UIResource) {
      header.setDefaultRenderer(new HeaderRenderer());
    }
    updateStyle(header);
  }

  private void updateStyle(JTableHeader c) {
    SynthContext context = getContext(c, ENABLED);
    SynthStyle oldStyle = style;
    style = SynthLookAndFeel.updateStyle(context, this);
    if (style != oldStyle) {
      if (oldStyle != null) {
        uninstallKeyboardActions();
        installKeyboardActions();
      }
    }
    context.dispose();
  }

  protected void installListeners() {
    super.installListeners();
    header.addPropertyChangeListener(this);
  }

  protected void uninstallDefaults() {
    if (header.getDefaultRenderer() instanceof HeaderRenderer) {
      header.setDefaultRenderer(prevRenderer);
    }

    SynthContext context = getContext(header, ENABLED);

    style.uninstallDefaults(context);
    context.dispose();
    style = null;
  }

  protected void uninstallListeners() {
    header.removePropertyChangeListener(this);
    super.uninstallListeners();
  }

  public void update(Graphics g, JComponent c) {
    SynthContext context = getContext(c);

    SynthLookAndFeel.update(context, g);
    context.getPainter().paintTableHeaderBackground(context, g, 0, 0, c.getWidth(), c.getHeight());
    paint(context, g);
    context.dispose();
  }

  public void paint(Graphics g, JComponent c) {
    SynthContext context = getContext(c);

    paint(context, g);
    context.dispose();
  }

  protected void paint(SynthContext context, Graphics g) {
    super.paint(g, context.getComponent());
  }

  public void paintBorder(SynthContext context, Graphics g, int x, int y, int w, int h) {
    context.getPainter().paintTableHeaderBorder(context, g, x, y, w, h);
  }

  //
  // SynthUI
  //
  public SynthContext getContext(JComponent c) {
    return getContext(c, getComponentState(c));
  }

  private SynthContext getContext(JComponent c, int state) {
    return SynthContext.getContext(SynthContext.class, c, SynthLookAndFeel.getRegion(c), style, state);
  }

  private int getComponentState(JComponent c) {
    return SynthLookAndFeel.getComponentState(c);
  }

  public void propertyChange(PropertyChangeEvent evt) {
    if (SynthLookAndFeel.shouldUpdateStyle(evt)) {
      updateStyle((JTableHeader) evt.getSource());
    }
  }

  @Override
  protected void rolloverColumnUpdated(int oldColumn, int newColumn) {
    header.repaint(header.getHeaderRect(oldColumn));
    header.repaint(header.getHeaderRect(newColumn));
  }

  private class HeaderRenderer extends DefaultTableCellHeaderRenderer {
    HeaderRenderer() {
      setHorizontalAlignment(JLabel.LEADING);
      setName("TableHeader.renderer");
    }

    @Override
    public Component getTableCellRendererComponent(JTable table, Object value,
        boolean isSelected,
        boolean hasFocus,
        int row, int column) {

      boolean hasRollover = (column == getRolloverColumn());
      if (isSelected || hasRollover || hasFocus) {
        SynthLookAndFeel.setSelectedUI((SynthLabelUI) SynthLookAndFeel.getUIOfType(getUI(), SynthLabelUI.class),
            isSelected, hasFocus, table.isEnabled(), hasRollover);
      }
      else {
        SynthLookAndFeel.resetSelectedUI();
      }

      //stuff a variable into the client property of this renderer indicating the sort order,
      //so that different rendering can be done for the header based on sorted state.
      RowSorter<?> rs = table == null ? null : table.getRowSorter();
      java.util.List<? extends RowSorter.SortKey> sortKeys = rs == null ? null : rs.getSortKeys();
      if (sortKeys != null && sortKeys.size() > 0 && sortKeys.get(0).getColumn() == table.convertColumnIndexToModel(column)) {
        switch (sortKeys.get(0).getSortOrder()) {
          case ASCENDING:
            putClientProperty("Table.sortOrder", "ASCENDING");
            break;
          case DESCENDING:
            putClientProperty("Table.sortOrder", "DESCENDING");
            break;
          case UNSORTED:
            putClientProperty("Table.sortOrder", "UNSORTED");
            break;
          default:
            throw new AssertionError("Cannot happen");
        }
      }
      else {
        putClientProperty("Table.sortOrder", "UNSORTED");
      }

      super.getTableCellRendererComponent(table, value, isSelected, hasFocus, row, column);
      applyPaddingToFirstAndLastColumn(table, column);

      return this;
    }

    @Override
    public void setBorder(Border border) {
      if (border instanceof SynthBorder) {
        super.setBorder(border);
      }
    }

    /**
     * Use the insets defined for the renderer but use the padding properties defined for the table.
     * 
     * @param table
     * @param column
     */
    private void applyPaddingToFirstAndLastColumn(JTable table, int column) {
      SynthContext tableContext = ((SynthUI) table.getUI()).getContext(table);
      SynthContext rendererContext = ((SynthUI) getUI()).getContext(this);
      Insets insets = rendererContext.getStyle().getInsets(rendererContext, null);
      int padding = tableContext.getStyle().getInt(tableContext, "Table.header.padding", -1);
      if (padding > -1) {
        insets.top = padding;
      }
      int columnCount = table.getColumnCount();
      if (column == 0) {
        padding = tableContext.getStyle().getInt(tableContext, "Table.firstColumn.padding", -1);
        if (padding > -1) {
          insets.left = padding;
        }
      }
      else if (column == columnCount - 1) {
        padding = tableContext.getStyle().getInt(tableContext, "Table.lastColumn.padding", -1);
        if (padding > -1) {
          insets.right = padding;
        }
      }
      if (insets != null) {
        // intentionally omit call of this.setBorder() and call super.setBorder() instead.
        super.setBorder(new EmptyBorder(insets));
      }
    }
  }
}
