/*
 * Copyright (c) 2011, BSI Business Systems Integration AG. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. BSI Business Systems Integration AG
 * designates this particular file as subject to the "Classpath" exception as provided
 * by BSI Business Systems Integration AG in the LICENSE_BSI file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact BSI Business Systems Integration AG, Taefernstrasse 16a,
 * CH-5405 Baden, Switzerland or visit www.bsiag.com if you need additional
 * information or have any questions.
 */
package com.bsiag.scout.rt.ui.swing.laf.rayo.painters;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;

import javax.swing.UIManager;
import javax.swing.plaf.UIResource;
import javax.swing.text.JTextComponent;

import org.eclipse.scout.rt.ui.swing.basic.ColorUtility;
import org.eclipse.scout.rt.ui.swing.ext.LookAndFeelUtility;

/**
 * Calculates the colors used by a text field.
 * 
 * @author awe
 */
public class TextFieldColors {

  Color foreground = UIManager.getColor("TextField.foreground");

  Color background = UIManager.getColor("TextField.background");

  Color border = UIManager.getColor("TextField.border");

  Color border3D = UIManager.getColor("TextField.border3D");

  Color borderTranslucent;

  private boolean compEnabled = false;

  private Color compForeground;

  private Color compBackground;

  public TextFieldColors(Color compForeground, Color compBackground, boolean compEnabled) {
    this.compForeground = compForeground;
    this.compBackground = compBackground;
    this.compEnabled = compEnabled;
    setColors();
  }

  private void setColors() {
    if (useDefaultForegroundColors()) {
      setDefaultForegroundColors();
    }
    else {
      setCustomForegroundColors();
    }
    if (useDefaultBackgroundColors()) {
      setDefaultBackgroundColors();
    }
    else {
      setCustomBackgroundColors();
    }
    borderTranslucent = LookAndFeelUtility.translucent(border, 30);
  }

  boolean useDefaultForegroundColors() {
    if (compForeground != null && !(compForeground instanceof UIResource)) {
      return false;
    }
    return true;
  }

  boolean useDefaultBackgroundColors() {
    if (compBackground != null && !(compBackground instanceof UIResource)) {
      return false;
    }
    return true;
  }

  private void setCustomForegroundColors() {
    if (compForeground != null) {
      foreground = compForeground;
    }
  }

  private void setCustomBackgroundColors() {
    if (compBackground != null) {
      background = compBackground;
    }
    if (!compEnabled) {
      background = ColorUtility.lighter(background, 0.7f);
    }
    if (compBackground != null) {
      border = ColorUtility.darker(background, 0.2f);
      border3D = ColorUtility.darker(background, 0.5f);
    }
  }

  private void setDefaultForegroundColors() {
    // nop
  }

  private void setDefaultBackgroundColors() {
    if (!compEnabled) {
      background = UIManager.getColor("TextField.backgroundDisabled");
    }
  }

  public static TextFieldColors createFromComponent(Component comp) {
    // We want to draw fields as "disabled" not only when they ARE disabled, but
    // also when they are enabled but not editable. This is a workaround, see
    // comment in org.eclipse.scout.rt.ui.swing.form.fields.SwingScoutFieldComposite.setEnabledFromScout(boolean)
    // for more information.
    // TODO BSH Shouldn't we consider the SynthContext, too? Maybe this code should also be added to SynthLookAndFeel.getComponentState(Component)!
    boolean seemsEnabled = comp.isEnabled();
    if (comp instanceof JTextComponent) {
      JTextComponent jtc = (JTextComponent) comp;
      seemsEnabled &= jtc.isEditable();
    }
    return new TextFieldColors(comp.getForeground(), comp.getBackground(), seemsEnabled);
  }

  public static TextFieldColors createFromContainer(Container container) {
    if (container == null) {
      return null;
    }
    return new TextFieldColors(container.getForeground(), container.getBackground(), container.isEnabled());
  }

  public boolean requiresBorder3D() {
    return compEnabled;
  }
}
