/*
 * Copyright (c) 2011, 2012 BSI Business Systems Integration AG. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. BSI Business Systems Integration AG
 * designates this particular file as subject to the "Classpath" exception as provided
 * by BSI Business Systems Integration AG in the LICENSE_BSI file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact BSI Business Systems Integration AG, Taefernstrasse 16a,
 * CH-5405 Baden, Switzerland or visit www.bsiag.com if you need additional
 * information or have any questions.
 */
package com.bsiag.scout.rt.ui.swing.laf.rayo;

import java.io.InputStream;
import java.text.ParseException;

import javax.swing.JDialog;
import javax.swing.JFrame;
import javax.swing.UIManager;
import javax.swing.UnsupportedLookAndFeelException;

import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.rt.ui.swing.ILookAndFeelConfigurator;
import org.eclipse.scout.rt.ui.swing.ILookAndFeelProvider;
import org.eclipse.scout.rt.ui.swing.SwingUtility;

/**
 * Installs the rayo look and feel which is based on synth look and feel.
 * 
 * @author awe
 */
public class Rayo implements ILookAndFeelProvider {

  private static final IScoutLogger LOG = ScoutLogManager.getLogger(Rayo.class);

  private static final String EXTENSION_POINT_PLUGIN_ID = "org.eclipse.scout.rt.ui.swing";
  private static final String EXTENSION_POINT_ID = "lafconfigurator";

  /**
   * bsi ticket 86924, 111082
   * When using L&F decorated frame and dialog, the Windows-UP/DOWN/LEFT/RIGHT keys do not work anymore. Also resizing
   * issues arise and in general some native OS functions are simply disabled. Thus do not use true here. Nimbus, Metal
   * and Synth in JRE are also not using this feature due to the mentioned issues.
   * <p>
   * bsi ticket 117148 Depending on whether OS or Look & Feel decorated frames and dialogs is used, it should be
   * possible to configure the borders in the config.ini file. useLafFrameAndDialog == true or unset <=> Rayo Look &
   * Feel borders useLafFrameAndDialog == false <=> OS decorated borders
   */
  private static class DefaultRayoConfigurator implements ILookAndFeelConfigurator {

    private boolean useLafFrameAndDialog = SwingUtility.hasScoutLookAndFeelFrameAndDialog();

    @Override
    public void configure() {
      JFrame.setDefaultLookAndFeelDecorated(useLafFrameAndDialog);
      JDialog.setDefaultLookAndFeelDecorated(useLafFrameAndDialog);
      System.setProperty("awt.useSystemAAFontSettings", "off");
      System.setProperty("swing.aatext", "false");
    }

    @Override
    public InputStream getConfigurationFile() {
      String rayoXmlFile = (useLafFrameAndDialog ? "rayoFrameAndDialog.xml" : "rayo.xml");
      return getClass().getResourceAsStream(rayoXmlFile);
    }

  }

  @Override
  public void installLookAndFeel() {
    com.bsiag.javax.swing.plaf.synth.SynthLookAndFeel synthLaf = new com.bsiag.javax.swing.plaf.synth.SynthLookAndFeel();
    try {
      ILookAndFeelConfigurator configurator = readExtensionPoint();
      configurator.configure();
      synthLaf.load(configurator.getConfigurationFile(), Rayo.class);
      UIManager.setLookAndFeel(synthLaf);
    }
    catch (ParseException e) {
      throw new RuntimeException(e);
    }
    catch (UnsupportedLookAndFeelException e) {
      throw new RuntimeException(e);
    }
  }

  private ILookAndFeelConfigurator readExtensionPoint() {
    IExtensionPoint ext = Platform.getExtensionRegistry().getExtensionPoint(EXTENSION_POINT_PLUGIN_ID, EXTENSION_POINT_ID);
    for (IConfigurationElement element : ext.getConfigurationElements()) {
      try {
        ILookAndFeelConfigurator configurator = (ILookAndFeelConfigurator) element.createExecutableExtension("class");
        LOG.info("using contributed laf configurator class " + configurator.getClass().getCanonicalName() + " " + configurator.toString());
        return configurator;
      }
      catch (CoreException e) {
        LOG.error("failed to create instance of ILookAndFeelConfigurator", e);
      }
    }
    LOG.info("using default Rayo configurator");
    return new DefaultRayoConfigurator();
  }

}
