/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package com.bsiag.scout.rt.ui.swing.rayo.window.desktop.toolbar;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.HashMap;
import java.util.Map;

import javax.swing.Icon;
import javax.swing.UIManager;

import org.eclipse.scout.rt.client.ui.desktop.IDesktop;
import org.eclipse.scout.rt.ui.swing.Activator;
import org.eclipse.scout.rt.ui.swing.ISwingEnvironment;
import org.eclipse.scout.rt.ui.swing.ext.job.SwingProgressHandler;
import org.eclipse.scout.rt.ui.swing.window.desktop.toolbar.AbstractJNavigationPanel;

import com.bsiag.scout.rt.ui.swing.rayo.RayoIcons;
import com.bsiag.scout.rt.ui.swing.rayo.window.desktop.toolbar.RayoNavigationWidgetButton.ActionType;
import com.bsiag.scout.rt.ui.swing.rayo.window.desktop.toolbar.RayoNavigationWidgetButton.ButtonName;

public class RayoJNavigationPanel extends AbstractJNavigationPanel {

  private static final long serialVersionUID = 1L;
  static final int INSET_TOP = 11;
  static final int INSET_LEFT = 9;
  private static final int SHADOW_DISTANCE = 2;
  private static final Dimension SIZE = new Dimension(INSET_LEFT + 109 + SHADOW_DISTANCE, INSET_TOP + 40 + SHADOW_DISTANCE);
  private static final Point HISTORY_MENU_LOCATION = new Point(INSET_LEFT + 38, INSET_TOP + 38);

  private RayoNavigationWidgetButton m_mouseOverButton;
  private ButtonName m_dropshadowBelowButton = ButtonName.NONE;
  private Map<ButtonName, RayoNavigationWidgetButton> m_buttonMap = new HashMap<ButtonName, RayoNavigationWidgetButton>();

  public RayoJNavigationPanel(ISwingEnvironment env) {
    super(env);

    setPreferredSize(SIZE);
    setOpaque(false);
    initButtons();

    //progress listener
    SwingProgressHandler progressHandler = SwingProgressHandler.getInstance();
    if (progressHandler != null) {
      progressHandler.addStateChangeListener(new SwingProgressHandler.IStateChangeListener() {
        @Override
        public void stateChanged(SwingProgressHandler h) {
          RayoStopRefreshButton button = getStopRefreshButton();
          if (button == null) {
            return;
          }
          button.setLoading(h.isJobRunning());
        }
      });
    }

    addMouseMotionListener(new MouseMotionAdapter() {
      @Override
      public void mouseMoved(MouseEvent e) {
        detectMouseOverButton(e.getX(), e.getY());
      }
    });

    addMouseListener(new MouseAdapter() {
      @Override
      public void mouseExited(MouseEvent e) {
        setMouseOverButton(ButtonName.NONE);
      }

      @Override
      public void mousePressed(MouseEvent e) {
        setPressedButton(true);
      }

      @Override
      public void mouseReleased(MouseEvent e) {
        setPressedButton(false);
        buttonClicked(e.getButton());
      }
    });
  }

  private boolean isNavigationHistoryEnabled() {
    return UIManager.getBoolean("Synth.NavigationPanel.showHistory");
  }

  private void initButtons() {
    RayoNavigationWidgetButton back = new RayoNavigationWidgetButton(ButtonName.BACK, this);
    back.setButtonPos(new Point(INSET_LEFT + 5, INSET_TOP + 4));
    back.setIconPos(new Point(INSET_LEFT + 13, INSET_TOP + 9));
    m_buttonMap.put(ButtonName.BACK, back);
    addDisabledListener(back);

    final RayoNavigationWidgetButton forward;
    if (isNavigationHistoryEnabled()) {
      Point historyIconPosition = new Point(INSET_LEFT + 57, INSET_TOP + 9);
      forward = new RayoForwardButton(this, historyIconPosition, back.isEnabled());
    }
    else {
      forward = new RayoNavigationWidgetButton(ButtonName.FORWARD, this);
    }
    forward.setButtonPos(new Point(INSET_LEFT + 39, INSET_TOP + 4));
    forward.setIconPos(new Point(INSET_LEFT + 44, INSET_TOP + 9));
    forward.setEnabled(false);
    m_buttonMap.put(ButtonName.FORWARD, forward);
    addDisabledListener(forward);

    RayoStopRefreshButton stopRefresh = new RayoStopRefreshButton(this);
    stopRefresh.setButtonPos(new Point(INSET_LEFT + 73, INSET_TOP + 4));
    stopRefresh.setIconPos(new Point(INSET_LEFT + 79, INSET_TOP + 11));
    stopRefresh.loadGlowAnimation(RayoIcons.NavigationGlow);
    stopRefresh.showStopIcon();
    m_buttonMap.put(ButtonName.STOP_REFRESH, stopRefresh);
    addDisabledListener(stopRefresh);

    if (isNavigationHistoryEnabled()) {
      PropertyChangeListener historyEnabledListener = new PropertyChangeListener() {
        @Override
        public void propertyChange(PropertyChangeEvent evt) {
          if (RayoNavigationWidgetButton.PROP_ENABLED.equals(evt.getPropertyName())) {
            boolean enabled = (Boolean) evt.getNewValue();
            if (forward instanceof RayoForwardButton) {
              ((RayoForwardButton) forward).setHistoryEnabled(enabled);
            }
          }
        }
      };
      back.addPropertyChangeListener(historyEnabledListener);
    }
  }

  @Override
  public void rebuild(IDesktop desktop) {
    // initialize buttons with actions

    getBackButton().setPrimaryAction(getBackAction());
    getBackButton().setWidgetIconId(RayoIcons.NavigationWidgetBack);
    getBackButton().setIconId(ActionType.PrimaryAction, RayoIcons.NavigationBack);
    getBackButton().setActiveActionType(ActionType.PrimaryAction);

    getForwardButton().setPrimaryAction(getForwardAction());
    getForwardButton().setWidgetIconId(RayoIcons.NavigationWidgetForward);
    getForwardButton().setIconId(ActionType.PrimaryAction, RayoIcons.NavigationForward);
    getForwardButton().setActiveActionType(ActionType.PrimaryAction);
    if (isNavigationHistoryEnabled()) {
      getForwardButton().setSecondaryAction(getHistoryAction());
      getForwardButton().setIconId(ActionType.SecondaryAction, RayoIcons.NavigationHistory);
    }

    getStopRefreshButton().setPrimaryAction(getRefreshAction());
    getStopRefreshButton().setSecondaryAction(getStopAction());
    getStopRefreshButton().setWidgetIconId(RayoIcons.NavigationWidgetStopRefresh);
    getStopRefreshButton().setIconId(ActionType.PrimaryAction, RayoIcons.NavigationRefresh);
    getStopRefreshButton().setIconId(ActionType.SecondaryAction, RayoIcons.NavigationStop);
    getStopRefreshButton().setActiveActionType(ActionType.PrimaryAction);

    repaint();
  }

  /**
   * Remove orange glow below button, when a button is disabled.
   * 
   * @param button
   */
  private void addDisabledListener(RayoNavigationWidgetButton button) {
    button.addPropertyChangeListener(new PropertyChangeListener() {
      @Override
      public void propertyChange(PropertyChangeEvent evt) {
        if (RayoNavigationWidgetButton.PROP_ENABLED.equals(evt.getPropertyName())) {
          boolean enabled = (Boolean) evt.getNewValue();
          if (!enabled) {
            RayoNavigationWidgetButton disabledButton = (RayoNavigationWidgetButton) evt.getSource();
            if (m_mouseOverButton != null && m_mouseOverButton.getName() == disabledButton.getName()) {
              setMouseOverButton(ButtonName.NONE);
            }
          }
        }
      }
    });
  }

  // TODO awe: bessere kollisions erkennung für runde bereiche (evt. B/W bitmap verwenden)
  private void detectMouseOverButton(int x, int y) {
    ButtonName moButtonName = ButtonName.getMouseOverButton(x, y);
    setMouseOverButton(moButtonName);
  }

  @Override
  protected void paintComponent(Graphics g) {
    super.paintComponent(g);
    paintDropshadow(g);
    paintButtons(g);
  }

  private void paintDropshadow(Graphics g) {
    Icon icon = Activator.getIcon(m_dropshadowBelowButton.getDropshadowIcon());
    if (icon != null) {
      icon.paintIcon(this, g, INSET_LEFT + SHADOW_DISTANCE, INSET_TOP + SHADOW_DISTANCE);
    }
  }

  private RayoNavigationWidgetButton getButton(ButtonName button) {
    return m_buttonMap.get(button);
  }

  private void paintButtons(Graphics g) {
    getButton(ButtonName.BACK).paintComponent(this, g);
    getButton(ButtonName.FORWARD).paintComponent(this, g);
    getButton(ButtonName.STOP_REFRESH).paintComponent(this, g);
  }

  private void setPressedButton(boolean pressed) {
    if (m_mouseOverButton != null) {
      m_mouseOverButton.setPressed(pressed);
    }
  }

  private void buttonClicked(int button) {
    if (m_mouseOverButton != null) {
      m_mouseOverButton.buttonClicked(button);
    }
  }

  private void setMouseOverButton(ButtonName moButtonName) {
    if (m_mouseOverButton != null) {
      m_mouseOverButton.setMouseOver(false);
      m_mouseOverButton.setPressed(false);
    }

    m_mouseOverButton = getButton(moButtonName);
    if (m_mouseOverButton != null && m_mouseOverButton.isEnabled()) {
      m_dropshadowBelowButton = moButtonName;
    }
    else {
      m_dropshadowBelowButton = ButtonName.NONE;
    }

    if (m_mouseOverButton != null) {
      m_mouseOverButton.setMouseOver(true);
    }
  }

  public RayoNavigationWidgetButton getBackButton() {
    return getButton(ButtonName.BACK);
  }

  public RayoNavigationWidgetButton getForwardButton() {
    return getButton(ButtonName.FORWARD);
  }

  public RayoStopRefreshButton getStopRefreshButton() {
    return (RayoStopRefreshButton) getButton(ButtonName.STOP_REFRESH);
  }

  @Override
  public Point getHistoryMenuLocation() {
    return HISTORY_MENU_LOCATION;
  }
}
