/*
 * Copyright (c) 2011, BSI Business Systems Integration AG. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. BSI Business Systems Integration AG
 * designates this particular file as subject to the "Classpath" exception as provided
 * by BSI Business Systems Integration AG in the LICENSE_BSI file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact BSI Business Systems Integration AG, Taefernstrasse 16a,
 * CH-5405 Baden, Switzerland or visit www.bsiag.com if you need additional
 * information or have any questions.
 */
package com.bsiag.scout.rt.ui.swing.laf.rayo.painters;

import java.awt.Color;
import java.awt.Component;
import java.awt.Graphics;
import java.awt.Rectangle;

import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.plaf.ColorUIResource;

import com.bsiag.javax.swing.plaf.synth.ColorType;
import com.bsiag.javax.swing.plaf.synth.SynthContext;
import com.bsiag.javax.swing.plaf.synth.SynthPainter;
import com.bsiag.javax.swing.plaf.synth.SynthTabbedPaneUI;
import com.bsiag.sun.swing.plaf.synth.CustomSynthNames;

/**
 * @author awe
 */
public class RayoPainter extends SynthPainter {

  private static final int ARROW_HEIGHT = 6;

  @Override
  public void paintTextFieldBackground(SynthContext context, Graphics g, int x, int y, int width, int height) {
    // Clear the area for the textfield with the parent's background color.
    // This is necessary, because Rayo textfields have rounded corners.
    TextFieldColors bgColors = TextFieldColors.createFromContainer(context.getComponent().getParent());
    g.setColor(bgColors.background);
    g.fillRect(x, y, width, height);
    // Now draw the field itself
    TextFieldColors colors = TextFieldColors.createFromComponent(context.getComponent());
    g.setColor(colors.background);
    g.fillRoundRect(x, y, width - 1, height - 1, 5, 5);
    paintBorder(colors, g, x, y, width, height);
  }

  @Override
  public void paintTextAreaBackground(SynthContext context, Graphics g, int x, int y, int w, int h) {
    // Clear the area for the textfield with the parent's background color.
    // This is necessary, because Rayo textfields have rounded corners.
    TextFieldColors bgColors = TextFieldColors.createFromContainer(context.getComponent().getParent());
    g.setColor(bgColors.background);
    g.fillRect(x, y, w, h);
    // Now draw the field itself
    TextFieldColors colors = TextFieldColors.createFromComponent(context.getComponent());
    g.setColor(colors.background);
    g.fillRect(x, y, w, h);

    // Note: No border for TextAreas (the surrounding ScrollPane will paint one already)
  }

  @Override
  public void paintPasswordFieldBackground(SynthContext context, Graphics g, int x, int y, int w, int h) {
    paintTextFieldBackground(context, g, x, y, w, h);
  }

  @Override
  public void paintScrollPaneBorder(SynthContext context, Graphics g, int x, int y, int w, int h) {
    // Use color from inner component to draw the border
    TextFieldColors colors = TextFieldColors.createFromComponent(guessInnerComponent(context.getComponent()));
    if (CustomSynthNames.TABLE_SCROLL_PANE.equals(context.getComponent().getName())) {
      // No 3D border for tables (bsh 2010-09-28)
      colors.border3D = colors.border;
    }
    paintBorder(colors, g, x, y, w, h);
  }

  @Override
  public void paintPanelBorder(SynthContext context, Graphics g, int x, int y, int w, int h) {
    TextFieldColors colors = TextFieldColors.createFromComponent(guessInnerComponent(context.getComponent()));
    paintBorder(colors, g, x, y, w, h);
  }

  @Override
  public void paintScrollPaneBackground(SynthContext context, Graphics g, int x, int y, int w, int h) {
    // Clear the area for the textfield with the parent's background color.
    // This is necessary, because Rayo textfields have rounded corners.
    TextFieldColors bgColors = TextFieldColors.createFromContainer(context.getComponent().getParent());
    g.setColor(bgColors.background);
    g.fillRect(x, y, w, h);
    // Use color from inner component to draw the background (because the
    // background of the inner component will not reach to the border of
    // the scroll pane).
    TextFieldColors colors = TextFieldColors.createFromComponent(guessInnerComponent(context.getComponent()));
    g.setColor(colors.background);
    g.fillRoundRect(x, y, w - 1, h - 1, 5, 5);
  }

  /**
   * (bsh 2010-09-29)
   * Method to get the inner component of a scroll pane. If the given
   * component is not a scroll pane or if the scroll pane does not
   * contain a component, the component itself is returned.
   */
  private Component guessInnerComponent(Component presumablyAScrollPane) {
    Component innerComponent = null;
    if (presumablyAScrollPane instanceof JScrollPane) {
      JScrollPane sp = (JScrollPane) presumablyAScrollPane;
      if (sp.getViewport() != null) {
        innerComponent = sp.getViewport().getView();
      }
    }
    if (innerComponent == null) {
      innerComponent = presumablyAScrollPane; // fallback
    }
    return innerComponent;
  }

  @Override
  public void paintTabbedPaneTabBorder(SynthContext context, Graphics g, int x, int y, int w, int h, int tabIndex) {
    JTabbedPane tabbedPane = (JTabbedPane) context.getComponent();
    if (tabIndex == tabbedPane.getSelectedIndex()) {
      Rectangle tabAreaBounds = ((SynthTabbedPaneUI) tabbedPane.getUI()).getTabAreaBounds();
      Color arrowColor = getLAFOrTabbedPaneColor(context, ColorType.FOREGROUND);
      g.setColor(arrowColor);
      paintSelectedTabArrow(g, x, y, w, tabAreaBounds.height);
    }
    super.paintTabbedPaneTabBorder(context, g, x, y, w, h, tabIndex);
  }

  /**
   * Returns the background color of the JTabbedPane or the Synth color defined by colorType.
   * The colorType is different since its always the same JTabbedPane but different Synth regions
   * which call this method.
   * 
   * @param context
   * @param colorType
   * @return
   */
  private Color getLAFOrTabbedPaneColor(SynthContext context, ColorType colorType) {
    Color color = context.getComponent().getBackground();
    if (color == null || color instanceof ColorUIResource || !context.getComponent().isBackgroundSet()) {
      color = context.getStyle().getColor(context, colorType);
    }
    return color;
  }

  @Override
  public void paintTabbedPaneContentBackground(SynthContext context, Graphics g, int x, int y, int w, int h) {
    Color bgColor = getLAFOrTabbedPaneColor(context, ColorType.BACKGROUND);
    // <bsh 2010-09-24>
    // Set the background color of the tabbed pane as an explicit color. Otherwise,
    // inner controls couldn't know that they are painted on an non-default color
    // and would assume white (which is not correct an a tabbed pane).
    // By constructing a new Color object, the class TextFieldColors will use it as a
    // background for inner fields (because it is not instanceof UIResource anymore).
    Color myColor = new Color(bgColor.getRGB());
    context.getComponent().setBackground(myColor);
    // </bsh>
    g.setColor(bgColor);
    g.fillRect(x, y, w, h);
  }

  private void paintSelectedTabArrow(Graphics g, int x, int y, int w, int h) {
    int iconY, iconX, iconWidth;
    for (int i = 0; i < ARROW_HEIGHT; i++) {
      iconY = h - ARROW_HEIGHT + i;
      iconWidth = i * 2;
      iconX = x + (w / 2) - i;
      g.drawLine(iconX, iconY, iconX + iconWidth, iconY);
    }
  }

  private void paintBorder(TextFieldColors colors, Graphics g, int x, int y, int width, int height) {
    g.setColor(colors.border);
    g.drawRoundRect(x, y, width - 1, height - 1, 5, 5);
    if (colors.requiresBorder3D()) {
      paintBorder3D(g, y, width, colors.border3D);
    }
    paintAntiAliasing(g, width, height, colors.borderTranslucent);
  }

  private void paintBorder3D(Graphics g, int y, int width, Color border3D) {
    g.setColor(border3D);
    g.drawLine(3, y, width - 1 - 3, y);
  }

  private void paintAntiAliasing(Graphics g, int width, int height, Color borderTranslucent) {
    g.setColor(borderTranslucent);
    pixel(g, 1, 0);
    pixel(g, 0, 1);
    pixel(g, 2, 1);
    pixel(g, 1, 2);

    pixel(g, width - 2, 0);
    pixel(g, width - 1, 1);
    pixel(g, width - 3, 1);
    pixel(g, width - 2, 2);

    pixel(g, width - 2, height - 3);
    pixel(g, width - 1, height - 2);
    pixel(g, width - 3, height - 2);
    pixel(g, width - 2, height - 1);

    pixel(g, 1, height - 3);
    pixel(g, 0, height - 2);
    pixel(g, 2, height - 2);
    pixel(g, 1, height - 1);
  }

  private void pixel(Graphics g, int x, int y) {
    g.drawLine(x, y, x, y);
  }
}
