/*
 * Copyright (c) 1997, 2005, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */
package com.bsiag.javax.swing.plaf.synth;

import java.awt.Color;
import java.awt.Component;
import java.awt.Container;
import java.awt.Dialog;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Frame;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Insets;
import java.awt.LayoutManager;
import java.awt.Rectangle;
import java.awt.Window;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.event.WindowListener;
import java.awt.event.WindowStateListener;

import javax.accessibility.AccessibleContext;
import javax.swing.AbstractAction;
import javax.swing.Action;
import javax.swing.ActionMap;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JRootPane;
import javax.swing.JSeparator;
import javax.swing.SwingUtilities;
import javax.swing.UIManager;
import javax.swing.plaf.ActionMapUIResource;

import sun.swing.SwingUtilities2;

/**
 * The class that manages a root pane title bar
 * <p>
 * <strong>Warning:</strong> Serialized objects of this class will not be compatible with future Swing releases. The
 * current serialization support is appropriate for short term storage or RMI between applications running the same
 * version of Swing. As of 1.4, support for long term storage of all JavaBeans<sup><font size="-2">TM</font></sup> has
 * been added to the <code>java.beans</code> package. Please see {@link java.beans.XMLEncoder}.
 * 
 * @author David Kloba
 * @author Steve Wilson
 */
public class BasicRootPaneTitlePane extends JComponent {
  protected JRootPane rootPane;

  protected Icon maximizeIcon;
  protected Icon minimizeIcon;
  protected Icon iconifyIcon;
  protected Icon closeIcon;
  protected Icon maximizeIconRollOver;
  protected Icon minimizeIconRollOver;
  protected Icon iconifyIconRollOver;
  protected Icon closeIconRollOver;

  protected Action closeAction;
  protected Action maximizeAction;
  protected Action minimizeAction;
  protected Action iconifyAction;

  private String closeButtonToolTip;
  private String iconifyButtonToolTip;
  private String minimizeButtonToolTip;
  private String maximizeButtonToolTip;

  protected JButton iconifyButton;
  protected JButton toggleButton;
  protected JButton closeButton;

  protected Color selectedTitleColor;
  protected Color selectedTextColor;
  protected Color notSelectedTitleColor;
  protected Color notSelectedTextColor;

  protected JMenu windowMenu;
  protected JMenuBar menuBar;

  protected WindowStateListener windowStateListener;
  protected WindowListener windowListener;

  private Window window;
  private int windowState;

  public BasicRootPaneTitlePane(JRootPane rootPane) {
    this.rootPane = rootPane;
    installDefaults();
    createActions();
    createButtons();
    setLayout(createLayout());
    assembleSystemMenu();
    addSubComponents();
    createActionMap();
  }

  @Override
  public String getUIClassID() {
    return "RootPaneTitlePaneUI";
  }

  private Window getWindow() {
    return window;
  }

  @Override
  public void addNotify() {
    super.addNotify();
    uninstallListeners();
    window = SwingUtilities.getWindowAncestor(this);
    if (window != null) {
      if (window instanceof Frame) {
        windowState = ((Frame) window).getExtendedState();
      }
      installListeners();
      updateActionStyles();
      updateButtonStates();
    }
  }

  @Override
  public void removeNotify() {
    super.removeNotify();
    uninstallListeners();
    window = null;
  }

  /**
   * Returns the Frame rendering in. This will return null if the <code>JRootPane</code> is not contained in a
   * <code>Frame</code>.
   */
  protected Frame getFrame() {
    Window window = getWindow();
    if (window instanceof Frame) {
      return (Frame) window;
    }
    return null;
  }

  protected void doClose() {
    Window window = getWindow();
    if (window != null) {
      window.dispatchEvent(new WindowEvent(window, WindowEvent.WINDOW_CLOSING));
    }
  }

  protected void doIconify() {
    Frame frame = getFrame();
    if (frame != null) {
      int state = frame.getExtendedState();
      frame.setExtendedState(state | Frame.ICONIFIED);
    }
  }

  protected void doMaximize() {
    Frame frame = getFrame();
    if (frame != null) {
      int state = frame.getExtendedState();
      frame.setExtendedState(state | Frame.MAXIMIZED_BOTH);
    }
  }

  protected void doMinimize() {
    Frame frame = getFrame();
    if (frame != null) {
      int state = frame.getExtendedState();
      state = state & ~Frame.ICONIFIED;
      state = state & ~Frame.MAXIMIZED_BOTH;
      frame.setExtendedState(state);
    }
  }

  protected boolean isFrameIconified() {
    Frame frame = getFrame();
    return frame != null && (windowState & Frame.ICONIFIED) != 0;
  }

  protected boolean isFrameMaximized() {
    Frame frame = getFrame();
    return frame != null && (windowState & Frame.MAXIMIZED_BOTH) != 0;
  }

  protected boolean isFrameClosable() {
    return true;
  }

  protected boolean isFrameIconifiable() {
    return getFrame() != null;
  }

  protected boolean isFrameMaximizable() {
    return getFrame() != null;
  }

  /**
   * Updates state dependant upon the Window's active state.
   */
  private void setActive(boolean isActive) {
    if (rootPane.getWindowDecorationStyle() != JRootPane.NONE) {
      Boolean activeB = isActive ? Boolean.TRUE : Boolean.FALSE;

      if (iconifyButton != null) iconifyButton.putClientProperty("paintActive", activeB);
      if (closeButton != null) closeButton.putClientProperty("paintActive", activeB);
      if (toggleButton != null) toggleButton.putClientProperty("paintActive", activeB);
    }
    // Repaint the whole thing as the Borders that are used have
    // different colors for active vs inactive
    getRootPane().repaint();
  }

  protected void addSubComponents() {
    add(menuBar);
    add(iconifyButton);
    add(toggleButton);
    add(closeButton);
  }

  protected void createActions() {
    maximizeAction = new MaximizeAction();
    minimizeAction = new MinimizeAction();
    iconifyAction = new IconifyAction();
    closeAction = new CloseAction();
  }

  ActionMap createActionMap() {
    ActionMap map = new ActionMapUIResource();
    map.put("showSystemMenu", new ShowSystemMenuAction(true));
    map.put("hideSystemMenu", new ShowSystemMenuAction(false));
    return map;
  }

  protected void installListeners() {
    if (windowStateListener == null) {
      windowStateListener = createWindowStateListener();
    }
    if (windowListener == null) {
      windowListener = createWindowListener();
    }
    if (window != null) {
      window.addWindowStateListener(windowStateListener);
      window.addWindowListener(windowListener);
    }
  }

  protected void uninstallListeners() {
    if (window != null) {
      if (windowListener != null) {
        window.removeWindowListener(windowListener);
      }
      if (windowStateListener != null) {
        window.removeWindowStateListener(windowStateListener);
      }
    }
  }

  protected void installDefaults() {
    maximizeIcon = UIManager.getIcon("InternalFrameTitlePane.maximizeIcon");
    minimizeIcon = UIManager.getIcon("InternalFrameTitlePane.minimizeIcon");
    iconifyIcon = UIManager.getIcon("InternalFrameTitlePane.iconifyIcon");
    closeIcon = UIManager.getIcon("InternalFrameTitlePane.closeIcon");
    selectedTitleColor = UIManager.getColor("InternalFrameTitlePane.activeTitleBackground");
    selectedTextColor = UIManager.getColor("InternalFrameTitlePane.activeTitleForeground");
    notSelectedTitleColor = UIManager.getColor("InternalFrameTitlePane.inactiveTitleBackground");
    notSelectedTextColor = UIManager.getColor("InternalFrameTitlePane.inactiveTitleForeground");
    setFont(UIManager.getFont("InternalFrameTitlePane.titleFont"));
    closeButtonToolTip = UIManager.getString("InternalFrameTitlePane.closeButtonToolTip");
    iconifyButtonToolTip = UIManager.getString("InternalFrameTitlePane.iconButtonToolTip");
    minimizeButtonToolTip = UIManager.getString("InternalFrameTitlePane.restoreButtonToolTip");
    maximizeButtonToolTip = UIManager.getString("InternalFrameTitlePane.maxButtonToolTip");
  }

  protected void uninstallDefaults() {
  }

  protected void createButtons() {
    iconifyButton = new NoFocusButton(
        "InternalFrameTitlePane.iconifyButtonAccessibleName",
        "InternalFrameTitlePane.iconifyButtonOpacity");
    iconifyButton.getAccessibleContext().setAccessibleName("Iconify");
    if (iconifyButtonToolTip != null && iconifyButtonToolTip.length() != 0) {
      iconifyButton.setToolTipText(iconifyButtonToolTip);
    }

    toggleButton = new NoFocusButton(
        "InternalFrameTitlePane.maximizeButtonAccessibleName",
        "InternalFrameTitlePane.maximizeButtonOpacity");
    toggleButton.getAccessibleContext().setAccessibleName("Maximize");

    closeButton = new NoFocusButton(
        "InternalFrameTitlePane.closeButtonAccessibleName",
        "InternalFrameTitlePane.closeButtonOpacity");
    closeButton.getAccessibleContext().setAccessibleName("Close");
    if (closeButtonToolTip != null && closeButtonToolTip.length() != 0) {
      closeButton.setToolTipText(closeButtonToolTip);
    }
  }

  protected void assembleSystemMenu() {
    menuBar = createSystemMenuBar();
    windowMenu = createSystemMenu();
    menuBar.add(windowMenu);
    addSystemMenuItems(windowMenu);
  }

  protected void addSystemMenuItems(JMenu systemMenu) {
    JMenuItem mi = (JMenuItem) systemMenu.add(iconifyAction);
    mi.setMnemonic('n');
    mi = (JMenuItem) systemMenu.add(maximizeAction);
    mi.setMnemonic('x');
    systemMenu.add(new JSeparator());
    mi = (JMenuItem) systemMenu.add(closeAction);
    mi.setMnemonic('C');
  }

  protected JMenu createSystemMenu() {
    return new JMenu("    ");
  }

  protected JMenuBar createSystemMenuBar() {
    menuBar = new SystemMenuBar();
    menuBar.setBorderPainted(false);
    return menuBar;
  }

  protected void showSystemMenu() {
    //      windowMenu.setPopupMenuVisible(true);
    //      windowMenu.setVisible(true);
    windowMenu.doClick();
  }

  public void paintComponent(Graphics g) {
    // As state isn't bound, we need a convenience place to check if it has changed.
    if (getFrame() != null) {
      setState(getFrame().getExtendedState());
    }
    //
    paintTitleBackground(g);
    String title = getTitle();
    if (title != null) {
      Font f = g.getFont();
      g.setFont(getFont());
      g.setColor(rootPane.getForeground());

      // Center text vertically.
      FontMetrics fm = rootPane.getFontMetrics(g.getFont());
      int baseline = (getHeight() + fm.getAscent() - fm.getLeading() -
          fm.getDescent()) / 2;

      int titleX;
      Rectangle r = new Rectangle(0, 0, 0, 0);
      if (isFrameIconifiable()) r = iconifyButton.getBounds();
      else if (isFrameMaximizable()) r = toggleButton.getBounds();
      else if (isFrameClosable()) r = closeButton.getBounds();
      int titleW;

      if (r.x == 0) r.x = rootPane.getWidth() - rootPane.getInsets().right;
      titleX = menuBar.getX() + menuBar.getWidth() + 2;
      titleW = r.x - titleX - 3;
      title = getTitle(title, fm, titleW);
      SwingUtilities2.drawString(this, g, title, titleX, baseline);
      g.setFont(f);
    }
  }

  protected void setState(int state) {
    if (state != windowState) {
      windowState = state;
      updateButtonStates();
      revalidate();
      repaint();
    }
  }

  /**
   * Invoked from paintComponent.
   * Paints the background of the titlepane. All text and icons will
   * then be rendered on top of this background.
   * 
   * @param g
   *          the graphics to use to render the background
   * @since 1.4
   */
  protected void paintTitleBackground(Graphics g) {
    //nop
  }

  protected String getTitle() {
    Window w = getWindow();
    if (w instanceof Frame) {
      return ((Frame) w).getTitle();
    }
    else if (w instanceof Dialog) {
      return ((Dialog) w).getTitle();
    }
    return null;
  }

  protected String getTitle(String text, FontMetrics fm, int availTextWidth) {
    return SwingUtilities2.clipStringIfNecessary(this, fm, text, availTextWidth);
  }

  protected void updateActionStyles() {
    if (iconifyAction != null) {
      iconifyAction.putValue(Action.SMALL_ICON, iconifyIcon);
      iconifyAction.putValue(Action.SHORT_DESCRIPTION, iconifyButtonToolTip);
      if (iconifyButton.getAction() == null || iconifyButton.getAction() == iconifyAction) {
        iconifyButton.setAction(null);
        iconifyButton.setAction(iconifyAction);
      }
    }
    if (minimizeAction != null) {
      minimizeAction.putValue(Action.SMALL_ICON, minimizeIcon);
      minimizeAction.putValue(Action.SHORT_DESCRIPTION, minimizeButtonToolTip);
      if (toggleButton.getAction() == null || toggleButton.getAction() == minimizeAction) {
        toggleButton.setAction(null);
        toggleButton.setAction(minimizeAction);
      }
    }
    if (maximizeAction != null) {
      maximizeAction.putValue(Action.SMALL_ICON, maximizeIcon);
      maximizeAction.putValue(Action.SHORT_DESCRIPTION, maximizeButtonToolTip);
      if (toggleButton.getAction() == null || toggleButton.getAction() == maximizeAction) {
        toggleButton.setAction(null);
        toggleButton.setAction(maximizeAction);
      }
    }
    if (closeAction != null) {
      closeAction.putValue(Action.SMALL_ICON, closeIcon);
      closeAction.putValue(Action.SHORT_DESCRIPTION, closeButtonToolTip);
      if (closeButton.getAction() == null || closeButton.getAction() == closeAction) {
        closeButton.setAction(null);
        closeButton.setAction(closeAction);
      }
    }
  }

  protected void updateButtonStates() {
    if (isFrameIconified()) {
      //nop
    }
    else if (isFrameMaximized()) {
      toggleButton.setAction(minimizeAction);
      // Add dynamic icons
      if (minimizeIconRollOver != null) {
        toggleButton.setRolloverEnabled(true);
        toggleButton.setIcon(minimizeIcon);
        toggleButton.setRolloverIcon(minimizeIconRollOver);
      }
    }
    else {
      toggleButton.setAction(maximizeAction);
      // Add dynamic icons
      if (maximizeIconRollOver != null) {
        toggleButton.setRolloverEnabled(true);
        toggleButton.setIcon(maximizeIcon);
        toggleButton.setRolloverIcon(maximizeIconRollOver);
      }
    }
    minimizeAction.setEnabled(isFrameMaximizable());
    maximizeAction.setEnabled(isFrameMaximizable());
    iconifyAction.setEnabled(isFrameIconifiable());
    closeAction.setEnabled(isFrameClosable());

    // Add dynamic icons
    if (iconifyIconRollOver != null) {
      iconifyButton.setRolloverEnabled(true);
      iconifyButton.setIcon(iconifyIcon);
      iconifyButton.setRolloverIcon(iconifyIconRollOver);
    }
    // Add dynamic icons
    if (closeIconRollOver != null) {
      closeButton.setRolloverEnabled(true);
      closeButton.setIcon(closeIcon);
      closeButton.setRolloverIcon(closeIconRollOver);
    }
  }

  protected WindowStateListener createWindowStateListener() {
    return new WindowStateHandler();
  }

  protected WindowListener createWindowListener() {
    return new WindowHandler();
  }

  protected LayoutManager createLayout() {
    return new TitlePaneLayout();
  }

  private class TitlePaneLayout implements LayoutManager {
    public void addLayoutComponent(String name, Component c) {
    }

    public void removeLayoutComponent(Component c) {
    }

    public Dimension preferredLayoutSize(Container c) {
      return minimumLayoutSize(c);
    }

    public Dimension minimumLayoutSize(Container c) {
      // Calculate width.
      int width = 22;

      if (isFrameClosable()) {
        width += 19;
      }
      if (isFrameMaximizable()) {
        width += 19;
      }
      if (isFrameIconifiable()) {
        width += 19;
      }

      FontMetrics fm = rootPane.getFontMetrics(rootPane.getFont());
      String frameTitle = getTitle();
      int title_w = frameTitle != null ? SwingUtilities2.stringWidth(rootPane, fm, frameTitle) : 0;
      int title_length = frameTitle != null ? frameTitle.length() : 0;

      // Leave room for three characters in the title.
      if (title_length > 3) {
        int subtitle_w = SwingUtilities2.stringWidth(rootPane, fm, frameTitle.substring(0, 3) + "...");
        width += (title_w < subtitle_w) ? title_w : subtitle_w;
      }
      else {
        width += title_w;
      }

      // Calculate height.
      Icon icon = UIManager.getIcon("Window.icon");
      int fontHeight = fm.getHeight();
      fontHeight += 2;
      int iconHeight = 0;
      if (icon != null) {
        // SystemMenuBar forces the icon to be 16x16 or less.
        iconHeight = Math.min(icon.getIconHeight(), 16);
      }
      iconHeight += 2;

      int height = Math.max(fontHeight, iconHeight);

      Dimension dim = new Dimension(width, height);

      // Take into account the border insets if any.
      if (getBorder() != null) {
        Insets insets = getBorder().getBorderInsets(c);
        dim.height += insets.top + insets.bottom;
        dim.width += insets.left + insets.right;
      }
      return dim;
    }

    public void layoutContainer(Container c) {
      boolean leftToRight = true;

      int w = getWidth();
      int h = getHeight();
      int x;

      int buttonHeight = closeButton.getIcon().getIconHeight();

      Icon icon = UIManager.getIcon("Window.icon");
      int iconHeight = 0;
      if (icon != null) {
        iconHeight = icon.getIconHeight();
      }
      x = (leftToRight) ? 2 : w - 16 - 2;
      menuBar.setBounds(x, (h - iconHeight) / 2, 16, 16);

      x = (leftToRight) ? w - 16 - 2 : 2;

      if (isFrameClosable()) {
        closeButton.setBounds(x, (h - buttonHeight) / 2, 16, 14);
        x += (leftToRight) ? -(16 + 2) : 16 + 2;
      }

      if (isFrameMaximizable()) {
        toggleButton.setBounds(x, (h - buttonHeight) / 2, 16, 14);
        x += (leftToRight) ? -(16 + 2) : 16 + 2;
      }

      if (isFrameIconifiable()) {
        iconifyButton.setBounds(x, (h - buttonHeight) / 2, 16, 14);
      }
    }
  }

  /**
   * This class should be treated as a &quot;protected&quot; inner class.
   * Instantiate it only within subclasses of <Foo>.
   */
  public class WindowStateHandler implements WindowStateListener {
    @Override
    public void windowStateChanged(WindowEvent e) {
      setState(e.getNewState());
    }
  }

  public class WindowHandler extends WindowAdapter {
    @Override
    public void windowActivated(WindowEvent ev) {
      setActive(true);
    }

    @Override
    public void windowDeactivated(WindowEvent ev) {
      setActive(false);
    }
  }

  /**
   * This class should be treated as a &quot;protected&quot; inner class.
   * Instantiate it only within subclasses of <Foo>.
   */
  public class CloseAction extends AbstractAction {
    public CloseAction() {
      super();
    }

    public void actionPerformed(ActionEvent e) {
      doClose();
    }
  } // end CloseAction

  /**
   * This class should be treated as a &quot;protected&quot; inner class.
   * Instantiate it only within subclasses of <Foo>.
   */
  public class MaximizeAction extends AbstractAction {
    public MaximizeAction() {
      super();
    }

    public void actionPerformed(ActionEvent evt) {
      doMaximize();
    }
  }

  /**
   * This class should be treated as a &quot;protected&quot; inner class.
   * Instantiate it only within subclasses of <Foo>.
   */
  public class MinimizeAction extends AbstractAction {
    public MinimizeAction() {
      super();
    }

    public void actionPerformed(ActionEvent evt) {
      doMinimize();
    }
  }

  /**
   * This class should be treated as a &quot;protected&quot; inner class.
   * Instantiate it only within subclasses of <Foo>.
   */
  public class IconifyAction extends AbstractAction {
    public IconifyAction() {
      super();
    }

    public void actionPerformed(ActionEvent e) {
      doIconify();
    }
  } // end IconifyAction

  /*
   * Handles showing and hiding the system menu.
   */
  private class ShowSystemMenuAction extends AbstractAction {
    private boolean show; // whether to show the menu

    public ShowSystemMenuAction(boolean show) {
      this.show = show;
    }

    public void actionPerformed(ActionEvent e) {
      if (show) {
        windowMenu.doClick();
      }
      else {
        windowMenu.setVisible(false);
      }
    }
  }

  /**
   * This class should be treated as a &quot;protected&quot; inner class.
   * Instantiate it only within subclasses of <Foo>.
   */
  public class SystemMenuBar extends JMenuBar {
    public boolean isFocusTraversable() {
      return false;
    }

    public void requestFocus() {
    }

    public void paint(Graphics g) {
      Icon icon = UIManager.getIcon("Window.icon");
      if (icon != null) {
        // Resize to 16x16 if necessary.
        if (icon instanceof ImageIcon && (icon.getIconWidth() > 16 || icon.getIconHeight() > 16)) {
          Image img = ((ImageIcon) icon).getImage();
          ((ImageIcon) icon).setImage(img.getScaledInstance(16, 16, Image.SCALE_SMOOTH));
        }
        icon.paintIcon(this, g, 0, 0);
      }
    }

    public boolean isOpaque() {
      return true;
    }
  } // end SystemMenuBar

  private class NoFocusButton extends JButton {
    private String uiKey;

    public NoFocusButton(String uiKey, String opacityKey) {
      setFocusPainted(false);
      setMargin(new Insets(0, 0, 0, 0));
      this.uiKey = uiKey;

      Object opacity = UIManager.get(opacityKey);
      if (opacity instanceof Boolean) {
        setOpaque(((Boolean) opacity).booleanValue());
      }
    }

    public boolean isFocusTraversable() {
      return false;
    }

    public void requestFocus() {
    };

    public AccessibleContext getAccessibleContext() {
      AccessibleContext ac = super.getAccessibleContext();
      if (uiKey != null) {
        ac.setAccessibleName(UIManager.getString(uiKey));
        uiKey = null;
      }
      return ac;
    }
  }; // end NoFocusButton
}
