/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.jaxws.service.internal;

import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.Future;

import javax.jws.WebMethod;
import javax.jws.WebResult;
import javax.jws.WebService;
import javax.xml.ws.Response;

import org.eclipse.scout.commons.BeanUtility;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.beans.FastBeanInfo;
import org.eclipse.scout.commons.beans.FastPropertyDescriptor;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;

/**
 * This class collects {@link WebMethod}s on a particular type and all its super types, where a type is expected to be a
 * {@link WebService} end point.
 * 
 * @since 4.1 (back-ported)
 */
public class JaxWsWebMethods {

  private static final IScoutLogger LOG = ScoutLogManager.getLogger(JaxWsWebMethods.class);

  private final Class<?> m_portTypeClass;
  private final Set<Method> m_webMethods = new HashSet<Method>();
  private final Map<Method, JaxWsAsyncWebMethod> m_syncToWebMethods;

  public JaxWsWebMethods(Class<?> portTypeClass) {
    m_portTypeClass = portTypeClass;
    if (!m_portTypeClass.isAnnotationPresent(WebService.class)) {
      LOG.warn("given portTypeClass [{}] does not describe a web service", m_portTypeClass);
    }

    // port methods cache
    Map<String, Method> syncMethodsByName = new HashMap<String, Method>();
    Map<String, Method> asyncMethodsByName = new HashMap<String, Method>();
    for (Method m : m_portTypeClass.getMethods()) {
      WebMethod webMethod = m.getAnnotation(WebMethod.class);
      if (webMethod == null) {
        continue;
      }
      final String webMethodName;
      if (StringUtility.hasText(webMethod.operationName())) {
        webMethodName = webMethod.operationName();
      }
      else {
        webMethodName = m.getName();
      }

      m_webMethods.add(m);
      if (Response.class.isAssignableFrom(m.getReturnType())) {
        // async method without callback
        asyncMethodsByName.put(webMethodName, m);
      }
      else if (Future.class.isAssignableFrom(m.getReturnType())) {
        // async method with callback
        // ignore for mapping
      }
      else {
        // sync operation
        syncMethodsByName.put(webMethodName, m);
      }
    }

    m_syncToWebMethods = new HashMap<Method, JaxWsAsyncWebMethod>();
    for (Entry<String, Method> syncMethodEntry : syncMethodsByName.entrySet()) {
      Method asyncMethod = asyncMethodsByName.get(syncMethodEntry.getKey());
      JaxWsAsyncWebMethod webMethod = createJaxWsWebMethod(syncMethodEntry.getValue(), asyncMethod);
      if (webMethod == null) {
        continue;
      }
      m_syncToWebMethods.put(syncMethodEntry.getValue(), webMethod);
    }
  }

  private JaxWsAsyncWebMethod createJaxWsWebMethod(Method syncWebMethod, Method asyncMethod) {
    if (asyncMethod == null) {
      return null;
    }

    Type genericReturnType = asyncMethod.getGenericReturnType();
    if (!(genericReturnType instanceof ParameterizedType)) {
      return null;
    }

    Type asyncMethodReturnParameterType = ((ParameterizedType) genericReturnType).getActualTypeArguments()[0];
    if (!(asyncMethodReturnParameterType instanceof Class)) {
      return null;
    }

    if (asyncMethodReturnParameterType == syncWebMethod.getReturnType()) {
      return new JaxWsAsyncWebMethod(syncWebMethod, asyncMethod);
    }

    WebResult webResult = syncWebMethod.getAnnotation(WebResult.class);
    if (webResult == null || "".equals(webResult.name())) {
      return null;
    }

    FastBeanInfo beanInfo = BeanUtility.getFastBeanInfo((Class) asyncMethodReturnParameterType, Object.class);
    FastPropertyDescriptor propDesc = beanInfo.getPropertyDescriptor(webResult.name());
    if (propDesc == null || propDesc.getReadMethod() == null) {
      return null;
    }

    return new JaxWsAsyncWebMethod(syncWebMethod, asyncMethod, propDesc.getReadMethod());
  }

  public Class<?> getPortTypeClass() {
    return m_portTypeClass;
  }

  public boolean isWebMethod(Method m) {
    return m_webMethods.contains(m);
  }

  public JaxWsAsyncWebMethod getAsyncMethod(Method m) {
    return m_syncToWebMethods.get(m);
  }

  public static class JaxWsAsyncWebMethod {

    private final Method m_syncMethod;
    private final Method m_asyncMethod;
    private final Method m_asyncResultExtractorMethod;
    private boolean m_enabled;

    public JaxWsAsyncWebMethod(Method syncWebMethod, Method asyncWebMethod) {
      this(syncWebMethod, asyncWebMethod, null);
    }

    public JaxWsAsyncWebMethod(Method syncWebMethod, Method asyncWebMethod, Method asyncResultExtrationMethod) {
      m_syncMethod = syncWebMethod;
      m_asyncMethod = asyncWebMethod;
      m_asyncResultExtractorMethod = asyncResultExtrationMethod;
      m_enabled = true;
    }

    public Method getSyncMethod() {
      return m_syncMethod;
    }

    public Method getAsyncMethod() {
      return m_asyncMethod;
    }

    public Method getAsyncResultExtractorMethod() {
      return m_asyncResultExtractorMethod;
    }

    public void disable() {
      m_enabled = false;
    }

    public boolean isEnabled() {
      return m_enabled;
    }
  }
}