/*******************************************************************************
 * Copyright (c) 2013,2014 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Ivan Motsch (BSI Business Systems Integration AG) - initial API and implementation
 *     Stephan Leicht Vogt (BSI Business Systems Integration AG) - adaption to JaxWS
 ******************************************************************************/
package org.eclipse.scout.jaxws.service.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.Set;
import java.util.TreeSet;

import javax.xml.ws.Service;

import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.annotations.ConfigProperty;
import org.eclipse.scout.commons.annotations.Order;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.jaxws.service.IJaxWsConnectionHandler;
import org.eclipse.scout.jaxws.service.IJaxWsConnectionProviderService;
import org.eclipse.scout.jaxws.service.IWebServiceClient;
import org.eclipse.scout.rt.server.admin.diagnostic.DiagnosticFactory;
import org.eclipse.scout.service.AbstractService;
import org.osgi.framework.ServiceRegistration;

public class JaxWsConnectionProviderService extends AbstractService implements IJaxWsConnectionProviderService {
  private static final long serialVersionUID = 1L;
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(JaxWsConnectionProvider.class);//all connection provider classes use log of JaxWsConnectionProvider

  private static final boolean DIAGNOSTICS_EXIST;

  private final Set<JaxWsConnectionProvider> m_webServciceConnectionProviders = new TreeSet<JaxWsConnectionProvider>();
  private final Object m_webServciceConnectionProvidersLock = new Object();

  private long m_connectionMaxAge;
  private long m_statementBusyTimeout;

  private JaxWsConnectionProviderDiagnostic m_diagnostic;

  static {
    boolean diagnosticsExists = false;
    try {
      diagnosticsExists = StringUtility.hasText(DiagnosticFactory.class.getName());
    }
    catch (Exception e) {
      LOG.debug("Class DiagnosticFactory couldn't be loaded.", e);
    }
    DIAGNOSTICS_EXIST = diagnosticsExists;
  }

  @Override
  public void initializeService(ServiceRegistration registration) {
    initConfig();
    super.initializeService(registration);

    if (DIAGNOSTICS_EXIST) {
      m_diagnostic = new JaxWsConnectionProviderDiagnostic(this);
      DiagnosticFactory.addDiagnosticStatusProvider(m_diagnostic);
      DiagnosticFactory.addActionToDiagnosticStatusProvider(JaxWsConnectionProviderDiagnostic.ACTION_CANCEL_CONNECTION, m_diagnostic);
    }
  }

  @Override
  public <S extends Service, P> IJaxWsConnectionHandler<S, P> createConnectionProvider(IWebServiceClient<S> webServiceClient, Class<? extends P> portTypeClazz) {
    JaxWsConnectionProvider<S, P> connectionProvider = new JaxWsConnectionProvider<S, P>(webServiceClient, portTypeClazz, getConnectionMaxAge());
    connectionProvider.init();
    connectionProvider.setStatementBusyTimeout(m_statementBusyTimeout);
    connectionProvider.startHousekeeping();
    synchronized (m_webServciceConnectionProvidersLock) {
      m_webServciceConnectionProviders.add(connectionProvider);
    }
    return connectionProvider;
  }

  public Collection<JaxWsConnectionProvider> getAllConnectionProviders() {
    synchronized (m_webServciceConnectionProvidersLock) {
      return new ArrayList<JaxWsConnectionProvider>(m_webServciceConnectionProviders);
    }
  }

  protected void initConfig() {
//    setMinPoolSize(getConfiguredMaxPoolSizePerSession());TODO does it need a min-poolsize? Like at least one connection per session/user...
    setConnectionMaxAge(getConfiguredConnectionMaxAge());
    setStatementBusyTimeout(getConfiguredStatementBusyTimeout());
  }

  @ConfigProperty(ConfigProperty.LONG)
  @Order(50)
  protected Long getConfiguredConnectionMaxAge() {
    return DEFAULT_CONNECTION_MAX_AGE;
  }

  @ConfigProperty(ConfigProperty.LONG)
  @Order(70)
  protected Long getConfiguredStatementBusyTimeout() {
    return DEFAULT_STATEMENT_BUSY_TIMEOUT;
  }

  public long getConnectionMaxAge() {
    return m_connectionMaxAge;
  }

  /**
   * @param connectionMaxAge
   *          max age in seconds
   */
  public void setConnectionMaxAge(long connectionMaxAge) {
    m_connectionMaxAge = connectionMaxAge * MILLIS_PER_SECOND;
  }

  public long getStatementBusyTimeout() {
    return m_statementBusyTimeout;
  }

  /**
   * @param statementBusyTimeout
   *          busy timeout in seconds
   */
  public void setStatementBusyTimeout(long statementBusyTimeout) {
    m_statementBusyTimeout = statementBusyTimeout * MILLIS_PER_SECOND;
  }

  @Override
  public void disposeServices() {
    if (DIAGNOSTICS_EXIST) {
      DiagnosticFactory.removeDiagnosticStatusProvider(m_diagnostic);
    }

    super.disposeServices();
    synchronized (m_webServciceConnectionProvidersLock) {
      for (Iterator it = m_webServciceConnectionProviders.iterator(); it.hasNext();) {
        IJaxWsConnectionHandler provider = (IJaxWsConnectionHandler) it.next();
        provider.stop();
        it.remove();
      }
    }
  }
}
