/*******************************************************************************
 * Copyright (c) 2013,2014 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.jaxws.service.internal;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.rt.server.admin.diagnostic.DiagnosticFactory;
import org.eclipse.scout.rt.server.admin.diagnostic.IDiagnostic;

public class JaxWsConnectionProviderDiagnostic implements IDiagnostic {
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(JaxWsConnectionProvider.class);//all connection provider classes use log of JaxWsConnectionProvider

  public static final String ACTION_CANCEL_CONNECTION = "cancelConnection";

  private final JaxWsConnectionProviderService m_jaxWsConnectionProviderService;
  private final Map<String, ButtonActionBean> m_actionMap;

  /**
   * @param jaxWsConnectionProviderService
   */
  public JaxWsConnectionProviderDiagnostic(JaxWsConnectionProviderService jaxWsConnectionProviderService) {
    m_jaxWsConnectionProviderService = jaxWsConnectionProviderService;
    m_actionMap = new HashMap<String, ButtonActionBean>();
  }

  /*
   * Diagnostics
   */
  @Override
  public void addDiagnosticItemToList(List<List<String>> result) {
    try {
      DiagnosticFactory.addDiagnosticItemToList(result, "JAX-WS", "", DiagnosticFactory.STATUS_TITLE);
      for (JaxWsConnectionProvider provider : m_jaxWsConnectionProviderService.getAllConnectionProviders()) {
        String newStateSnapshot = provider.getStats().newStateSnapshot();
        newStateSnapshot = StringUtility.replaceNewLines(newStateSnapshot, "<br>");
        DiagnosticFactory.addDiagnosticItemToList(result, "JAX-WS Connection Provider Status", newStateSnapshot, DiagnosticFactory.STATUS_INFO);
      }
    }
    catch (Exception e) {
      LOG.warn("Exception in collecting diagnostic informations", e);
      DiagnosticFactory.addDiagnosticItemToList(result, "Read JAX-WS Connection provider Stats", "", DiagnosticFactory.STATUS_FAILED);
    }
  }

  @Override
  public String[] getPossibleActions() {
    return new String[]{ACTION_CANCEL_CONNECTION};
  }

  @Override
  public void addSubmitButtonsHTML(List<List<String>> result) {
    m_actionMap.clear();
    try {
      for (JaxWsConnectionProvider provider : m_jaxWsConnectionProviderService.getAllConnectionProviders()) {
        List<ButtonActionBean> providerButtons = provider.getStats().createButtonActions();
        for (ButtonActionBean buttonBean : providerButtons) {
          String buttonBeanHash = String.valueOf(buttonBean.hashCode());
          m_actionMap.put(buttonBeanHash, buttonBean);
          DiagnosticFactory.addDiagnosticItemToList(result, "Cancel JAX-WS Connection " + buttonBean.getName() + " ", buttonBean.getDescription(), "<input type='checkbox' name='" + ACTION_CANCEL_CONNECTION + "' value='" + buttonBeanHash + "'/>");
        }
      }
    }
    catch (Exception e) {
      LOG.warn("Exception in collecting diagnostic informations", e);
      DiagnosticFactory.addDiagnosticItemToList(result, "Create JAX-WS Connection provider Button action", "", DiagnosticFactory.STATUS_FAILED);
    }
  }

  @Override
  public void call(String action, Object[] value) {
    if (ACTION_CANCEL_CONNECTION.equals(action) && value != null && value.length > 0) {
      ButtonActionBean buttonActionBean = m_actionMap.get(value[0]);
      if (buttonActionBean != null) {
        buttonActionBean.getCallback().execute();
      }
    }
  }
}
