/*******************************************************************************
 * Copyright (c) 2014 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Ivan Motsch (BSI Business Systems Integration AG) - initial API and implementation
 *     Stephan Leicht Vogt (BSI Business Systems Integration AG) - adaption to JaxWS
 ******************************************************************************/
package org.eclipse.scout.jaxws.service.internal;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Locale;
import java.util.concurrent.Future;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import javax.xml.ws.Response;

import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.jaxws.service.internal.JaxWsWebMethods.JaxWsAsyncWebMethod;

import com.sun.xml.ws.client.ResponseContext;
import com.sun.xml.ws.client.ResponseContextReceiver;
import com.sun.xml.ws.client.sei.SEIStub;

public class JaxWsConnectionHandler<P> extends AbstractInvocationHandler<P> {
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(JaxWsConnectionProvider.class);//all connection provider classes use log of JaxWsConnectionProvider

  private static final AtomicInteger CONNECTION_COUNTER = new AtomicInteger(1); // counter to create a unique id for each connection
  static final AtomicBoolean THROWABLE_ENRICHABLE = new AtomicBoolean(true); // if enriching the Throwable by reflection fails, never try again due to performance reasons

  private final AtomicReference<Response> m_responseReference;
  private final JaxWsWebMethods m_webMethods;
  private final Class m_portTypeClass;

  private final long m_created;
  private final int m_id; // unique id for connections, used to detect if statements fail always on the same connection
  private long m_leaseBegin;
  private int m_leaseCount;
  private Thread m_leaseThread;
  private long m_invocationStartTime;

  public JaxWsConnectionHandler(P impl, Class portTypeClass, JaxWsWebMethods webMethods) {
    super(impl, impl.getClass().getInterfaces()); // the JavaProxy should support all declared interfaces.
    m_portTypeClass = portTypeClass;
    m_webMethods = webMethods;
    m_created = System.currentTimeMillis();
    m_id = CONNECTION_COUNTER.getAndIncrement();
    m_responseReference = new AtomicReference<Response>();
  }

  public long getCreated() {
    return m_created;
  }

  public long getLeaseBegin() {
    return m_leaseBegin;
  }

  public long getInvocationStartTime() {
    return m_invocationStartTime;
  }

  public void beginLease() {
    m_leaseCount++;
    m_leaseBegin = System.currentTimeMillis();
    m_leaseThread = Thread.currentThread();
  }

  public void endLease() {
    m_leaseBegin = 0L;
    m_leaseThread = null;
  }

  public int getId() {
    return m_id;
  }

  @Override
  public int hashCode() {
    return (int) m_created;
  }

  @Override
  public boolean equals(Object obj) {
    return obj == this;
  }

  public static String stripPackageName(final String classname) {
    int idx = classname.lastIndexOf('.');

    if (idx != -1) {
      return classname.substring(idx + 1, classname.length());
    }
    return classname;
  }

  private static final String HTML_TABSTOP = "&nbsp;&nbsp;&nbsp;&nbsp;";
  private static final String TABSTOP = "\t";
  private static final String HTML_NEWLINE = "<br>";
  private static final String NEWLINE = "\n";

  @Override
  public String toString() {
    return toString(false);
  }

  public String toString(boolean forHtml) {
    String tabstop = TABSTOP;
    String nl = NEWLINE;
    if (forHtml) {
      tabstop = HTML_TABSTOP;
      nl = HTML_NEWLINE;
    }
    SimpleDateFormat fmt = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss.SSSS", Locale.GERMAN);
    StringBuilder buf = new StringBuilder();
    Thread thread = m_leaseThread;
    if (thread != null) {
      StackTraceElement[] stackTrace = m_leaseThread.getStackTrace();
      StackTraceElement traceElement = getOriginatingStackTraceElement(stackTrace);
      if (traceElement != null) {
        buf.append("calling method=").append(stripPackageName(traceElement.getClassName())).append(".").append(traceElement.getMethodName()).append("(").append(traceElement.getLineNumber()).append(")");
        buf.append(nl).append(tabstop);
      }
    }
    buf.append("class=").append(getImpl().getClass().getName());
    buf.append(", id=").append(getId());
    buf.append(", created=").append(fmt.format(new Date(m_created)));
    buf.append(", usageCount=").append(m_leaseCount);
    buf.append(nl).append(tabstop);
    if (m_leaseBegin > 0) {
      buf.append("busySince=").append(fmt.format(new Date(m_leaseBegin)));
    }
    else {
      buf.append("age=").append((System.currentTimeMillis() - m_created) / 1000).append(" seconds");
    }
    if (thread != null) {
      buf.append(nl).append(tabstop);
      buf.append("leaseThreadName: ").append(thread.getName());
      if (LOG.isDebugEnabled()) {
        Exception stack = new Exception("Stack trace");
        stack.setStackTrace(thread.getStackTrace());
        StringWriter w = new StringWriter();
        stack.printStackTrace(new PrintWriter(w, true));
        buf.append(nl).append(tabstop);
        buf.append("leaseThreadStacktrace:").append(nl).append(w.toString().trim());
      }
    }
    return buf.toString();
  }

  public StackTraceElement getOriginatingStackTraceElement(StackTraceElement[] stackTrace) {
    StackTraceElement retval = null;
    if (stackTrace != null) {
      int traceIndex = 0;
      // find constructor
      while (traceIndex + 1 < stackTrace.length && !getClass().getName().equals(stackTrace[traceIndex].getClassName())) {
        traceIndex++;
      }
      // find origin
      while (traceIndex + 1 < stackTrace.length && getClass().getName().equals(stackTrace[traceIndex].getClassName())) {
        traceIndex++;
      }
      if (traceIndex + 1 < stackTrace.length) {
        retval = stackTrace[traceIndex + 1];
      }
    }
    return retval;
  }

  public boolean cancel() {
    Future<?> future = m_responseReference.get();
    if (future == null) {
      return false;
    }
    return future.cancel(true);
  }

  public boolean reset() {
    SEIStub seiStub = extractSEIStub();
    if (seiStub == null) {
      return false;
    }
    seiStub.resetRequestContext();
    return true;
  }

  public boolean close() {
    SEIStub seiStub = extractSEIStub();
    if (seiStub == null) {
      return false;
    }
    seiStub.close();
    return true;
  }

  /**
   * @return Extracts the {@link SEIStub} if {@link #getImpl()} is backed by one. Otherwise <code>null</code>.
   */
  private SEIStub extractSEIStub() {
    P impl = getImpl();
    if (!Proxy.isProxyClass(impl.getClass())) {
      return null;
    }
    InvocationHandler invocationHandler = Proxy.getInvocationHandler(impl);
    if (!(invocationHandler instanceof SEIStub)) {
      return null;
    }
    SEIStub seiStub = (SEIStub) invocationHandler;
    return seiStub;
  }

  @Override
  public Object invoke(final Object proxy, final Method method, final Object[] args) throws Throwable {
    try {
      try {
        return invokeAsyncMethod(proxy, method, args);
      }
      catch (P_AsyncInvocationResultExtractionFailedException p) {
        // invoke again - invocation type has been changed to sync
        return invokeAsyncMethod(proxy, method, args);
      }
    }
    catch (Exception e) {
      if (THROWABLE_ENRICHABLE.get()) {
        try {
          Field detailMessage = Throwable.class.getDeclaredField("detailMessage");
          detailMessage.setAccessible(true);
          detailMessage.set(e, (e.getMessage() == null ? "" : e.getMessage().trim()) + " [JaxWsConnectionHandler.id=" + getId() + "]");
        }
        catch (Exception e1) {
          THROWABLE_ENRICHABLE.set(false);
          LOG.debug("unable to enrich exception [id=" + getId() + "]", e1);
        }
      }
      throw e;
    }
  }

  private Object invokeAsyncMethod(final Object proxy, final Method method, final Object[] args) throws Throwable {
    final boolean startTimer = m_invocationStartTime == 0 && m_webMethods.isWebMethod(method);
    if (startTimer) {
      m_invocationStartTime = System.currentTimeMillis();
    }
    try {
      JaxWsAsyncWebMethod asyncWebMethod = m_webMethods.getAsyncMethod(method);
      if (asyncWebMethod != null && asyncWebMethod.isEnabled()) {
        final Object result;
        Response response = (Response) super.invoke(proxy, asyncWebMethod.getAsyncMethod(), args);
        try {
          m_responseReference.set(response);
          result = extractAsyncResult(response.get(), asyncWebMethod);

          // Make the response context available to the caller, e.g. to access the HTTP status code.
          ResponseContextReceiver responseContextReceiver = (ResponseContextReceiver) Proxy.getInvocationHandler(getImpl()); // the PortType-stub is a JavaProxy itself with no support for ResponseContextReceiver.
          responseContextReceiver.setResponseContext((ResponseContext) response.getContext());
        }
        finally {
          m_responseReference.set(null);
        }
        return result;
      }
      else {
        return super.invoke(proxy, method, args);
      }
    }
    finally {
      if (startTimer) {
        m_invocationStartTime = 0;
      }
    }
  }

  /**
   * Extracts the result of an async invocation. Depending on the operation, the result of the async method may be
   * wrapped in an additional response object. If the extraction raises any exceptions, all consecutive invocations of
   * the service operation are performed sync.
   */
  private Object extractAsyncResult(Object o, JaxWsAsyncWebMethod asyncWebMethod) {
    if (o == null) {
      return null;
    }
    Method asyncResultExtractor = asyncWebMethod.getAsyncResultExtractorMethod();
    if (asyncResultExtractor == null) {
      return o;
    }
    try {
      return asyncResultExtractor.invoke(o);
    }
    catch (Exception e) {
      LOG.warn("async result extraction raised exception. falling back to sync invocation", e);
      asyncWebMethod.disable();
      throw new P_AsyncInvocationResultExtractionFailedException();
    }
  }

  /**
   * Marker exception for falling back to synchronous invocation.
   */
  static class P_AsyncInvocationResultExtractionFailedException extends RuntimeException {
    private static final long serialVersionUID = 1L;
  }
}
