/*******************************************************************************
 * Copyright (c) 2013,2014 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Ivan Motsch (BSI Business Systems Integration AG) - initial API and implementation
 *     Stephan Leicht Vogt (BSI Business Systems Integration AG) - adaption to JaxWS
 ******************************************************************************/
package org.eclipse.scout.jaxws.service.internal;

import java.lang.reflect.Proxy;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.TimeUnit;

import javax.xml.ws.Service;
import javax.xml.ws.WebServiceFeature;

import org.eclipse.scout.commons.CompareUtility;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.jaxws.service.IJaxWsConnectionProviderService;
import org.eclipse.scout.jaxws.service.IWebServiceClient;

/**
 * System-wide connection pool for pooling connections. There is one pool for
 * every IWebServiceClient sub class type. If possible, every scout Session is provided
 * with always the same connection it had in the last request.
 * <p>
 * This class is thread-safe.
 */
public class InternalJaxWsConnectionProvider<S extends Service, P> extends AbstractNonBlockingPoolWithTimeout<JaxWsConnectionHandler<P>, WebServiceFeature[]> implements Comparable<InternalJaxWsConnectionProvider<S, P>> {
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(JaxWsConnectionProvider.class);//all connection provider classes use log of JaxWsConnectionProvider

  private static final int ONE_THOUSEND_MILLISECCONDS = 1000;
  private static final long TWO_SECCONDS = 2000L;
  private static final long FIVE_SECCONDS = 5000L;

  private final WebServicePool<S> m_webServicePool;
  private final Class<? extends P> m_portTypeClazz;
  private final JaxWsWebMethods m_webMethods;

  //
  private final transient AbstractJaxWsConnectionProviderStats m_stats;
  private transient volatile Thread m_managePoolWorker;
  //configuration
  private long m_checkInterval;
  private long m_statementBusyTimeout;

  public InternalJaxWsConnectionProvider(IWebServiceClient<S> webServiceClient, Class<? extends P> portTypeClazz, long connectionMaxAgeMillis) {
    super(connectionMaxAgeMillis, TimeUnit.MILLISECONDS);
    m_webServicePool = new WebServicePool<S>(webServiceClient);
    m_portTypeClazz = portTypeClazz;
    m_webMethods = new JaxWsWebMethods(m_portTypeClazz);
    m_stats = new AbstractJaxWsConnectionProviderStats() {
      private static final long serialVersionUID = 1L;

      @Override
      public String newStateSnapshot() {
        return InternalJaxWsConnectionProvider.this.createStateSnapshot(m_stats);
      }

      @Override
      public List<ButtonActionBean> createButtonActions() {
        return InternalJaxWsConnectionProvider.this.createButtonActions(m_stats);
      }
    };
  }

  public void init() {
    setCheckInterval(IJaxWsConnectionProviderService.DEFAULT_CHECK_INTERVAL * IJaxWsConnectionProviderService.MILLIS_PER_SECOND);
    setStatementBusyTimeout(IJaxWsConnectionProviderService.DEFAULT_STATEMENT_BUSY_TIMEOUT * IJaxWsConnectionProviderService.MILLIS_PER_SECOND);
  }

  @Override
  public int compareTo(InternalJaxWsConnectionProvider<S, P> o) {
    if (o == null) {
      return 1;
    }
    return CompareUtility.compareTo(m_portTypeClazz.getSimpleName(), o.m_portTypeClazz.getSimpleName());
  }

  protected IJaxWsConnectionProviderStats getStatsImpl() {
    return m_stats;
  }

  /**
   * @return busy timeout in milliseconds
   */
  protected long getStatementBusyTimeout() {
    return m_statementBusyTimeout;
  }

  /**
   * @param statementBusyTimeout
   *          busy timeout in milliseconds
   */
  public void setStatementBusyTimeout(long statementBusyTimeout) {
    m_statementBusyTimeout = statementBusyTimeout;
  }

  protected void setCheckInterval(long checkInterval) {
    m_checkInterval = checkInterval;
  }

  protected long getCheckInterval() {
    return m_checkInterval;
  }

  public void startHousekeeping() {
    m_managePoolWorker = new Thread(InternalJaxWsConnectionProvider.this.getClass().getSimpleName() + ".managePool") {
      @Override
      public void run() {
        while (m_managePoolWorker == this) {
          try {
            Thread.sleep(m_checkInterval);
          }
          catch (InterruptedException ie) {
            //nop
          }
          managePool(false);
        }
      }
    };
    m_managePoolWorker.setDaemon(true);
    m_managePoolWorker.start();
  }

  protected void stopHousekeeping() {
    m_managePoolWorker = null;
    managePool(true);
  }

  public void closeAllConnections() {
    managePool(true);
  }

  /**
   * @param webserviceClient
   * @return a web service connection
   */
  protected P leaseConnection(WebServiceFeature... webServiceFeatures) {
    JaxWsConnectionHandler<P> h = lease(webServiceFeatures);
    h.beginLease();
    m_stats.connectionLease();
    LOG.info("lease   {}", h);
    return h.getProxy();
  }

  /**
   * @return release a proxied connection
   */
  protected void releaseConnection(P proxy) {
    releaseConnection(proxy, false);
  }

  /**
   * @return cancel a open proxied connection
   */
  protected void cancelOpenConnection(P proxy) {
    releaseConnection(proxy, true);
  }

  private void releaseConnection(P proxy, boolean cancel) {
    @SuppressWarnings("unchecked")
    JaxWsConnectionHandler<P> h = (JaxWsConnectionHandler<P>) Proxy.getInvocationHandler(proxy);

    if (cancel) {
      h.cancel();
    }
    m_stats.connectionRelease();
    LOG.info("release {}", h);

    // reset connection
    boolean reset = false;
    if (!cancel) {
      try {
        reset = h.reset();
      }
      catch (Exception e) {
        LOG.info("cannot reset port", e);
      }
    }

    boolean recycleConnection = !cancel && reset;
    release(h, recycleConnection);
    h.endLease();
  }

  @Override
  protected JaxWsConnectionHandler<P> createElement(WebServiceFeature[] webServiceFeatures) {
    S webService = m_webServicePool.lease(null);
    try {
      P port = webService.getPort(m_portTypeClazz, webServiceFeatures);
      JaxWsConnectionHandler<P> handler = new JaxWsConnectionHandler<P>(port, m_portTypeClazz, m_webMethods);
      LOG.info("created jax ws connection {}", handler);
      m_stats.connectionOpen();
      return handler;
    }
    finally {
      m_webServicePool.release(webService, true);
    }
  }

  @Override
  protected void cleanup(JaxWsConnectionHandler<P> h, boolean sync) {
    enqueueCloseConnectionImpl(sync ? "cleanup (sync)" : "cleanup (async)", h, sync);
  }

  /**
   * Thread worker to manage pool
   * <p>
   * must be called inside lock {@link #m_poolLock}
   */
  @Override
  protected void managePool(boolean closeAll) {
    super.managePool(closeAll);

    if (!closeAll) {
      // close timed out statements
      for (Iterator<JaxWsConnectionHandler<P>> it = busyElementsIterator(); it.hasNext();) {
        JaxWsConnectionHandler<P> h = it.next();
        long invocationStartTime = h.getInvocationStartTime();
        if (invocationStartTime > 0 && System.currentTimeMillis() - invocationStartTime >= getStatementBusyTimeout()) {
          LOG.warn("Cancelling timed out statement " + h.toString());
          enqueueCancelStatement("Cancelling timed out statement (async)", h, false);
        }
      }
    }
  }

  protected void enqueueCloseConnectionImpl(final String msg, final JaxWsConnectionHandler h, boolean sync) {
    m_stats.connectionClosePending();
    Thread job = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          LOG.info(msg);
          h.close();
        }
        catch (Exception e) {
          LOG.info("enqueueCloseStatement closing connection exception", e);
        }
        finally {
          m_stats.connectionClose();
        }
      }
    }, msg);
    job.start();
    try {
      if (sync) {
        job.join();
      }
      else {
        //optimistically wait up to 5 seconds
        job.join(FIVE_SECCONDS);
      }
    }
    catch (Exception e) {
      LOG.info("enqueueCancelStatement Threading exception", e);
    }
  }

  protected void enqueueCancelStatement(final String msg, final JaxWsConnectionHandler<P> h, boolean sync) {
    Thread job = new Thread(new Runnable() {
      @Override
      public void run() {
        try {
          LOG.info(msg);
          h.cancel();
        }
        catch (Exception e) {
          LOG.info("enqueueCancelStatement closing connection exception", e);
        }
      }
    }, msg);
    job.start();
    try {
      if (sync) {
        job.join();
      }
      else {
        //optimistically wait up to 2 seconds
        job.join(TWO_SECCONDS);
      }
    }
    catch (Exception e) {
      LOG.info("enqueueCancelStatement Threading exception", e);
    }
  }

  @Override
  protected String createElementStateSnapshot(JaxWsConnectionHandler<P> h) {
    return h.toString(true);
  }

  protected String createStateSnapshot(IJaxWsConnectionProviderStats stats) {
    StringBuilder sb = new StringBuilder();
    sb.append(JaxWsConnectionProvider.class.getName() + " for " + m_portTypeClazz.getSimpleName() + " stats<br>");
    sb.append("<br>");

    sb.append("&nbsp;statementBusyTimeout: " + getStatementBusyTimeout() + "<br>");
    sb.append("<br>");

    sb.append("<b>Connection statistics:</b><br>");
    sb.append("&nbsp;connectionCount: " + stats.getConnectionCount() + "<br>");
    sb.append("&nbsp;connectionClosePendingCount: " + stats.getConnectionClosePendingCount() + "<br>");
    sb.append("&nbsp;connectionLeaseCount: " + stats.getConnectionLeaseCount() + "<br>");
    sb.append("&nbsp;connectionReleaseCount: " + stats.getConnectionReleaseCount() + "<br>");

    sb.append("<br><b>Web Service Instances:</b><br>");
    m_webServicePool.createStateSnapshot(sb);

    sb.append("<br><b>Service End Points:</b><br>");
    createStateSnapshot(sb);

    return sb.toString();
  }

  protected List<ButtonActionBean> createButtonActions(IJaxWsConnectionProviderStats stats) {
    List<ButtonActionBean> actionList = new ArrayList<ButtonActionBean>();
    for (Iterator<JaxWsConnectionHandler<P>> it = busyElementsIterator(); it.hasNext();) {
      JaxWsConnectionHandler<P> h = it.next();
      long invocationStartTime = h.getInvocationStartTime();
      if (invocationStartTime == 0) {
        continue;
      }
      long busyTime = System.currentTimeMillis() - invocationStartTime;
      ButtonActionBean action = new ButtonActionBean();
      action.setName(new StringBuilder(m_portTypeClazz.getSimpleName()).append(" busy for ").append(busyTime / ONE_THOUSEND_MILLISECCONDS).append(" seconds").toString());
      action.setDescription(h.toString(true));
      action.setCallback(new P_ActionCancelCallback(h, m_portTypeClazz.getSimpleName()));
      actionList.add(action);
    }
    return actionList;
  }

  private static class P_ActionCancelCallback implements IButtonActionCallback {
    private final JaxWsConnectionHandler<?> m_conHandler;
    private final String m_connectionName;

    public P_ActionCancelCallback(JaxWsConnectionHandler<?> handler, String connectionName) {
      m_conHandler = handler;
      m_connectionName = connectionName;
    }

    @Override
    public void execute() {
      LOG.info("Canceling JAX-WS Connection {0}", m_connectionName);
      m_conHandler.cancel();
    }
  }

  private static class WebServicePool<S extends Service> extends AbstractNonBlockingPoolWithTimeout<S, Void> {

    private final IWebServiceClient<S> m_webServiceClient;

    public WebServicePool(IWebServiceClient<S> webServiceClient) {
      super(12, TimeUnit.HOURS);
      m_webServiceClient = webServiceClient;
    }

    @Override
    protected S createElement(Void o) {
      LOG.debug("creating new web service client using '{}'", m_webServiceClient.getClass().getName());
      return m_webServiceClient.getWebService();
    }
  }
}
