package org.eclipse.scout.jaxws.service.internal;

import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Iterator;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;

/**
 * Non-blocking, unlimited pool. Elements are removed after a given timeout.
 * 
 * @param <T>
 *          type of pooled elements.
 * @param <P>
 *          type of additional parameters used for creating a new pool element.
 */
public abstract class AbstractNonBlockingPoolWithTimeout<T, P> {

  private static final IScoutLogger LOG = ScoutLogManager.getLogger(AbstractNonBlockingPoolWithTimeout.class);

  private final long m_maxAgeMillis;
  private final AtomicLong m_poolSize = new AtomicLong();

  private final Map<T, State> m_idleElements = new ConcurrentHashMap<T, State>();
  private final Map<T, State> m_busyElements = new ConcurrentHashMap<T, State>();

  public AbstractNonBlockingPoolWithTimeout(long maxAge, TimeUnit timeUnit) {
    m_maxAgeMillis = timeUnit.toMillis(maxAge);
  }

  public T lease(P createParam) {
    for (T candidate : m_idleElements.keySet()) {
      final State state = m_idleElements.remove(candidate);
      if (state == null) {
        // Another thread took the element concurrently.
        continue;
      }

      if (state.isExpired()) {
        // The element reached its max age.
        cleanupInternal(candidate, false);
        continue;
      }

      state.incUsageCount();
      m_busyElements.put(candidate, state);
      return candidate;
    }

    // no element available - create new element, increasing the pool size
    final T result = createElement(createParam);
    m_poolSize.incrementAndGet();
    m_busyElements.put(result, new State(System.currentTimeMillis() + m_maxAgeMillis));
    return result;
  }

  public void release(T o, boolean recycle) {
    final State state = m_busyElements.remove(o);
    if (state == null) {
      // element was not managed by this pool. Hence do not invoke cleanupInternal
      cleanup(o, false);
      return;
    }
    if (state.isExpired() || !recycle) {
      cleanupInternal(o, false);
    }
    else {
      m_idleElements.put(o, state);
    }
  }

  protected abstract T createElement(P createParam);

  private void cleanupInternal(T o, boolean sync) {
    try {
      cleanup(o, sync);
    }
    finally {
      m_poolSize.decrementAndGet();
    }
  }

  /**
   * This method is called when an element is removed from the pool because it is expired. Override it to
   * free up resources held by the element, if any.
   */
  protected void cleanup(T o, boolean sync) {
  }

  protected void managePool(boolean closeAll) {
    try {
      for (T idleElement : m_idleElements.keySet()) {
        final State state = m_idleElements.remove(idleElement);
        if (state == null) {
          // Another thread took the element concurrently.
          continue;
        }
        if (closeAll || state.isExpired()) {
          cleanupInternal(idleElement, true);
          continue;
        }

        // Element is still valid. Put it back.
        m_idleElements.put(idleElement, state);
      }
    }
    catch (Exception e) {
      LOG.warn("Exception while managing pool", e);
    }
  }

  protected Iterator<T> busyElementsIterator() {
    final Iterator<T> it = m_busyElements.keySet().iterator();
    return new Iterator<T>() {
      @Override
      public boolean hasNext() {
        return it.hasNext();
      }

      @Override
      public T next() {
        return it.next();
      }

      @Override
      public void remove() {
        throw new UnsupportedOperationException("remove is not supported");
      }
    };
  }

  public void createStateSnapshot(StringBuilder sb) {
    final int busySize = m_busyElements.size();
    final int idleSize = m_idleElements.size();
    final long poolSize = m_poolSize.get();
    sb.append(String.format("%s - pooling %d elements (%d busy, %d idle, %d being created, recycled or destroyed), maxAge %dms<br>",
        getClass().getSimpleName(), poolSize, busySize, idleSize, poolSize - busySize - idleSize, m_maxAgeMillis));

    for (Entry<T, State> e : m_busyElements.entrySet()) {
      sb.append("&nbsp;Busy: ").append(e.getValue()).append(", ").append(createElementStateSnapshot(e.getKey())).append("<br>");
    }

    for (Entry<T, State> e : m_idleElements.entrySet()) {
      sb.append("&nbsp;Idle: ").append(e.getValue()).append(", ").append(createElementStateSnapshot(e.getKey())).append("<br>");
    }
  }

  protected String createElementStateSnapshot(T o) {
    if (o == null) {
      return "[null]";
    }
    return o.toString();
  }

  private static class State {

    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss.SSSS", Locale.US);

    private final long m_expiry;
    private long m_usageCount;

    private State(long expiry) {
      m_expiry = expiry;
      m_usageCount = 1;
    }

    public boolean isExpired() {
      return m_expiry < System.currentTimeMillis();
    }

    public long getUsageCount() {
      return m_usageCount;
    }

    public void incUsageCount() {
      m_usageCount++;
    }

    @Override
    public String toString() {
      return "State [m_expiry=" + DATE_FORMAT.format(new Date(m_expiry)) + ", m_usageCount=" + m_usageCount + "]";
    }
  }
}
