/*******************************************************************************
 * Copyright (c) 2011 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Daniel Wiehl (BSI Business Systems Integration AG) - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.jaxws.internal.resources;

import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.scout.jaxws.Activator;
import org.osgi.framework.Bundle;
import org.xml.sax.InputSource;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

import com.sun.xml.ws.transport.http.ResourceLoader;

public class BundleProxyResourceLoader implements ResourceLoader {
  private static final Logger LOG = Logger.getLogger("org.eclipse.scout.jaxws.internal.resources");

  private static final String MSG_BUNDLE_NOT_RESOLVED = "JAXWS resources could not be loaded as bundle '{0}' is not in state RESOLVED.";

  private Bundle m_bundle;

  public BundleProxyResourceLoader(Bundle bundle) {
    m_bundle = bundle;
  }

  @Override
  public URL getCatalogFile() throws MalformedURLException {
    return getResource("/WEB-INF/jax-ws-catalog.xml");
  }

  @Override
  public URL getResource(String path) throws MalformedURLException {
    List<URL> urls = new ArrayList<URL>();

    List<Bundle> bundles = Arrays.asList(Activator.getDefault().getBundle(), m_bundle);
    for (Bundle bundle : bundles) {
      // bundle must be in state RESOLVED at minimum
      if (bundle.getState() < Bundle.RESOLVED) {
        LOG.log(Level.SEVERE, MSG_BUNDLE_NOT_RESOLVED, bundle.getSymbolicName());
        continue;
      }

      try {
        // do not use {@link Bundle#bundle.getEntry(String)} as the m_bundle's classloader must be used in order to work for fragments.
        URL url = bundle.getResource(path);
        if (url != null) {
          urls.add(url);
        }
      }
      catch (IllegalStateException e) {
        //nop
      }
    }

    return urls.isEmpty() ? null : urls.get(0);
  }

  @Override
  public Set<String> getResourcePaths(String path) {
    Set<String> paths = new HashSet<String>();

    List<Bundle> bundles = Arrays.asList(Activator.getDefault().getBundle(), m_bundle);
    for (Bundle bundle : bundles) {
      // bundle must be in state RESOLVED at minimum
      if (bundle.getState() < Bundle.RESOLVED) {
        LOG.log(Level.SEVERE, MSG_BUNDLE_NOT_RESOLVED, bundle.getSymbolicName());
        continue;
      }

      try {
        // Collect resources of bundle and its attached fragments
        Enumeration entries = m_bundle.findEntries(path, "*", false);
        if (entries == null || !entries.hasMoreElements()) {
          continue;
        }
        paths.addAll(getPaths(entries));
      }
      catch (IllegalStateException e) {
        //nop
      }
    }
    return paths;
  }

  private boolean isValidXml(URL url) {
    try {
      XMLReader xmlReader = XMLReaderFactory.createXMLReader();
      // skip files which are not valid XML as JAX-WS fails to start properly otherwise.
      if (new File(url.toExternalForm()).exists()) {
        xmlReader.parse(url.toExternalForm());
      }
      else {
        xmlReader.parse(new InputSource(url.openStream()));
      }
      return true;
    }
    catch (Exception e) {
      return false;
    }
  }

  private Set<String> getPaths(Enumeration entries) {
    Set<String> paths = new HashSet<String>();
    while (entries.hasMoreElements()) {
      URL url = (URL) entries.nextElement();
      if (url == null) {
        continue;
      }
      // Skip hidden files like /WEB-INF/wsdl/.svn/
      if (url.getPath().matches("^.+/\\..+/$")) {
        continue;
      }
      // Skip files in webservice consumer folder as they might interfere with webservice provider files.
      // E.g. if having a consumer WSDL file whose service or port is the very same as from a provider to be published. See @{link EndpointFactory#findPrimary(List)}.
      if (url.getPath().endsWith("/consumer/")) {
        continue;
      }

      if (url.getPath().endsWith("/")) { // to support sub-folders. In turn, JAX-WS recursively browses those folders in @{link DeploymentDescriptorParser#collectDocs(String)} by calling this resolver anew.
        paths.add(url.getPath());
      }
      else if (isValidXml(url)) {
        paths.add(url.getPath());
      }
      else {
        LOG.log(Level.INFO, "Resource '{0}' skipped for webservice resource as file is corrupt or does not contain valid XML. [bundle={1}]", new String[]{url.getPath(), m_bundle.getSymbolicName()});
      }
    }
    return paths;
  }
}
