/*******************************************************************************
 * Copyright (c) 2011 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Daniel Wiehl (BSI Business Systems Integration AG) - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.jaxws.internal.resources;

import java.io.IOException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.eclipse.scout.jaxws.Activator;
import org.osgi.framework.Bundle;

public class BundleProxyClassLoader extends ClassLoader {
  private static final Logger LOG = Logger.getLogger("org.eclipse.scout.jaxws.internal.resources");

  private static final String MSG_BUNDLE_NOT_RESOLVED = "JAXWS resources could not be loaded as bundle '{0}' is not in state RESOLVED.";

  private final ClassLoader m_rootContextClassLoader;
  private final List<Bundle> m_bundles;

  public BundleProxyClassLoader(ClassLoader rootContextClassLoader, Bundle bundle) {
    m_rootContextClassLoader = rootContextClassLoader;
    m_bundles = Arrays.asList(Activator.getDefault().getBundle(), bundle);
  }

  @Override
  protected Class<?> findClass(String name) throws ClassNotFoundException {
    for (Bundle bundle : m_bundles) {
      // bundle must be in state RESOLVED at minimum
      if (bundle.getState() < Bundle.RESOLVED) {
        LOG.log(Level.SEVERE, MSG_BUNDLE_NOT_RESOLVED, bundle.getSymbolicName());
        continue;
      }
      try {
        return bundle.loadClass(name);
      }
      catch (ClassNotFoundException e) {
        //nop
      }
      catch (IllegalStateException e) {
        //nop
      }
    }

    return m_rootContextClassLoader.loadClass(name);
  }

  @Override
  protected URL findResource(String name) {
    for (Bundle bundle : m_bundles) {
      // bundle must be in state RESOLVED at minimum
      if (bundle.getState() < Bundle.RESOLVED) {
        LOG.log(Level.SEVERE, MSG_BUNDLE_NOT_RESOLVED, bundle.getSymbolicName());
        continue;
      }

      try {
        URL url = bundle.getResource(name);
        if (url != null) {
          return url;
        }
      }
      catch (IllegalStateException e) {
        //nop
      }
    }
    return m_rootContextClassLoader.getResource(name);
  }

  @Override
  protected Enumeration<URL> findResources(String name) throws IOException {
    List<URL> urls = new ArrayList<URL>();
    for (Bundle bundle : m_bundles) {
      // bundle must be in state RESOLVED at minimum
      if (bundle.getState() < Bundle.RESOLVED) {
        LOG.log(Level.SEVERE, MSG_BUNDLE_NOT_RESOLVED, bundle.getSymbolicName());
        continue;
      }

      try {
        Enumeration<URL> resources = bundle.getResources(name);
        while (resources != null && resources.hasMoreElements()) {
          urls.add(resources.nextElement());
        }
      }
      catch (IllegalStateException e) {
        //nop
      }
      catch (IOException e) {
        //nop
      }
    }

    Enumeration<URL> rootResources = m_rootContextClassLoader.getResources(name);
    while (rootResources != null && rootResources.hasMoreElements()) {
      urls.add(rootResources.nextElement());
    }
    return Collections.enumeration(urls);
  }
}
