/*******************************************************************************
 * Copyright (c) 2013 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Ivan Motsch (BSI Business Systems Integration AG) - initial API and implementation
 *     Stephan Leicht Vogt (BSI Business Systems Integration AG) - adaption to JaxWS
 ******************************************************************************/
package org.eclipse.scout.jaxws.service.internal;

import java.io.Serializable;
import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;

import javax.xml.ws.Service;
import javax.xml.ws.WebServiceFeature;

import org.eclipse.scout.commons.NumberUtility;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.jaxws.service.IJaxWsConnectionProviderStats;
import org.eclipse.scout.jaxws.service.IWebServiceClient;
import org.eclipse.scout.jaxws.service.JaxWsConnectionProvider;
import org.eclipse.scout.rt.shared.ISession;
import org.eclipse.scout.rt.shared.services.common.session.ISessionService;
import org.eclipse.scout.service.AbstractService;
import org.eclipse.scout.service.IService;
import org.eclipse.scout.service.SERVICES;

import com.sun.xml.ws.client.sei.SEIStub;

/**
 * System-wide connection pool for pooling connections. There is one pool for
 * every IWebServiceClient sub class type. If possible, every scout Session is provided
 * with always the same connection it had in the last request.
 * <p>
 * This class is thread-safe.
 */
public class InternalJaxWsConnectionProvider<S extends Service, P> extends AbstractService implements IService, Serializable {
  private static final long serialVersionUID = 1L;
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(JaxWsConnectionProvider.class);//all connection provider classes use log of JaxWsConnectionProvider

  protected static final long MILLIS_PER_SECOND = 1000L;

  private final Object m_poolLock;
  private final AbstractJaxWsConnectionProviderStats m_stats;
  private final Map<ISession, HashSet<P>> m_idleEntries;
  private final Map<ISession, HashSet<P>> m_busyEntries;
  private final Map<P, Long> m_whenIsPortTypeCreated;
  private final Map<P, Long> m_whenIsLeaseBegun;

  private final IWebServiceClient<S> m_webServiceClient;
  private final Class<?> m_portTypeClazz;
  //
  private long m_lastStatusDump;
  private long m_statusDumpInterval = 60L * MILLIS_PER_SECOND;//1 min
  private transient Thread m_managePoolWorker;
  //configuration
  private long m_checkInterval;
  private int m_maxPoolSizePerSession;
  private int m_maxPoolSizeOverAll;
  private long m_connectionMaxAge;
  private long m_connectionMaxAgeOfLastConnectionPerSession;
  private long m_statementBusyTimeout;

  public InternalJaxWsConnectionProvider(IWebServiceClient<S> webServiceClient, Class<? extends P> portTypeClazz) {
    m_webServiceClient = webServiceClient;
    m_portTypeClazz = portTypeClazz;
    m_poolLock = new Object();
    m_stats = new AbstractJaxWsConnectionProviderStats() {
      @Override
      public String newStateSnapshot() {
        return createStateSnapshot(m_stats);
      }
    };
    m_idleEntries = new HashMap<ISession, HashSet<P>>();
    m_busyEntries = new HashMap<ISession, HashSet<P>>();
    m_whenIsPortTypeCreated = new HashMap<P, Long>();
    m_whenIsLeaseBegun = new HashMap<P, Long>();
    setCheckInterval(30L * MILLIS_PER_SECOND);//30 seconds
    setMaxPoolSizePerSession(5);
    setMaxPoolSizeOverAll(5000);
    setConnectionMaxAge(600L * MILLIS_PER_SECOND);//10 min
    setStatementBusyTimeout(3600L * MILLIS_PER_SECOND);//60 min
  }

  protected IJaxWsConnectionProviderStats getStatsImpl() {
    return m_stats;
  }

  protected int getMaxPoolSizePerSession() {
    return m_maxPoolSizePerSession;
  }

  public void setMaxPoolSizePerSession(int maxPoolSizePerSession) {
    m_maxPoolSizePerSession = maxPoolSizePerSession;
  }

  protected int getMaxPoolSizeOverAll() {
    return m_maxPoolSizeOverAll;
  }

  public void setMaxPoolSizeOverAll(int maxPoolSizeOverAll) {
    m_maxPoolSizeOverAll = maxPoolSizeOverAll;
  }

  /**
   * @return max age in milliseconds
   */
  protected long getConnectionMaxAge() {
    return m_connectionMaxAge;
  }

  /**
   * @param connectionMaxAge
   *          max age in milliseconds
   */
  public void setConnectionMaxAge(long connectionMaxAge) {
    m_connectionMaxAge = connectionMaxAge;
  }

  /**
   * @return max age in milliseconds
   */
  protected long getConnectionMaxAgeOfLastConnectionPerSession() {
    return m_connectionMaxAgeOfLastConnectionPerSession;
  }

  /**
   * @param connectionMaxAgeOfLastConnectionPerSession
   *          max age in milliseconds for the last connection for each session
   */
  public void setConnectionMaxAgeOfLastConnectionPerSession(long connectionMaxAgeOfLastConnectionPerSession) {
    m_connectionMaxAgeOfLastConnectionPerSession = connectionMaxAgeOfLastConnectionPerSession;
  }

  /**
   * @return busy timeout in milliseconds
   */
  protected long getStatementBusyTimeout() {
    return m_statementBusyTimeout;
  }

  /**
   * @param statementBusyTimeout
   *          busy timeout in milliseconds
   */
  public void setStatementBusyTimeout(long statementBusyTimeout) {
    m_statementBusyTimeout = statementBusyTimeout;
  }

  protected void setCheckInterval(long checkInterval) {
    m_checkInterval = checkInterval;
  }

  protected long getCheckInterval() {
    return m_checkInterval;
  }

  public void startHousekeeping() {
    m_managePoolWorker = new Thread(InternalJaxWsConnectionProvider.this.getClass().getSimpleName() + ".managePool") {
      @Override
      public void run() {
        while (m_managePoolWorker == this) {
          try {
            Thread.sleep(m_checkInterval);
          }
          catch (InterruptedException ie) {
            //nop
          }
          synchronized (m_poolLock) {
            managePoolInsideLock(false);
          }
        }
      }
    };
    m_managePoolWorker.setDaemon(true);
    m_managePoolWorker.start();
  }

  protected void stopHousekeeping() {
    m_managePoolWorker = null;
    synchronized (m_poolLock) {
      managePoolInsideLock(true);
    }
  }

  protected int countAllPools() {
    int idleSize = 0;
    for (Iterator<HashSet<P>> entriesSetIterator = m_idleEntries.values().iterator(); entriesSetIterator.hasNext();) {
      HashSet<P> idleEntries = entriesSetIterator.next();
      idleSize = idleSize + idleEntries.size();
    }

    int busySize = 0;
    for (Iterator<HashSet<P>> entriesSetIterator = m_busyEntries.values().iterator(); entriesSetIterator.hasNext();) {
      HashSet<P> busyEntries = entriesSetIterator.next();
      busySize = busySize + busyEntries.size();
    }
    return idleSize + busySize;
  }

  protected HashSet<P> getBusyEntries(ISession session) {
    HashSet<P> busyEntries = m_busyEntries.get(session);
    if (busyEntries == null) {
      busyEntries = new HashSet<P>();
      m_busyEntries.put(session, busyEntries);
    }
    return busyEntries;
  }

  protected HashSet<P> getIdleEntries(ISession session) {
    HashSet<P> idleEntries = m_idleEntries.get(session);
    if (idleEntries == null) {
      idleEntries = new HashSet<P>();
      m_idleEntries.put(session, idleEntries);
    }
    return idleEntries;
  }

  /**
   * @param webserviceClient
   * @return a web service connection
   */
  @SuppressWarnings({"unchecked"})
  protected P leaseConnection(String url, WebServiceFeature... usesJAXBContextFeature) {
    ISessionService service = SERVICES.getService(ISessionService.class);
    if (service == null) {
      throw new IllegalStateException("No session service found!");
    }
    ISession session = service.getCurrentSession();
    if (session == null) {
      throw new IllegalStateException("No session found on session service!");
    }
    int poolSizeOverAll = countAllPools();
    HashSet<P> idleEntries = getIdleEntries(session);
    HashSet<P> busyEntries = getBusyEntries(session);

    synchronized (m_poolLock) {
      P portType = null;

      while (true) {
        managePoolInsideLock(false, session);
        // get next available conn
        if (!idleEntries.isEmpty()) {
          portType = idleEntries.iterator().next();
          // get endpoint URL
          String portTypeUrl = (String) ((javax.xml.ws.BindingProvider) portType).getRequestContext().get(javax.xml.ws.BindingProvider.ENDPOINT_ADDRESS_PROPERTY);
          if (StringUtility.equalsIgnoreCase(url, portTypeUrl)) {
            break;
          }
        }
        //try to create new connection
        if (idleEntries.size() + busyEntries.size() < getMaxPoolSizePerSession()
            && poolSizeOverAll < getMaxPoolSizeOverAll()) {
          // create new connection
          Service webService = m_webServiceClient.getWebService();
          P test = (P) webService.getPort(m_portTypeClazz, usesJAXBContextFeature);
          m_whenIsPortTypeCreated.put(test, System.currentTimeMillis());
          // set endpoint URL
          ((javax.xml.ws.BindingProvider) test).getRequestContext().put(javax.xml.ws.BindingProvider.ENDPOINT_ADDRESS_PROPERTY, url);
          m_stats.connectionOpen();
          if (LOG.isInfoEnabled()) {
            LOG.info("created jax ws connection " + test);
          }
          idleEntries.add(test);
          portType = test;
          break;
        }
        //no connection left, wait
        try {
          if (LOG.isInfoEnabled()) {
            LOG.info("waiting for connection");
          }
          dumpWarningStatus();
          m_poolLock.wait();
        }
        catch (InterruptedException ie) {
          //nop
        }
      }
      //move to busy pool
      idleEntries.remove(portType);
      busyEntries.add(portType);
      m_whenIsPortTypeCreated.put(portType, System.currentTimeMillis());
      if (LOG.isInfoEnabled()) {
        LOG.info("lease   " + portType);
      }
      m_stats.connectionLease();
      return (P) portType;
    }
  }

  /**
   * @return release a proxied connection
   */
  protected void releaseConnection(P portType) {
    ISessionService service = SERVICES.getService(ISessionService.class);
    if (service == null) {
      throw new IllegalStateException("No session service found!");
    }
    ISession session = service.getCurrentSession();
    if (session == null) {
      throw new IllegalStateException("No session found on session service!");
    }
    HashSet<P> idleEntries = getIdleEntries(session);
    HashSet<P> busyEntries = getBusyEntries(session);

    Object h = Proxy.getInvocationHandler(portType);
    m_stats.connectionRelease();
    if (LOG.isInfoEnabled()) {
      LOG.info("release " + h);
    }
    synchronized (m_poolLock) {
      //remove from busy
      if (!busyEntries.remove(portType)) {
        //not one of ours
        enqueueCloseConnectionImpl("Closing unknown connection (async)", portType, false);
        return;
      }
      m_whenIsLeaseBegun.remove(portType);
      idleEntries.add(portType);
      m_poolLock.notifyAll();
    }
  }

  /**
   * Thread worker to manage pool
   * <p>
   * must be called inside lock {@link #m_poolLock}
   */
  protected void managePoolInsideLock(boolean closeAll) {
    managePoolInsideLock(closeAll, null);
  }

  /**
   * Thread worker to manage pool
   * <p>
   * must be called inside lock {@link #m_poolLock}
   */
  protected void managePoolInsideLock(boolean closeAll, ISession session) {
    try {
      // close all or old idle connections
      if (session != null) {
        HashSet<P> idleEntries = m_idleEntries.get(session);
        manageIdleConnections(closeAll, idleEntries);
      }
      else {
        for (Iterator<HashSet<P>> entriesSetIterator = m_idleEntries.values().iterator(); entriesSetIterator.hasNext();) {
          HashSet<P> idleEntries = entriesSetIterator.next();

          manageIdleConnections(closeAll, idleEntries);
        }
      }

      if (!closeAll) {
        // close timed out statements
        if (session != null) {
          HashSet<P> busyEntries = m_busyEntries.get(session);
          manageBusyConnections(closeAll, busyEntries);
        }
        else {
          for (Iterator<HashSet<P>> entriesSetIterator = m_busyEntries.values().iterator(); entriesSetIterator.hasNext();) {
            HashSet<P> busyEntries = entriesSetIterator.next();

            manageBusyConnections(closeAll, busyEntries);
          }
        }
      }
    }
    catch (Throwable t) {
      LOG.warn(null, t);
    }
    //check for alarms
    if (m_stats.getConnectionClosePendingCount() > m_maxPoolSizeOverAll / 2 || warnPoolPerSession()) {
      dumpWarningStatus();
    }
  }

  private void manageIdleConnections(boolean closeAll, HashSet<P> idleEntries) {
    for (Iterator<P> it2 = idleEntries.iterator(); it2.hasNext();) {
      P h = it2.next();
      long created = NumberUtility.nvl(m_whenIsPortTypeCreated.get(h), 0L);
      if (closeAll
          || isLastConnectionAndOlderThanMaxAge(it2, created)
          || isNotLastConnectionAndOlderThanMaxAge(it2, created)) {
        it2.remove();
        m_whenIsPortTypeCreated.remove(h);
        enqueueCloseConnectionImpl("Close old idle connection (sync)", h, true/* || closeAll*/);
      }
    }
  }

  private void manageBusyConnections(boolean closeAll, HashSet<P> busyEntries) {
    for (Iterator<P> it2 = busyEntries.iterator(); it2.hasNext();) {
      P h = it2.next();
      long leaseBegin = NumberUtility.nvl(m_whenIsLeaseBegun.get(h), 0L);
      if (leaseBegin > 0 && leaseBegin + getStatementBusyTimeout() < System.currentTimeMillis()) {
        LOG.warn("Cancelling timed out statement " + h.toString());
        enqueueCancelStatement("Cancelling timed out statement (" + (closeAll ? "sync" : "async") + ")", h, false);
      }
    }
  }

  protected boolean isLastConnectionAndOlderThanMaxAge(Iterator<P> it2, long created) {
    return (!it2.hasNext()) && created + getConnectionMaxAgeOfLastConnectionPerSession() < System.currentTimeMillis();
  }

  protected boolean isNotLastConnectionAndOlderThanMaxAge(Iterator<P> it2, long created) {
    return it2.hasNext() && created + getConnectionMaxAge() < System.currentTimeMillis();
  }

  protected boolean warnPoolPerSession() {
    for (Iterator<HashSet<P>> entriesSetIterator = m_busyEntries.values().iterator(); entriesSetIterator.hasNext();) {
      HashSet<P> busyEntries = entriesSetIterator.next();
      if (busyEntries.size() > m_maxPoolSizePerSession / 2) {
        return true;
      }
    }
    return false;
  }

  protected void enqueueCloseConnectionImpl(final String msg, final P h, boolean sync) {
    m_stats.connectionClosePending();
    Runnable job = new Runnable() {
      @Override
      public void run() {
        if (LOG.isInfoEnabled()) {
          LOG.info(msg);
        }
        SEIStub seiStub = (SEIStub) Proxy.getInvocationHandler(h);
        seiStub.close();
        m_stats.connectionClose();
      }
    };
    if (sync) {
      job.run();
      return;
    }
    Thread t = new Thread(job, msg);
    t.start();
    //optimistically wait up to 5 seconds
    try {
      t.join(5000L);
    }
    catch (Throwable ex) {
      //nop
    }
  }

  protected void enqueueCancelStatement(final String msg, final P h, boolean sync) {
    Runnable job = new Runnable() {
      @Override
      public void run() {
        try {
          SEIStub seiStub = (SEIStub) Proxy.getInvocationHandler(h);
          seiStub.close();
        }
        catch (Throwable t) {
          //nop
        }
      }
    };
    if (sync) {
      job.run();
      return;
    }
    Thread t = new Thread(job, msg);
    t.start();
    //optimistically wait up to 2 seconds
    try {
      t.join(2000L);
    }
    catch (Throwable ex) {
      //nop
    }
  }

  protected void dumpWarningStatus() {
    long now = System.currentTimeMillis();
    if (m_lastStatusDump + m_statusDumpInterval < now) {
      m_lastStatusDump = now;
    }
    else {
      //not now
      return;
    }
    LOG.warn(getStatsImpl().newStateSnapshot());
  }

  protected String createStateSnapshot(IJaxWsConnectionProviderStats stats) {
    StringBuilder buf = new StringBuilder();
    synchronized (m_poolLock) {
      buf.append("@@@ " + JaxWsConnectionProvider.class.getName() + " for " + m_portTypeClazz.getSimpleName() + " stats\n");
      buf.append("\n");

      buf.append(" maxPoolSizeOverAll: " + getMaxPoolSizeOverAll() + "\n");
      buf.append(" maxPoolSizePerSession: " + getMaxPoolSizePerSession() + "\n");
      buf.append(" connectionMaxAge: " + getConnectionMaxAge() + "\n");
      buf.append(" connectionMaxAgeOfLastConnectionPerSession: " + getConnectionMaxAgeOfLastConnectionPerSession() + "\n");
      buf.append(" statementBusyTimeout: " + getStatementBusyTimeout() + "\n");
      buf.append("\n");

      buf.append(" connectionCount: " + stats.getConnectionCount() + "\n");
      buf.append(" connectionClosePendingCount: " + stats.getConnectionClosePendingCount() + "\n");
      buf.append(" connectionLeaseCount: " + stats.getConnectionLeaseCount() + "\n");
      buf.append(" connectionReleaseCount: " + stats.getConnectionReleaseCount() + "\n");
      int busyEntryCount = 0;
      for (Iterator<HashSet<P>> entriesSetIterator = m_busyEntries.values().iterator(); entriesSetIterator.hasNext();) {
        HashSet<P> busyEntries = entriesSetIterator.next();
        busyEntryCount = busyEntryCount + busyEntries.size();
      }
      int idleEntryCount = 0;
      for (Iterator<HashSet<P>> entriesSetIterator = m_idleEntries.values().iterator(); entriesSetIterator.hasNext();) {
        HashSet<P> idleEntries = entriesSetIterator.next();
        idleEntryCount = idleEntryCount + idleEntries.size();
      }
      buf.append("Connections: " + busyEntryCount + " busy, " + idleEntryCount + " idle\n");
      for (Iterator<HashSet<P>> entriesSetIterator = m_busyEntries.values().iterator(); entriesSetIterator.hasNext();) {
        HashSet<P> busyEntries = entriesSetIterator.next();
        for (P h : busyEntries) {
          buf.append(" Busy: " + h.toString() + "\n");
        }
      }
      for (Iterator<HashSet<P>> entriesSetIterator = m_idleEntries.values().iterator(); entriesSetIterator.hasNext();) {
        HashSet<P> idleEntries = entriesSetIterator.next();
        for (P h : idleEntries) {
          buf.append(" Idle: " + h.toString() + "\n");
        }
      }
    }
    return buf.toString();
  }
}
