/*******************************************************************************
 * Copyright (c) 2013 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Ivan Motsch (BSI Business Systems Integration AG) - initial API and implementation
 *     Stephan Leicht Vogt (BSI Business Systems Integration AG) - adaption to JaxWS
 ******************************************************************************/
package org.eclipse.scout.jaxws.service;

import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;

import javax.xml.ws.Service;

import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.annotations.ConfigProperty;
import org.eclipse.scout.commons.annotations.Order;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.jaxws.service.internal.JaxWsConnectionProviderDiagnostic;
import org.eclipse.scout.rt.server.admin.diagnostic.DiagnosticFactory;
import org.eclipse.scout.service.AbstractService;
import org.osgi.framework.ServiceRegistration;

public class JaxWsConnectionProviderService extends AbstractService implements IJaxWsConnectionProvider {
  private static final long serialVersionUID = 1L;
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(JaxWsConnectionProvider.class);//all connection provider classes use log of JaxWsConnectionProvider

  protected static final long MILLIS_PER_SECOND = 1000L;
  private static final boolean DIAGNOSTICS_EXIST;

  private final Map<IWebServiceClient<?>, JaxWsConnectionProvider> m_webServciceConnectionProviderMap = new HashMap<IWebServiceClient<?>, JaxWsConnectionProvider>();
  private Object m_webServciceConnectionProviderMapLock = new Object();

  private int m_maxPoolSizePerSession;
  private int m_maxPoolSizeOverAll;
  private long m_connectionMaxAge;
  private long m_connectionMaxAgeOfLastConnectionPerSession;
  private long m_statementBusyTimeout;

  private JaxWsConnectionProviderDiagnostic m_diagnostic;

  static {
    boolean diagnosticsExists = false;
    try {
      diagnosticsExists = StringUtility.hasText(DiagnosticFactory.class.getName());
    }
    catch (Exception e) {
    }
    DIAGNOSTICS_EXIST = diagnosticsExists;
  }

  @Override
  public void initializeService(ServiceRegistration registration) {
    initConfig();
    super.initializeService(registration);

    if (DIAGNOSTICS_EXIST) {
      m_diagnostic = new JaxWsConnectionProviderDiagnostic(this);
      DiagnosticFactory.addDiagnosticStatusProvider(m_diagnostic);
    }
  }

  @Override
  public <S extends Service, P> JaxWsConnectionProvider<S, P> getConnectionProvider(IWebServiceClient<S> webServiceClient, Class<? extends P> portTypeClazz) {
    synchronized (m_webServciceConnectionProviderMapLock) {
      @SuppressWarnings("unchecked")
      JaxWsConnectionProvider<S, P> connectionProvider = m_webServciceConnectionProviderMap.get(webServiceClient);
      if (connectionProvider == null) {
        connectionProvider = new JaxWsConnectionProvider<S, P>(webServiceClient, portTypeClazz);
        connectionProvider.setMaxPoolSizePerSession(m_maxPoolSizePerSession);
        connectionProvider.setMaxPoolSizeOverAll(m_maxPoolSizeOverAll);
        connectionProvider.setConnectionMaxAge(m_connectionMaxAge);
        connectionProvider.setConnectionMaxAgeOfLastConnectionPerSession(m_connectionMaxAgeOfLastConnectionPerSession);
        connectionProvider.setStatementBusyTimeout(m_statementBusyTimeout);
        connectionProvider.startHousekeeping();
        m_webServciceConnectionProviderMap.put(webServiceClient, connectionProvider);
      }
      return connectionProvider;
    }
  }

  @Override
  public Collection<JaxWsConnectionProvider> getAllConnectionProviders() {
    return Collections.unmodifiableCollection(m_webServciceConnectionProviderMap.values());
  }

  protected void initConfig() {
//    setMinPoolSize(getConfiguredMaxPoolSizePerSession());TODO does it need a min-poolsize? Like at least one connection per session/user...
    setMaxPoolSizePerSession(getConfiguredMaxPoolSizePerSession());
    setMaxPoolSizeOverAll(getConfiguredMaxPoolSizeOverAll());
    setConnectionMaxAge(getConfiguredConnectionMaxAge());
    setConnectionMaxAgeOfLastConnectionPerSession(getConfiguredConnectionMaxAgeOfLastConnectionPerSession());
    setStatementBusyTimeout(getConfiguredStatementBusyTimeout());
  }

  @ConfigProperty(ConfigProperty.INTEGER)
  @Order(30)
  protected Integer getConfiguredMaxPoolSizePerSession() {
    return 5;
  }

  @ConfigProperty(ConfigProperty.INTEGER)
  @Order(40)
  protected Integer getConfiguredMaxPoolSizeOverAll() {
    return 5000;
  }

  @ConfigProperty(ConfigProperty.LONG)
  @Order(50)
  protected Long getConfiguredConnectionMaxAge() {
    return 600L;//10min
  }

  @ConfigProperty(ConfigProperty.LONG)
  @Order(60)
  protected Long getConfiguredConnectionMaxAgeOfLastConnectionPerSession() {
    return 3600L;//1hour
  }

  @ConfigProperty(ConfigProperty.LONG)
  @Order(70)
  protected Long getConfiguredStatementBusyTimeout() {
    return 3600L;//1hour
  }

  public int getMaxPoolSizePerSession() {
    return m_maxPoolSizePerSession;
  }

  /**
   * @param configuredMaxPoolSizePerSession
   */
  public void setMaxPoolSizePerSession(int configuredMaxPoolSizePerSession) {
    m_maxPoolSizePerSession = configuredMaxPoolSizePerSession;
  }

  public int getMaxPoolSizeOverAll() {
    return m_maxPoolSizeOverAll;
  }

  /**
   * @param configuredMaxPoolSizePerSession
   */
  public void setMaxPoolSizeOverAll(int configuredMaxPoolSizeOverAll) {
    m_maxPoolSizeOverAll = configuredMaxPoolSizeOverAll;
  }

  public long getConnectionMaxAge() {
    return m_connectionMaxAge;
  }

  /**
   * @param connectionMaxAge
   *          max age in seconds
   */
  public void setConnectionMaxAge(long connectionMaxAge) {
    m_connectionMaxAge = connectionMaxAge * MILLIS_PER_SECOND;//10 min
  }

  public long getConnectionMaxAgeOfLastConnectionPerSession() {
    return m_connectionMaxAgeOfLastConnectionPerSession;
  }

  /**
   * @param m_connectionMaxAgeOfLastConnectionPerSession
   *          max age in seconds for the last connection for each session
   */
  public void setConnectionMaxAgeOfLastConnectionPerSession(long connectionMaxAgeOfLastConnectionPerSession) {
    m_connectionMaxAgeOfLastConnectionPerSession = connectionMaxAgeOfLastConnectionPerSession * MILLIS_PER_SECOND;//60 min
  }

  public long getStatementBusyTimeout() {
    return m_statementBusyTimeout;
  }

  /**
   * @param statementBusyTimeout
   *          busy timeout in seconds
   */
  public void setStatementBusyTimeout(long statementBusyTimeout) {
    m_statementBusyTimeout = statementBusyTimeout * MILLIS_PER_SECOND;//60 min
  }

  @Override
  public void disposeServices() {
    if (DIAGNOSTICS_EXIST) {
      DiagnosticFactory.removeDiagnosticStatusProvider(m_diagnostic);
    }

    super.disposeServices();
    for (Iterator it = m_webServciceConnectionProviderMap.values().iterator(); it.hasNext();) {
      JaxWsConnectionProvider provider = (JaxWsConnectionProvider) it.next();
      provider.stop();
      it.remove();
    }
  }
}
