/*******************************************************************************
 * Copyright (c) 2011 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Daniel Wiehl (BSI Business Systems Integration AG) - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.jaxws.service;

import java.lang.reflect.Constructor;
import java.lang.reflect.ParameterizedType;
import java.net.HttpURLConnection;
import java.net.URL;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;

import javax.xml.namespace.QName;
import javax.xml.ws.Service;
import javax.xml.ws.WebServiceClient;
import javax.xml.ws.WebServiceException;
import javax.xml.ws.handler.Handler;
import javax.xml.ws.handler.HandlerResolver;
import javax.xml.ws.handler.PortInfo;
import javax.xml.ws.handler.soap.SOAPHandler;
import javax.xml.ws.handler.soap.SOAPMessageContext;

import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.scout.commons.NumberUtility;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.annotations.ConfigOperation;
import org.eclipse.scout.commons.annotations.ConfigProperty;
import org.eclipse.scout.commons.annotations.Order;
import org.eclipse.scout.commons.exception.ProcessingException;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.commons.serialization.SerializationUtility;
import org.eclipse.scout.jaxws.annotation.ScoutWebServiceClient;
import org.eclipse.scout.jaxws.security.consumer.IAuthenticationHandler;
import org.eclipse.scout.rt.server.ThreadContext;
import org.eclipse.scout.rt.server.transaction.AbstractTransactionMember;
import org.eclipse.scout.rt.server.transaction.ITransaction;
import org.eclipse.scout.service.AbstractService;
import org.eclipse.scout.service.IService;
import org.eclipse.scout.service.SERVICES;
import org.osgi.framework.ServiceRegistration;

import com.sun.xml.ws.client.BindingProviderProperties;

/**
 * <p>
 * To proxy a web service stub to an {@link IService}.
 * </p>
 * <ul>
 * <li>An authentication strategy can be specified by setting the annotation {@link ScoutWebServiceClient}<br/>
 * A user's credential can be configured in config.ini by setting the properties domain, username and password or
 * directly in respective <code>getConfigured</code> methods.</li>
 * <li>Custom handlers can be installed by overriding {@link AbstractWebServiceClient#execInstallHandlers(List)}.</li>
 * <li>The endpoint URL can be configured by specifying the property url in config.ini or directly in
 * {@link AbstractWebServiceClient#getConfiguredUrl()}. If it is about a dynamic URL, it also can be set at runtime when
 * obtaining the port type.</li>
 * </ul>
 * 
 * @param <S>
 *          The service to be proxied. The service is unique among all services defined within in the enclosing WSDL
 *          document and groups a set of related ports together.
 * @param <P>
 *          The port type to communicate with. The port type is unique among all port types defined within in the
 *          enclosing WSDL document. A port type is a named set of abstract operations and the abstract messages
 *          involved. The port type is unique among all port types defined within in the enclosing WSDL document.
 */
@ScoutWebServiceClient
public abstract class AbstractWebServiceClient<S extends Service, P> extends AbstractService implements IWebServiceClient<S> {
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(AbstractWebServiceClient.class);

  private WebServiceClient m_webServiceClientAnnotation;
  private String m_transactionMemberId;
  private String m_url;
  private URL m_wsdlLocation;
  private String m_targetNamespace;
  private String m_serviceName;
  private Integer m_requestTimeout;
  private Integer m_connectTimeout;

  private String m_username;
  private String m_password;

  private Class<? extends Service> m_serviceClazz;
  private Class<?> m_portTypeClazz;

  @SuppressWarnings("unchecked")
  @Override
  public void initializeService(ServiceRegistration registration) {
    m_serviceClazz = (Class<? extends Service>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[0];
    m_portTypeClazz = (Class<?>) ((ParameterizedType) getClass().getGenericSuperclass()).getActualTypeArguments()[1];
    m_webServiceClientAnnotation = m_serviceClazz.getAnnotation(WebServiceClient.class);

    if (m_webServiceClientAnnotation == null) {
      throw new WebServiceException("Missing required annotation '" + WebServiceClient.class.getName() + "' on web service client '" + getClass().getName() + "'");
    }

    initConfig();
    super.initializeService(registration);
  }

  protected void initConfig() {
    String tid = getConfiguredTransactionMemberId();
    if (tid == null) {
      tid = getClass().getSimpleName() + "." + "transaction";
    }
    setTransactionMemberId(tid);
    setWsdlLocation(getConfiguredWsdlLocation());
    setUrl(getConfiguredUrl());
    setTargetNamespace(getConfiguredTargetNamespace());
    setServiceName(getConfiguredServiceName());
    setUsername(getConfiguredUsername());
    setPassword(getConfiguredPassword());
    setConnectTimeout(getConfiguredConnectTimeout());
    setRequestTimeout(getConfiguredRequestTimeout());
  }

  /**
   * To get the service stub specified by generic type parameter {@link S}.<br/>
   * Please be in mind, that the endpoint URL is set on port type level. Therefore, when working directly on the
   * service, you have to set the endpoint URL manually when calling the service. <br/>
   * By using {@link AbstractWebServiceClient#getPortType()}, the URL is set accordingly.
   * 
   * @return
   */
  @SuppressWarnings("unchecked")
  @Override
  public S getWebService() {
    if (getWsdlLocation() == null) {
      throw new WebServiceException("No location for WSDL configured on web service proxy '" + getClass().getName() + "'.");
    }

    try {
      ClassLoader backup = Thread.currentThread().getContextClassLoader();
      S service;
      try {
        Thread.currentThread().setContextClassLoader(SerializationUtility.getClassLoader());
        Constructor<? extends Service> constructor = m_serviceClazz.getConstructor(URL.class, QName.class);
        service = (S) constructor.newInstance(getWsdlLocation(), new QName(getTargetNamespace(), getServiceName()));
      }
      finally {
        Thread.currentThread().setContextClassLoader(backup);
      }

      // install handlers
      service.setHandlerResolver(new HandlerResolver() {

        @Override
        public List<Handler> getHandlerChain(PortInfo portInfo) {
          ArrayList<Handler> list = new ArrayList<Handler>();

          // authentication handler
          IAuthenticationHandler authenticationHandler = createAuthenticationHandler();
          if (authenticationHandler != null) {
            try {
              (authenticationHandler).setUsername(getUsername());
              (authenticationHandler).setPassword(getPassword());

              if (execPrepareAuthenticationHandler(authenticationHandler)) {
                list.add(authenticationHandler);
              }
            }
            catch (ProcessingException e) {
              LOG.error("Authentication handler could not be installed.", e);
            }
          }

          List<SOAPHandler<SOAPMessageContext>> handlers = new LinkedList<SOAPHandler<SOAPMessageContext>>();
          execInstallHandlers(handlers);
          if (handlers.size() > 0) {
            list.addAll(handlers);
          }

          return list;
        }
      });

      return service;
    }
    catch (Exception e) {
      throw new WebServiceException("Webservice proxy '" + getClass().getName() + "' could not be created.", e);
    }
  }

  public Class<?> getPortTypeClass() {
    return m_portTypeClazz;
  }

  /**
   * To get the port type specified by generic type parameter {@link P}.
   * 
   * @return
   */
  public P getPortType() throws ProcessingException {
    return getPortType(getUrl());
  }

  /**
   * To get the port type specified by generic type parameter {@link P}.
   * 
   * @param url
   *          {@link URL} to connect with endpoint. Do not use this method to distinguish between development and
   *          production {@link URL}. Use property in config.ini instead.
   * @return
   */
  public P getPortType(String url) throws ProcessingException {
    return getConnection(url);
  }

  protected P getConnection(String url) throws ProcessingException {
    if (StringUtility.isNullOrEmpty(url)) {
      throw new WebServiceException("No endpoint URL configured for web service client '" + getClass().getName() + "'.");
    }

    ITransaction reg = ThreadContext.getTransaction();
    if (reg == null) {
      throw new ProcessingException("no ITransaction available, use ServerJob to run transaction");
    }
    @SuppressWarnings("unchecked")
    P_JaxWsTransactionMember member = (P_JaxWsTransactionMember) reg.getMember(getTransactionMemberId());
    if (member == null) {
      P conn;
      try {
        conn = leaseConnection(url);
        member = new P_JaxWsTransactionMember(getTransactionMemberId(), conn);
        reg.registerMember(member);
        // this is the start of the transaction
        execBeginTransaction();
      }
      catch (ProcessingException e) {
        throw e;
      }
      catch (Throwable e) {
        throw new ProcessingException("getTransaction", e);
      }
    }
    return member.getConnection();
  }

  /*
   * Internals
   */
  private P leaseConnection(String url) throws Throwable {
    P conn = execCreateConnection(url);
    return conn;
  }

  private void releaseConnection(P conn) {
    try {
      execReleaseConnection(conn);
    }
    catch (Throwable e) {
      LOG.error(null, e);
    }
  }

  @SuppressWarnings("unchecked")
  private void releaseConnectionInternal(P conn) throws Throwable {
    JaxWsConnectionProvider connectionProvider = SERVICES.getService(IJaxWsConnectionProvider.class).getConnectionProvider(this, m_portTypeClazz);
    connectionProvider.closeConnection(conn);
  }

  @ConfigOperation
  @Order(15)
  protected void execAfterConnectionCreated(P conn) throws ProcessingException {
  }

  /**
   * called just after the transaction has started
   */
  @ConfigOperation
  @Order(20)
  protected void execBeginTransaction() throws ProcessingException {
  }

  @ConfigOperation
  @Order(30)
  protected P execCreateConnection(String url) throws Throwable {
    JaxWsConnectionProvider<S, ?> connectionProvider = SERVICES.getService(IJaxWsConnectionProvider.class).getConnectionProvider(this, m_portTypeClazz);
    @SuppressWarnings("unchecked")
    P portType = (P) connectionProvider.getConnection(url);

    // set request timeout
    if (NumberUtility.nvl(getRequestTimeout(), 0) > 0) {
      ((javax.xml.ws.BindingProvider) portType).getRequestContext().put(BindingProviderProperties.REQUEST_TIMEOUT, getRequestTimeout());
    }

    // set connect timeout
    if (NumberUtility.nvl(getConnectTimeout(), 0) > 0) {
      ((javax.xml.ws.BindingProvider) portType).getRequestContext().put(BindingProviderProperties.CONNECT_TIMEOUT, getConnectTimeout());
    }
    return portType;
  }

  @ConfigOperation
  @Order(35)
  protected void execReleaseConnection(P conn) throws Throwable {
    releaseConnectionInternal(conn);
  }

  /**
   * Called just before the transaction is being committed or rollbacked.<br>
   * Do not call commit here, the flag is just meant as a hint.
   * Statements are executed, even if the transaction is canceled.
   */
  @ConfigOperation
  @Order(50)
  protected void execEndTransaction(boolean willBeCommitted) throws ProcessingException {
  }

  /**
   * Possibility to call a commit phase 1 on implementing web service</p>
   * Here all other transaction members are still usable.
   */
  @ConfigOperation
  @Order(70)
  public boolean execPreCommit(P conn) {
    return true;
  }

  /**
   * Possibility to call a commit phase 2 on implementing web service</p>
   * Do NOT use any other transaction members. They could already be committed!
   */
  @ConfigOperation
  @Order(70)
  public void execCommit(P conn) {
  }

  /**
   * Possibility to call a rollback on implementing web service
   */
  @ConfigOperation
  @Order(80)
  public void execRollback(P conn) {
  }

  @ConfigProperty(ConfigProperty.STRING)
  @Order(90)
  protected String getConfiguredTransactionMemberId() {
    return null;
  }

  /**
   * To be overwritten if WSDL file is located somewhere else than specified in {@link WebServiceClient#wsdlLocation()}.
   * This location can be specified when generating the stub with the option -wsdllocation.
   * 
   * @return URL to WSDL file
   */
  @ConfigProperty(ConfigProperty.OBJECT)
  @Order(20)
  protected URL getConfiguredWsdlLocation() {
    String wsdlLocation = m_webServiceClientAnnotation.wsdlLocation();
    if (StringUtility.isNullOrEmpty(wsdlLocation)) {
      throw new WebServiceException("No WSDL file location configured on web service proxy '" + getClass().getName() + "'");
    }
    IPath path = new Path("").addTrailingSeparator().append(new Path(wsdlLocation)); // ensure root relative path
    URL urlWsdlLocation = getClass().getResource(path.toPortableString());
    if (urlWsdlLocation == null) {
      throw new WebServiceException("Could not find WSDL file '" + StringUtility.nvl(wsdlLocation, "?") + "' of web service client '" + getClass().getName() + "'");
    }
    return urlWsdlLocation;
  }

  /**
   * To configure a static endpoint URL
   * 
   * @return
   */
  @ConfigProperty(ConfigProperty.STRING)
  @Order(30)
  protected String getConfiguredUrl() {
    return null;
  }

  /**
   * To be overwritten if service name to be called is different from {@link WebServiceClient#name()} in {@link Service}
   * 
   * @return unqualified service name to be called.
   */
  @Order(40)
  protected String getConfiguredServiceName() {
    return m_webServiceClientAnnotation.name();
  }

  /**
   * To be overwritten if targetNamespace of service to be called is different from
   * {@link WebServiceClient#targetNamespace()} in {@link Service}
   * 
   * @return
   */
  @Order(50)
  protected String getConfiguredTargetNamespace() {
    return m_webServiceClientAnnotation.targetNamespace();
  }

  /**
   * To configure a static user's credential
   * 
   * @return
   */
  @ConfigProperty(ConfigProperty.STRING)
  @Order(60)
  protected String getConfiguredUsername() {
    return null;
  }

  /**
   * To configure a static user's credential
   * 
   * @return
   */
  @ConfigProperty(ConfigProperty.STRING)
  @Order(70)
  protected String getConfiguredPassword() {
    return null;
  }

  /**
   * To configure the maximum timeout in milliseconds to wait for the connection to be established.
   * 
   * @See {@link HttpURLConnection#setConnectTimeout(int)}
   * @return the maximal timeout to wait for the connection to be established.
   */
  @ConfigProperty(ConfigProperty.INTEGER)
  @Order(80)
  protected Integer getConfiguredConnectTimeout() {
    return null;
  }

  /**
   * To configure the maximum timeout in milliseconds to wait for response data to be ready to be read.
   * 
   * @See {@link HttpURLConnection#setReadTimeout(int)}
   * @return the maximal timeout to wait for response data to be ready to be read.
   */
  @ConfigProperty(ConfigProperty.INTEGER)
  @Order(90)
  protected Integer getConfiguredRequestTimeout() {
    return null;
  }

  /**
   * Is called before the authentication handler is installed.
   * 
   * @param authenticationHandler
   *          the authentication handler
   * @return true to install the authentication handler or false to not install it
   * @throws ProcessingException
   */
  @ConfigOperation
  @Order(10)
  protected boolean execPrepareAuthenticationHandler(IAuthenticationHandler authenticationHandler) throws ProcessingException {
    return true;
  }

  /**
   * Add custom handlers to the handler chain
   * 
   * @param handlers
   *          handlers to be installed
   * @throws ProcessingException
   */
  @ConfigOperation
  @Order(20)
  protected void execInstallHandlers(List<SOAPHandler<SOAPMessageContext>> handlers) {
  }

  private IAuthenticationHandler createAuthenticationHandler() {
    ScoutWebServiceClient annotation = null;
    Class<?> declaringClass = getClass();
    while (annotation == null && declaringClass != null && declaringClass != Object.class) {
      annotation = declaringClass.getAnnotation(ScoutWebServiceClient.class);
      declaringClass = declaringClass.getSuperclass();
    }

    if (annotation == null) {
      return null;
    }

    Class<? extends IAuthenticationHandler> authenticationHandlerClazz = annotation.authenticationHandler();
    if (authenticationHandlerClazz == null || authenticationHandlerClazz == IAuthenticationHandler.NONE.class) {
      return null;
    }

    IAuthenticationHandler authenticationHandler = null;
    try {
      authenticationHandler = authenticationHandlerClazz.newInstance();
    }
    catch (Exception e) {
      LOG.error("Failed to instantiate authentication handler '" + authenticationHandlerClazz.getName() + "'.", e);
      return null;
    }

    return authenticationHandler;
  }

  @Override
  public String getTransactionMemberId() {
    return m_transactionMemberId;
  }

  @Override
  public String getUsername() {
    return m_username;
  }

  @Override
  public String getPassword() {
    return m_password;
  }

  @Override
  public String getServiceName() {
    return m_serviceName;
  }

  @Override
  public String getTargetNamespace() {
    return m_targetNamespace;
  }

  @Override
  public URL getWsdlLocation() {
    return m_wsdlLocation;
  }

  public void setTransactionMemberId(String s) {
    m_transactionMemberId = s;
  }

  @Override
  public void setUsername(String username) {
    m_username = username;
  }

  @Override
  public void setPassword(String password) {
    m_password = password;
  }

  @Override
  public void setServiceName(String serviceName) {
    m_serviceName = serviceName;
  }

  @Override
  public void setTargetNamespace(String targetNamespace) {
    m_targetNamespace = targetNamespace;
  }

  @Override
  public Integer getRequestTimeout() {
    return m_requestTimeout;
  }

  @Override
  public void setRequestTimeout(Integer requestTimeout) {
    m_requestTimeout = requestTimeout;
  }

  @Override
  public Integer getConnectTimeout() {
    return m_connectTimeout;
  }

  @Override
  public void setConnectTimeout(Integer connectTimeout) {
    m_connectTimeout = connectTimeout;
  }

  @Override
  public void setWsdlLocation(URL wsdlLocation) {
    m_wsdlLocation = wsdlLocation;
  }

  @Override
  public String getUrl() {
    return m_url;
  }

  @Override
  public void setUrl(String url) {
    m_url = url;
  }

  private final class P_JaxWsTransactionMember extends AbstractTransactionMember {
    private final P m_conn;
    /** true during completion phase (commit/rollback) */
    private boolean m_finishingTransaction;

    public P_JaxWsTransactionMember(String transactionMemberId, P conn) {
      super(transactionMemberId);
      m_conn = conn;
    }

    public P getConnection() {
      return m_conn;
    }

    protected void setFinishingTransaction(boolean finishingTransaction) {
      m_finishingTransaction = finishingTransaction;
    }

    @Override
    public boolean needsCommit() {
      return true;
    }

    @Override
    public boolean commitPhase1() {
      return execPreCommit(m_conn);
    }

    @Override
    public void commitPhase2() {
      try {
        // this is the end of the transaction
        try {
          setFinishingTransaction(true);
          execEndTransaction(false);
        }
        finally {
          setFinishingTransaction(false);
        }
        execCommit(m_conn);
      }
      catch (Exception e) {
        LOG.error(null, e);
      }
    }

    @Override
    public void rollback() {
      try {
        // this is the end of the transaction
        try {
          setFinishingTransaction(true);
          execEndTransaction(false);
        }
        finally {
          setFinishingTransaction(false);
        }
        execRollback(m_conn);
      }
      catch (Exception e) {
        if (!ThreadContext.getTransaction().isCancelled()) {
          LOG.error(null, e);
        }
      }
    }

    @Override
    public void release() {
      releaseConnection(m_conn);
    }
  }// end private class

}
