/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.create;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.PathUtility;

/**
 * Create the delta of zip 1 and 2 and detect renamings
 */
public class ZipDeltaBuilder {
  private int m_maxDeltaLevel;
  private int m_maxRenameLevel;
  private int m_currentProcessLevel;
  private Map<String, String> m_renameMap;
  private File m_tempDir;

  public ZipDeltaBuilder() {
    this(0, 0);
  }

  public ZipDeltaBuilder(int maxDeltaLevel, int maxRenameLevel) {
    m_maxDeltaLevel = maxDeltaLevel;
    m_maxRenameLevel = maxRenameLevel;
    m_renameMap = new HashMap<String, String>();
  }

  /**
   * @return true if the two zip are equal and nothing has to be done, otherwise create a delta zip and return false
   */
  public boolean process(File fOld, File fNew, File fOut, File tmpDir) throws Exception {
    m_tempDir = tmpDir;
    m_currentProcessLevel = 0;
    if (!acceptDeltaProcessing(m_currentProcessLevel)) {
      return true;
    }
    ZipFile zOld = new ZipFile(fOld);
    try {
      ZipFile zNew = new ZipFile(fNew);
      try {
        return processImpl(null, "", zOld, zNew, fOut);
      }
      finally {
        zNew.close();
      }
    }
    finally {
      zOld.close();
    }
  }

  public void addRenameMapping(String oldPath, String newPath) {
    m_renameMap.put(oldPath, newPath);
  }

  protected void writeRenameMappings(ZipOutputStream zOut) throws IOException {
    StringBuilder buf = new StringBuilder();
    for (Map.Entry<String, String> e : m_renameMap.entrySet()) {
      buf.append(e.getKey());
      buf.append("\t");
      buf.append(e.getValue());
      buf.append("\n");
    }
    ZipEntry ze = new ZipEntry(PathUtility.RENAME_ENTRY_NAME);
    writeEntry(ze.getName(), ze, new ByteArrayInputStream(buf.toString().getBytes("UTF-8")), true, zOut);
  }

  /**
   * @param referenceEntry
   * @param pathPrefix
   * @param fOld
   * @param fNew
   * @param fOut
   */
  protected boolean processImpl(ZipEntry referenceEntry, String pathPrefix, ZipFile fOld, ZipFile fNew, File fOut) throws Exception {
    ZipOutputStream zOut = new ZipOutputStream(new FileOutputStream(fOut));
    int outEntryCount = 0;
    try {
      zOut.setLevel(Deflater.BEST_COMPRESSION);
      //read
      TreeMap<String, ZipEntry> entryMapOld = new TreeMap<String, ZipEntry>();
      readArchive(fOld, entryMapOld);
      TreeMap<String, ZipEntry> entryMapNew = new TreeMap<String, ZipEntry>();
      readArchive(fNew, entryMapNew);
      //detect rename operations
      if (acceptDetectRenameOperations(m_currentProcessLevel)) {
        detectRenameMappings(pathPrefix, new TreeSet<String>(entryMapOld.keySet()), new TreeSet<String>(entryMapNew.keySet()));
      }
      //remove equal, remove modified
      ZipEntry zOld;
      ZipEntry zNew;
      for (String oldName : new ArrayList<String>(entryMapOld.keySet())) {
        String newName = getNewPathFor(pathPrefix + oldName).substring(pathPrefix.length());
        zOld = entryMapOld.get(oldName);
        zNew = entryMapNew.get(newName);
        if (zOld == null || zNew == null) {
          continue;
        }
        try {
          if (FileUtility.isMatchingSizeAndCrc(zOld, zNew)) {
            continue;
          }
          if (acceptDeltaProcessing(m_currentProcessLevel + 1) && PathUtility.isArchiveEntry(zNew) && !FileUtility.isMatchingSizeAndCrc(zOld, zNew)) {
            //inner archives
            File tmpIn1 = File.createTempFile("old", ".zip", m_tempDir);
            File tmpIn2 = File.createTempFile("new", ".zip", m_tempDir);
            File tmpOut = File.createTempFile("delta", ".zip", m_tempDir);
            try {
              m_currentProcessLevel++;
              String subPathPrefix = pathPrefix + newName + (newName.endsWith("/") ? "" : "/");
              FileUtility.streamContent(zOld.getSize(), fOld.getInputStream(zOld), true, new FileOutputStream(tmpIn1), true, null);
              FileUtility.streamContent(zNew.getSize(), fNew.getInputStream(zNew), true, new FileOutputStream(tmpIn2), true, null);
              FileUtility.mkdirs(tmpOut.getParentFile());
              boolean equal = processImpl(zNew, subPathPrefix, new ZipFile(tmpIn1), new ZipFile(tmpIn2), tmpOut);
              if (equal) {
                continue;
              }
              zNew.setComment(PathUtility.DELTA_ARCHIVE_COMMENT);
              zNew.setMethod(ZipEntry.DEFLATED);
              zNew.setSize(tmpOut.length());
              outEntryCount++;
              writeEntry(newName, zNew, new FileInputStream(tmpOut), true, zOut);
            }
            finally {
              m_currentProcessLevel--;
              if (!tmpIn1.delete()) {
                tmpIn1.deleteOnExit();
              }
              if (!tmpIn2.delete()) {
                tmpIn2.deleteOnExit();
              }
              if (!tmpOut.delete()) {
                tmpOut.deleteOnExit();
              }
            }
            continue;
          }
          //default
          outEntryCount++;
          writeEntry(newName, zNew, zNew.isDirectory() ? null : fNew.getInputStream(zNew), true, zOut);
        }
        finally {
          entryMapOld.remove(oldName);
          entryMapNew.remove(newName);
        }
      }
      //remaining entries in 'new' are added entries
      for (Map.Entry<String, ZipEntry> e : entryMapNew.entrySet()) {
        String newName = e.getKey();
        zNew = e.getValue();
        outEntryCount++;
        writeEntry(newName, zNew, zNew.isDirectory() ? null : fNew.getInputStream(zNew), true, zOut);
      }
      //remaining entries in 'old' are removed entries, mark them with a comment of DELETED_COMMENT
      for (Map.Entry<String, ZipEntry> e : entryMapOld.entrySet()) {
        String oldName = e.getKey();
        String newName = getNewPathFor(pathPrefix + oldName).substring(pathPrefix.length());
        zOld = e.getValue();
        zOld.setComment(PathUtility.DELETED_COMMENT);
        zOld.setCrc(0L);
        zOld.setSize(0L);
        outEntryCount++;
        writeEntry(newName, zOld, zOld.isDirectory() ? null : fOld.getInputStream(zOld), true, zOut);
      }
      if (m_currentProcessLevel == 0) {
        outEntryCount++;
        writeRenameMappings(zOut);
      }
      if (outEntryCount == 0) {
        return true;
      }
      return false;
    }
    finally {
      if (outEntryCount > 0) {
        try {
          zOut.finish();
        }
        catch (Throwable t) {
        }
      }
      try {
        zOut.close();
      }
      catch (Throwable t) {
      }
      try {
        fOld.close();
      }
      catch (Throwable t) {
      }
      try {
        fNew.close();
      }
      catch (Throwable t) {
      }
    }
  }

  /**
   * default accepts levels that were passed in constructor, default 0
   */
  protected boolean acceptDeltaProcessing(int level) {
    return level <= m_maxDeltaLevel;
  }

  /**
   * default only detects rename operations on top level
   */
  protected boolean acceptDetectRenameOperations(int level) {
    return level <= m_maxRenameLevel;
  }

  protected String getNewPathFor(String oldPath) {
    String[] oldPathSplit = PathUtility.splitLastPart(oldPath);
    if (oldPathSplit[0].length() > 0) {
      //recursion
      String newBasePath = getNewPathFor(oldPathSplit[0]);
      String newPath = m_renameMap.get(newBasePath + oldPathSplit[1]);
      if (newPath == null) {
        newPath = newBasePath + oldPathSplit[1];
      }
      return newPath;
    }
    String newPath = m_renameMap.get(oldPath);
    if (newPath == null) {
      newPath = oldPath;
    }
    return newPath;
  }

  /**
   * @param pathPrefix
   * @param oldPaths
   *          is a mutable copy of all jar paths (including directories that end with '/')
   *          are checked before longer paths!
   * @param newPaths
   *          is a mutable copy of all jar paths (including directories that end with '/')
   * @return a map from pathPrefix||jarPath -> pathPrefix||newJarPath
   */
  protected void detectRenameMappings(String pathPrefix, TreeSet<String> oldPaths, TreeSet<String> newPaths) {
    if (pathPrefix.length() > 0 && !pathPrefix.endsWith("/")) {
      throw new IllegalArgumentException("pathPrefix must end in /: " + pathPrefix);
    }
    HashMap<String, String> unverToAbsOld = new HashMap<String, String>();
    for (String oldPath : oldPaths) {
      String abs = getNewPathFor(pathPrefix + oldPath);
      String unver = PathUtility.getPathWithoutVersion(abs);
      if (unver != null) {
        unverToAbsOld.put(unver, abs);
      }
    }
    HashMap<String, String> unverToAbsNew = new HashMap<String, String>();
    for (String newPath : newPaths) {
      String abs = getNewPathFor(pathPrefix + newPath);
      String unver = PathUtility.getPathWithoutVersion(abs);
      if (unver != null) {
        unverToAbsNew.put(unver, abs);
      }
    }
    for (String unver : unverToAbsOld.keySet()) {
      String absOld = unverToAbsOld.get(unver);
      String absNew = unverToAbsNew.get(unver);
      if (absOld != null && absNew != null && !absOld.equals(absNew)) {
        addRenameMapping(absOld, absNew);
      }
    }
  }

  protected void readArchive(ZipFile f, Map<String, ZipEntry> entryMap) throws Exception {
    for (Enumeration<?> en = f.entries(); en.hasMoreElements();) {
      ZipEntry ze = (ZipEntry) en.nextElement();
      entryMap.put(ze.getName(), ze);
    }
  }

  protected void writeEntry(String newName, ZipEntry srcEntry, InputStream in, boolean closeInputStream, ZipOutputStream dst) throws IOException {
    if (!srcEntry.getName().equals(newName)) {
      ZipEntry r = new ZipEntry(newName);
      r.setComment(srcEntry.getComment());
      r.setCrc(srcEntry.getCrc());
      r.setExtra(srcEntry.getExtra());
      r.setMethod(srcEntry.getMethod());
      r.setSize(srcEntry.getSize());
      r.setTime(srcEntry.getTime());
      srcEntry = r;
    }
    srcEntry.setCompressedSize(-1);
    try {
      dst.putNextEntry(srcEntry);
      if (in != null) {
        FileUtility.streamContent(srcEntry.getSize(), in, closeInputStream, dst, false, null);
      }
      dst.closeEntry();
    }
    catch (IOException e) {
      System.err.println("writing entry: " + newName);
      throw e;
    }
  }

}
