/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.update;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.channels.FileLock;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eclipse.update.f2.IUserAgent;
import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.IStreamingListener;
import org.eclipse.update.f2.internal.util.LogUtility;
import org.eclipse.update.f2.internal.util.PathUtility;

public class DefaultUpdateCommitter implements IUpdateCommitter {
  protected final IUserAgent m_ua;
  protected final UpdateContext m_ctx;

  public DefaultUpdateCommitter(IUserAgent ua, UpdateContext ctx) {
    m_ua = ua;
    m_ctx = ctx;
  }

  /**
   * Copy the zip as tmp-*.zip to the destination folder, extract it and if all succeeds rename it to its real zip name.
   */
  @Override
  public void commit() throws Throwable {
    File srcZip = new File(m_ctx.getTempDir(), m_ctx.getNewVersion() + ".zip");
    File dstZip = new File(m_ctx.getInstallRootDir(), m_ctx.getNewVersion() + ".zip");
    File dstZipTemporary = new File(m_ctx.getInstallRootDir(), "tmp-" + m_ctx.getNewVersion() + ".zip");
    checkExist(srcZip);
    checkExist(m_ctx.getInstallRootDir());
    File lockFile = new File(dstZip.getAbsolutePath() + ".lock");
    FileOutputStream lockOut = new FileOutputStream(lockFile);
    FileLock lock = lockOut.getChannel().lock();
    HashMap<File, File> backupFiles = new HashMap<File, File>();
    try {
      if (dstZip.exists()) {
        //nothing to do
        return;
      }
      LogUtility.info("Copy candidate zip to installation folder " + dstZipTemporary);
      dstZipTemporary.delete();
      FileUtility.rmdir(new File(m_ctx.getInstallRootDir(), m_ctx.getNewVersion()));
      IStreamingListener copyListener = new ProgressPhaseStreamingListener(m_ua, ProgressPhase.CommitCopy, srcZip.length());
      FileUtility.copyFile(srcZip, dstZipTemporary, copyListener);
      if (!FileUtility.isMatchingSizeAndCrc(srcZip, dstZipTemporary)) {
        throw new IOException("size/crc mismatch after copying " + srcZip + " to " + dstZipTemporary);
      }
      LogUtility.info("Extract zip");
      //extract only subfolders, no root files
      IStreamingListener extractListener = new ProgressPhaseStreamingListener(m_ua, ProgressPhase.CommitExtract, srcZip.length() * FileUtility.DEFAULT_COMPRESSION_RATIO);
      FileUtility.extractArchiveSubTree(dstZipTemporary, m_ctx.getInstallRootDir(), getPathsToIgnoreExtractErrors(), extractListener);
      //backup old top-level entries except folders, locks, zips
      for (File f : m_ctx.getInstallRootDir().listFiles()) {
        if (!acceptTopLevelBackup(f)) {
          continue;
        }
        File bak = new File(f.getAbsolutePath() + ".bak");
        backupFiles.put(f, bak);
        FileUtility.copyFile(f, bak, null);
      }
      //extract root files
      FileUtility.extractArchiveRoot(dstZipTemporary, m_ctx.getInstallRootDir(), getPathsToIgnoreExtractErrors(), null);
      //delete backup files
      for (Map.Entry<File, File> be : backupFiles.entrySet()) {
        File bak = be.getValue();
        if (bak.exists()) {
          bak.delete();
        }
      }
      //rename dst zip to final name
      dstZipTemporary.renameTo(dstZip);
      removeOldVersions();
    }
    catch (Throwable t) {
      //restore top-level entries
      for (Map.Entry<File, File> be : backupFiles.entrySet()) {
        File f = be.getKey();
        File bak = be.getValue();
        if (bak.exists()) {
          try {
            FileUtility.copyFile(bak, f, null);
          }
          catch (Throwable t2) {
            //nop
          }
          finally {
            bak.delete();
          }
        }
      }
      dstZipTemporary.delete();
      FileUtility.rmdir(new File(m_ctx.getInstallRootDir(), m_ctx.getNewVersion()));
      throw t;
    }
    finally {
      lock.release();
      lockOut.close();
    }
  }

  protected boolean acceptTopLevelBackup(File f) {
    if (f.isHidden()) {
      return false;
    }
    if (f.isDirectory()) {
      return false;
    }
    if (f.isHidden()) {
      return false;
    }
    if (f.getName().endsWith(".lock")) {
      return false;
    }
    if (f.getName().endsWith(".zip")) {
      return false;
    }
    return true;
  }

  /**
   * remove all old versions except the last current version and this new version
   * 
   * @throws IOException
   */
  protected void removeOldVersions() {
    // delete folders & files of old versions
    for (File f : m_ctx.getInstallRootDir().listFiles()) {
      if (f.isHidden()) {
        // do not touch hidden files
        continue;
      }
      if (f.isDirectory()) {
        // directory
        String directoryName = f.getName();
        if (directoryName.equalsIgnoreCase(m_ctx.getOldVersion())) {
          // old version
          continue;
        }
        if (directoryName.equalsIgnoreCase(m_ctx.getNewVersion())) {
          // new version
          continue;
        }
        if (f.equals(m_ctx.getTempDir())) {
          // temp directory
          continue;
        }
        if (!PathUtility.isValidF2DirectoryName(directoryName, m_ctx.getAppName())) {
          // directory not matching f2 structure
          continue;
        }
        FileUtility.rmdir(f);
      }
      else {
        // file
        String filename = f.getName();
        if (filename.equalsIgnoreCase(m_ctx.getOldVersion() + ".zip") || filename.equalsIgnoreCase(m_ctx.getOldVersion() + ".zip.lock")) {
          continue;
        }
        if (filename.equalsIgnoreCase(m_ctx.getNewVersion() + ".zip") || filename.equalsIgnoreCase(m_ctx.getNewVersion() + ".zip.lock")) {
          continue;
        }

        String filenameWithoutExtension = filename;
        if (filename.toLowerCase().endsWith(".zip.lock")) {
          // handle double extension of .zip.lock
          filenameWithoutExtension = filename.substring(0, filename.length() - ".zip.lock".length());
        }
        else {
          int pIndex = filename.lastIndexOf('.');
          if (pIndex > 0) {
            // > 0 to correctly handle files like .hidden, .svn, ...
            filenameWithoutExtension = filename.substring(0, pIndex);
          }
        }

        if (!PathUtility.isValidF2DirectoryName(filenameWithoutExtension, m_ctx.getAppName())) {
          // directory not matching f2 structure
          continue;
        }
        f.delete();
      }
    }
  }

  /**
   * Override to define paths (normally top level paths) that may have extract errors and are to be ignored.
   */
  protected Set<String> getPathsToIgnoreExtractErrors() {
    return null;
  }

  private static void checkExist(File f) throws IOException {
    if (!f.exists()) {
      throw new IOException(f + " does not exist");
    }
  }

}
