/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.update;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.TreeMap;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.IStreamingListener;
import org.eclipse.update.f2.internal.util.PathUtility;

/**
 * Create zip 2 by using zip 1 and a delta zip with rename mappings
 * <p>
 * The delta zip contains the file META-INF/rename-map.txt
 */
public class ZipDeltaUpdater {
  private int m_currentProcessLevel;
  private Map<String, String> m_renameMap;
  private File m_tempDir;

  public ZipDeltaUpdater() {
    m_renameMap = new HashMap<String, String>();
  }

  /**
   * @return true if the two zip are equal and nothing has to be done, otherwise create a delta zip and return false
   *         <p>
   *         The zip output stream is closed automatically
   */
  public void process(File fOld, File fDelta, File fOut, File tmpDir, IStreamingListener listener) throws Exception {
    m_tempDir = tmpDir;
    m_currentProcessLevel = 0;
    ZipFile zDelta = new ZipFile(fDelta);
    try {
      readRenameMappings(zDelta);
      ZipFile zOld = new ZipFile(fOld);
      try {
        processImpl(null, "", zOld, zDelta, fOut, listener);
      }
      finally {
        zOld.close();
      }
    }
    finally {
      zDelta.close();
    }
  }

  public void addRenameMapping(String oldPath, String newPath) {
    m_renameMap.put(oldPath, newPath);
  }

  protected void readRenameMappings(ZipFile fDelta) throws IOException {
    ZipEntry ze = fDelta.getEntry(PathUtility.RENAME_ENTRY_NAME);
    for (String line : new String(FileUtility.readContent(ze.getSize(), fDelta.getInputStream(ze), true), "UTF-8").split("[\\n\\r]+")) {
      if (line.length() > 0 && !line.startsWith("#")) {
        String[] pair = line.split("[\\t]", 2);
        m_renameMap.put(pair[0], pair[1]);
      }
    }
  }

  /**
   * @param referenceEntry
   * @param pathPrefix
   * @param fOld
   * @param fDelta
   * @param fOut
   */
  protected void processImpl(ZipEntry referenceEntry, String pathPrefix, ZipFile fOld, ZipFile fDelta, File fOut, IStreamingListener listener) throws Exception {
    int outEntryCount = 0;
    ZipOutputStream zOut = new ZipOutputStream(new FileOutputStream(fOut));
    try {
      if (m_currentProcessLevel == 0) {
        zOut.setLevel(Deflater.BEST_SPEED);
      }
      else {
        zOut.setLevel(Deflater.BEST_COMPRESSION);
      }
      //read
      TreeMap<String, ZipEntry> entryMapOld = new TreeMap<String, ZipEntry>();
      readArchive(fOld, entryMapOld);
      TreeMap<String, ZipEntry> entryMapDelta = new TreeMap<String, ZipEntry>();
      readArchive(fDelta, entryMapDelta);
      //handle modified and deleted entries
      for (String oldName : new ArrayList<String>(entryMapOld.keySet())) {
        String newName = getNewPathFor(pathPrefix + oldName).substring(pathPrefix.length());
        ZipEntry zOld = entryMapOld.get(oldName);
        ZipEntry zDelta = entryMapDelta.get(newName);
        if (!acceptUpdateEntry(m_currentProcessLevel, pathPrefix + newName)) {
          entryMapDelta.remove(newName);
          continue;
        }
        if (zDelta == null) {
          //unchanged
          outEntryCount++;
          writeEntry(newName, zOld, zOld.isDirectory() ? null : fOld.getInputStream(zOld), true, zOut, listener);
          entryMapDelta.remove(newName);
          continue;
        }
        if (PathUtility.DELETED_COMMENT.equals(zDelta.getComment())) {
          entryMapDelta.remove(newName);
          continue;
        }
        if (PathUtility.DELTA_ARCHIVE_COMMENT.equals(zDelta.getComment())) {
          //delta process archive and then stream out
          File tmpIn1 = File.createTempFile("old", ".zip", m_tempDir);
          File tmpIn2 = File.createTempFile("delta", ".zip", m_tempDir);
          File tmpOut = File.createTempFile("new", ".zip", m_tempDir);
          try {
            m_currentProcessLevel++;
            String subPathPrefix = pathPrefix + newName + (newName.endsWith("/") ? "" : "/");
            FileUtility.streamContent(zOld.getSize(), fOld.getInputStream(zOld), true, new FileOutputStream(tmpIn1), true, null);
            FileUtility.streamContent(zDelta.getSize(), fDelta.getInputStream(zDelta), true, new FileOutputStream(tmpIn2), true, null);
            FileUtility.mkdirs(tmpOut.getParentFile());
            processImpl(zDelta, subPathPrefix, new ZipFile(tmpIn1), new ZipFile(tmpIn2), tmpOut, null);
            zDelta.setComment(null);
            zDelta.setMethod(ZipEntry.DEFLATED);
            zDelta.setSize(tmpOut.length());
            outEntryCount++;
            writeEntry(newName, zDelta, new FileInputStream(tmpOut), true, zOut, listener);
          }
          finally {
            m_currentProcessLevel--;
            if (!tmpIn1.delete()) {
              tmpIn1.deleteOnExit();
            }
            if (!tmpIn2.delete()) {
              tmpIn2.deleteOnExit();
            }
            if (!tmpOut.delete()) {
              tmpOut.deleteOnExit();
            }
          }
          entryMapDelta.remove(newName);
          continue;
        }
        //modified or no delta-archive
        outEntryCount++;
        writeEntry(newName, zDelta, zDelta.isDirectory() ? null : fDelta.getInputStream(zDelta), true, zOut, listener);
        entryMapDelta.remove(newName);
      }
      //add new entries that are only in delta
      Iterator<Map.Entry<String, ZipEntry>> it = entryMapDelta.entrySet().iterator();
      while (it.hasNext()) {
        Map.Entry<String, ZipEntry> e = it.next();
        String newName = e.getKey();
        ZipEntry zDelta = e.getValue();
        if (!acceptUpdateEntry(m_currentProcessLevel, pathPrefix + newName)) {
          it.remove();
          continue;
        }
        //new entry
        outEntryCount++;
        writeEntry(newName, zDelta, zDelta.isDirectory() ? null : fDelta.getInputStream(zDelta), true, zOut, listener);
      }
    }
    finally {
      if (outEntryCount > 0) {
        try {
          zOut.finish();
        }
        catch (Throwable t) {
        }
        try {
          zOut.close();
        }
        catch (Throwable t) {
        }
      }
      try {
        fOld.close();
      }
      catch (Throwable t) {
      }
      try {
        fDelta.close();
      }
      catch (Throwable t) {
      }
    }
  }

  /**
   * default accepts al entries except {@link PathUtility#RENAME_ENTRY_NAME}
   * 
   * @param processLevel
   * @param pathPrefix
   * @param ze
   */
  protected boolean acceptUpdateEntry(int processLevel, String path) {
    if (processLevel == 0 && PathUtility.RENAME_ENTRY_NAME.equals(path)) {
      return false;
    }
    return true;
  }

  protected String getNewPathFor(String oldPath) {
    String[] oldPathSplit = PathUtility.splitLastPart(oldPath);
    if (oldPathSplit[0].length() > 0) {
      //recursion
      String newBasePath = getNewPathFor(oldPathSplit[0]);
      String newPath = m_renameMap.get(newBasePath + oldPathSplit[1]);
      if (newPath == null) {
        newPath = newBasePath + oldPathSplit[1];
      }
      return newPath;
    }
    String newPath = m_renameMap.get(oldPath);
    if (newPath == null) {
      newPath = oldPath;
    }
    return newPath;
  }

  protected void readArchive(ZipFile f, Map<String, ZipEntry> entryMap) throws Exception {
    for (Enumeration<?> en = f.entries(); en.hasMoreElements();) {
      ZipEntry ze = (ZipEntry) en.nextElement();
      entryMap.put(ze.getName(), ze);
    }
  }

  protected void writeEntry(String newName, ZipEntry srcEntry, InputStream in, boolean closeInputStream, ZipOutputStream dst, IStreamingListener listener) throws IOException {
    if (!srcEntry.getName().equals(newName)) {
      ZipEntry r = new ZipEntry(newName);
      r.setComment(srcEntry.getComment());
      r.setCrc(srcEntry.getCrc());
      r.setExtra(srcEntry.getExtra());
      r.setMethod(srcEntry.getMethod());
      r.setSize(srcEntry.getSize());
      r.setTime(srcEntry.getTime());
      srcEntry = r;
    }
    srcEntry.setCompressedSize(-1);
    if (PathUtility.DELETED_COMMENT.equals(srcEntry.getComment()) || PathUtility.DELTA_ARCHIVE_COMMENT.equals(srcEntry.getComment())) {
      srcEntry.setComment(null);
    }
    dst.putNextEntry(srcEntry);
    if (in != null) {
      FileUtility.streamContent(srcEntry.getSize(), in, closeInputStream, dst, false, listener);
    }
    dst.closeEntry();
  }

}
