/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.update.win32;

import java.io.File;
import java.util.Map;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.zip.ZipFile;

import org.eclipse.update.f2.F2Parameter;
import org.eclipse.update.f2.IUserAgent;
import org.eclipse.update.f2.internal.update.AbstractUpdateProcessor;
import org.eclipse.update.f2.internal.update.ProgressPhase;
import org.eclipse.update.f2.internal.update.ProgressPhaseStreamingListener;
import org.eclipse.update.f2.internal.update.UpdateContext;
import org.eclipse.update.f2.internal.update.win32.elev.Win32ElevationCaller;
import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.IStreamingListener;
import org.eclipse.update.f2.internal.util.LogUtility;

/**
 * apply a delta to migrate an old app to the new app
 */
public class Win32UpdateProcessor extends AbstractUpdateProcessor {

  public Win32UpdateProcessor(IUserAgent ua, Map<F2Parameter, String> optionMap) {
    super(ua, optionMap);
  }

  @Override
  protected void commit(UpdateContext ctx) throws Throwable {
    boolean containsUpdateExe = false;
    File archiveFile = new File(ctx.getTempDir(), ctx.getNewVersion() + ".zip");
    ZipFile z = new ZipFile(archiveFile);
    try {
      containsUpdateExe = (z.getEntry(ctx.getNewVersion() + "/update.exe") != null);
    }
    finally {
      z.close();
    }
    String uacOption = getOptionMap().get(F2Parameter.WindowsUAC);
    boolean useUAC = uacOption != null ? "true".equals(uacOption) : containsUpdateExe;
    LogUtility.info("Use admin elevation (UAC): " + useUAC);
    if (useUAC) {
      //extract zip and launch -startup org.eclipse.update.f2_x.x.x.jar to move the zip to the physical installation using admin rights
      File exDir = new File(ctx.getTempDir(), "ex");
      FileUtility.mkdirs(exDir);
      IStreamingListener extractListener = new ProgressPhaseStreamingListener(getUserAgent(), ProgressPhase.CommitCopy, archiveFile.length() * FileUtility.DEFAULT_COMPRESSION_RATIO);
      LogUtility.info("Extracting zip for UAC delegate");
      FileUtility.extractArchiveSubTree(archiveFile, exDir, null, extractListener);
      FileUtility.extractArchiveRoot(archiveFile, exDir, null, null);
      AtomicBoolean callDone = new AtomicBoolean(false);
      LogUtility.info("Launch UAC delegate");
      try {
        //simulate progress while waiting
        emulateProgress(callDone);
        new Win32ElevationCaller().call(ctx, new File(exDir, ctx.getNewVersion()));
      }
      finally {
        callDone.set(true);
      }
    }
    else {
      LogUtility.info("HINT: Not using UAC admin elevation");
      new Win32UpdateCommitter(getUserAgent(), ctx).commit();
    }
  }

  private void emulateProgress(final AtomicBoolean callDone) {
    final IUserAgent ua = getUserAgent();
    if (ua == null) {
      return;
    }
    new Thread() {
      @Override
      public void run() {
        long t = 0;
        while (!callDone.get()) {
          try {
            Thread.sleep(500);
          }
          catch (InterruptedException e) {
            break;
          }
          t += 500;
          ua.setProgressValue(ProgressPhase.CommitExtract.getProgress(t, 20000L));
        }
      }
    }.start();
  }
}
