/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.util;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.logging.Formatter;
import java.util.logging.Handler;
import java.util.logging.Level;
import java.util.logging.LogRecord;
import java.util.logging.Logger;
import java.util.logging.SimpleFormatter;

import org.eclipse.update.f2.F2Updater;

/**
 * Utility class for handling logging with "jul" {@link Logger}
 * <p>
 * The command line option -verbose activates info level.
 * 
 * @author BSI AG
 * @since 1.0
 */
public class LogUtility {
  private static final Logger LOG = Logger.getLogger(F2Updater.PLUGIN_ID);

  private LogUtility() {
  }

  public static void setVerbose() {
    LOG.setLevel(Level.ALL);
    //use (better human readable) single line formatter
    for (Handler h : Logger.getLogger("").getHandlers()) {
      if (h.getFormatter() instanceof SimpleFormatter) {
        h.setFormatter(new SingleLineFormatter());
      }
    }
  }

  public static void setSilent() {
    LOG.setLevel(Level.SEVERE);
    //use (better human readable) single line formatter
    for (Handler h : Logger.getLogger("").getHandlers()) {
      if (h.getFormatter() instanceof SimpleFormatter) {
        h.setFormatter(new SingleLineFormatter());
      }
    }
  }

  public static void info(String s) {
    LOG.log(Level.INFO, s);
  }

  public static void warn(String s, Throwable t) {
    if (t == null) {
      LOG.log(Level.WARNING, s);
    }
    else {
      LOG.log(Level.WARNING, s, t);
    }
  }

  public static void error(String s, Throwable t) {
    if (t == null) {
      LOG.log(Level.SEVERE, s);
    }
    else {
      LOG.log(Level.SEVERE, s, t);
    }
  }

  private static class SingleLineFormatter extends Formatter {
    @Override
    public synchronized String format(LogRecord record) {
      StringBuilder sb = new StringBuilder();
      sb.append(record.getLevel().getLocalizedName());
      sb.append(": ");
      sb.append(formatMessage(record));
      if (record.getThrown() != null) {
        sb.append("\n");
        try {
          StringWriter sw = new StringWriter();
          PrintWriter pw = new PrintWriter(sw);
          record.getThrown().printStackTrace(pw);
          pw.close();
          sb.append(sw.toString());
        }
        catch (Exception ex) {
          //nop
        }
      }
      sb.append("\n");
      return sb.toString();
    }
  }
}
