/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.create.linux;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.eclipse.update.f2.F2Parameter;
import org.eclipse.update.f2.internal.create.AbstractCreateProcessor;
import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.LogUtility;
import org.eclipse.update.f2.internal.util.PathUtility;

/**
 * Create F2 Updatesite
 */
public class LinuxCreateProcessor extends AbstractCreateProcessor {
  private static final Pattern JAVA_PATH_PATTERN = Pattern.compile("((.*/)?jre.*/bin/java)", Pattern.CASE_INSENSITIVE);

  public LinuxCreateProcessor(Map<F2Parameter, String> optionMap) {
    super(optionMap);
  }

  /**
   * Linux
   * <p>
   * Create a top level ini and launcher if missing.
   */
  @Override
  protected void autoCompleteFullVersionZipStructure(File f, String versionFolderName) throws IOException {
    LogUtility.info("verifying content of " + f.getName());
    String[] requiredFiles = new String[]{
        m_appName,
        m_appName + ".ini",
    };
    File t = new File(f.getAbsolutePath() + ".tmp");
    ZipFile z = new ZipFile(f);
    try {
      boolean complete = true;
      for (String s : requiredFiles) {
        if (z.getEntry(s) == null) {
          complete = false;
          break;
        }
      }
      if (complete) {
        return;
      }
      ZipOutputStream zOut = new ZipOutputStream(new FileOutputStream(t));
      zOut.setLevel(Deflater.BEST_COMPRESSION);
      try {
        //copy existing entries
        for (Enumeration<? extends ZipEntry> en = z.entries(); en.hasMoreElements();) {
          ZipEntry e = en.nextElement();
          zOut.putNextEntry(new ZipEntry(e.getName()));
          if (!e.isDirectory()) {
            FileUtility.streamContent(e.getSize(), z.getInputStream(e), true, zOut, false, null);
          }
          zOut.closeEntry();
        }
        //app.exe
        if (z.getEntry(requiredFiles[0]) == null) {
          LogUtility.info(" adding " + requiredFiles[0]);
          ZipEntry innerExe = z.getEntry(versionFolderName + "/" + m_appName);
          zOut.putNextEntry(new ZipEntry(requiredFiles[0]));
          FileUtility.streamContent(innerExe.getSize(), z.getInputStream(innerExe), true, zOut, false, null);
          zOut.closeEntry();
        }
        //app.ini
        if (z.getEntry(requiredFiles[1]) == null) {
          LogUtility.info(" adding " + requiredFiles[1]);
          ZipEntry innerIni = z.getEntry(versionFolderName + "/" + m_appName + ".ini");
          zOut.putNextEntry(new ZipEntry(requiredFiles[1]));
          byte[] data = createRootAppIni(z, versionFolderName, FileUtility.readContent(innerIni.getSize(), z.getInputStream(innerIni), true));
          zOut.write(data);
          zOut.closeEntry();
        }
      }
      finally {
        zOut.close();
      }
    }
    finally {
      z.close();
    }
    f.delete();
    t.renameTo(f);
  }

  /**
   * Ensure the following entries and qualify directory paths with sub dir name
   * 
   * <pre>
   * --launcher.library
   * 1.0.0/plugins/org.eclipse.equinox.launcher.gtk.linux.x86_1.1.2.R36x_v20101222
   * -startup
   * 1.0.0/plugins/org.eclipse.equinox.launcher_1.1.1.R36x_v20101122_1400.jar
   * -vm
   * 1.0.0/jre_1.6.0.26/bin/java
   * </pre>
   */
  protected byte[] createRootAppIni(ZipFile z, String versionFolderName, byte[] innerIni) throws IOException {
    //read ini file
    String startupJar = null;
    String vm = null;
    String launcherLibrary = null;
    List<String> args = new ArrayList<String>(Arrays.asList(new String(innerIni, "UTF-8").split("\\s+")));
    //add missing entries (in reverse order and adding to front of list)
    for (Enumeration<?> en = z.entries(); en.hasMoreElements();) {
      ZipEntry ze = (ZipEntry) en.nextElement();
      Matcher m;
      if ((m = JAVA_PATH_PATTERN.matcher(ze.getName())).matches()) {
        vm = m.group(1);
      }
      else if ((m = PathUtility.LAUNCHER_JAR_PATH_PATTERN.matcher(ze.getName())).matches()) {
        startupJar = m.group(1);
      }
      else if ((m = PathUtility.LAUNCHER_LIBRARY_PATH_PATTERN.matcher(ze.getName())).matches()) {
        launcherLibrary = m.group(1);
      }
    }
    //adapt paths to launcher specific paths
    for (int i = 0; i < args.size(); i++) {
      String s = args.get(i);
      if ("--launcher.library".equals(s) || "-startup".equals(s) || "-vm".equals(s)) {
        i++;
        if (!args.get(i).startsWith("/")) {
          args.set(i, versionFolderName + "/" + args.get(i));
        }
      }
      else if (s != null && s.startsWith("-Djava.util.logging.config.file=")) {
        String[] parts = s.split("=", 2);
        if (parts.length > 1) {
          args.set(i, parts[0] + "=" + versionFolderName + "/" + parts[1]);
        }
      }
    }
    //add everything else than vmargs BEFORE vmargs, vmargs must be at END!
    if (!args.contains("-vm")) {
      if (vm == null) {
        throw new IOException("could not find a jre in " + z.getName());
      }
      args.add(0, "-vm");
      args.add(1, vm);
    }
    if (!args.contains("-startup")) {
      if (startupJar == null) {
        throw new IOException("could not find a startup jar in " + z.getName());
      }
      args.add(0, "-startup");
      args.add(1, startupJar);
    }
    if (!args.contains("--launcher.library")) {
      if (launcherLibrary == null) {
        throw new IOException("could not find a launcher.library in " + z.getName());
      }
      args.add(0, "--launcher.library");
      args.add(1, launcherLibrary);
    }
    //
    StringBuilder buf = new StringBuilder();
    for (String s : args) {
      buf.append(s);
      buf.append("\n");
    }
    String rootIniContent = buf.toString().trim();
    //create new root ini file
    return rootIniContent.getBytes("UTF-8");
  }

}
