/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.create;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.eclipse.update.f2.F2Parameter;
import org.eclipse.update.f2.internal.DeltaDesc;
import org.eclipse.update.f2.internal.SiteDesc;
import org.eclipse.update.f2.internal.Version3Q;
import org.eclipse.update.f2.internal.VersionDesc;
import org.eclipse.update.f2.internal.update.ZipDeltaUpdater;
import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.LogUtility;

/**
 * Create F2 Updatesite
 */
public abstract class AbstractCreateProcessor implements ICreateProcessor {
  protected final File m_siteDir;
  protected final String m_appName;
  private final File m_tmpDir;
  private Map<F2Parameter, String> m_optionMap;

  public AbstractCreateProcessor(Map<F2Parameter, String> argMap) {
    m_optionMap = argMap != null ? argMap : new HashMap<F2Parameter, String>();
    File siteRoot = new File(m_optionMap.get(F2Parameter.SiteDirectory));
    String os = m_optionMap.get(F2Parameter.OS);
    String arch = m_optionMap.get(F2Parameter.Arch);
    m_siteDir = new File(siteRoot, os + (arch != null ? "/" + arch : ""));
    m_appName = m_optionMap.get(F2Parameter.Name);
    String tmpArg = m_optionMap.get(F2Parameter.TempDirectory);
    m_tmpDir = (tmpArg != null ? new File(tmpArg) : new File(m_siteDir, "_tmp"));
    if (!m_siteDir.exists()) {
      throw new IllegalArgumentException("site directory does not exist: " + m_siteDir);
    }
    if (m_appName == null) {
      throw new IllegalArgumentException("name is null");
    }
  }

  protected final Map<F2Parameter, String> getOptionMap() {
    return m_optionMap;
  }

  protected final File getTmpDir() {
    return m_tmpDir;
  }

  @Override
  public final void create() throws Exception {
    try {
      FileUtility.rmdir(getTmpDir());
      getTmpDir().mkdirs();
      createInternal();
    }
    finally {
      FileUtility.rmdir(getTmpDir());
    }
  }

  protected void createInternal() throws Exception {
    boolean changed = false;

    //check validity of full versions and deltas
    SiteDesc desc = readExistingSiteDesc();

    //cleanup full version files
    HashSet<String> validVersionNames = new HashSet<String>();
    TreeMap<Version3Q, VersionDesc> allVersionDescMap = new TreeMap<Version3Q, VersionDesc>();
    for (File f : m_siteDir.listFiles()) {
      if (f.isDirectory() || !f.getName().endsWith(".zip")) {
        continue;
      }
      String fname = f.getName();
      if (isDeltaName(fname)) {
        try {
          //check naming pattern
          DeltaDesc deltaDesc = new DeltaDesc(fname, 0, 0);
          if (!deltaDesc.getOldVersion().startsWith(m_appName + "_")) {
            throw new IllegalArgumentException("delta file name must contain the application name as prefixes (" + m_appName + ")");
          }
          if (!deltaDesc.getNewVersion().startsWith(m_appName + "_")) {
            throw new IllegalArgumentException("delta file name must contain the application name as prefixes (" + m_appName + ")");
          }
        }
        catch (Throwable t) {
          LogUtility.warn(t + ". Renaming it to " + fname + ".err", null);
          f.renameTo(new File(m_siteDir, fname + ".err"));
        }
        continue;
      }
      VersionDesc versionDesc;
      try {
        //check naming pattern
        versionDesc = new VersionDesc(fname, 0, 0, 0);
        if (!versionDesc.getVersion().startsWith(m_appName + "_")) {
          throw new IllegalArgumentException("version file name must contain the application name as prefix (" + m_appName + ")");
        }
      }
      catch (Throwable t) {
        LogUtility.warn(t + ". Renaming it to " + fname + ".err", null);
        f.renameTo(new File(m_siteDir, fname + ".err"));
        continue;
      }

      //check if full-version zip starts with simpleName folder
      if (!checkFullVersionZipStructure(f, versionDesc.getVersion() + "/")) {
        LogUtility.warn(fname + " is not a valid full-version zip, it does not have root folder '" + versionDesc.getVersion() + "'. Renaming it to " + fname + ".err", null);
        f.renameTo(new File(m_siteDir, fname + ".err"));
        continue;
      }

      //ensure that the full-version zip contains the required root files (ini and executable)
      autoCompleteFullVersionZipStructure(f, versionDesc.getVersion());
      //
      if (checkIntegrity(desc, f)) {
        validVersionNames.add(versionDesc.getVersion());
      }
      else {
        LogUtility.info(fname + " is not matching size/crc/content-hash of f2.txt. Invalidating it.");
        changed = true;
      }
      allVersionDescMap.put(versionDesc.getVersion3Q(), versionDesc);
    }

    //remove old application versions
    String numberOfVersionsToKeepString = getOptionMap().get(F2Parameter.NumberOfVersionsToKeep);
    if (numberOfVersionsToKeepString != null) {
      int numberOfVersionsToKeep = 0;
      try {
        numberOfVersionsToKeep = Integer.parseInt(numberOfVersionsToKeepString);
      }
      catch (NumberFormatException e) {
        LogUtility.warn("Failed to parse " + F2Parameter.NumberOfVersionsToKeep.name() + " parameter", e);
      }

      if (numberOfVersionsToKeep > 0) {
        int numberOfVersionsToRemove = allVersionDescMap.size() - numberOfVersionsToKeep;

        if (numberOfVersionsToRemove > 0) {
          // keep only the newest 'numberOfVersionsToKeep' application versions

          LogUtility.info("keeping " + numberOfVersionsToKeep + " out of " + allVersionDescMap.size() + " versions, thus removing the " + numberOfVersionsToRemove + " oldest ones.");
          Iterator<VersionDesc> it = allVersionDescMap.values().iterator();
          while (it.hasNext()) {
            if (numberOfVersionsToRemove == 0) {
              break;
            }

            // remove version
            VersionDesc versionDesc = it.next();
            if (!(new File(m_siteDir, versionDesc.getFileName())).delete()) {
              LogUtility.warn("failed to remove old application version file " + versionDesc.getFileName(), null);
            }
            else {
              LogUtility.info("successfully removed old application version file " + versionDesc.getFileName());
            }
            validVersionNames.remove(versionDesc.getVersion());
            it.remove();
            numberOfVersionsToRemove--;
          }
        }
        else {
          // more versions are to be removed than exists
          LogUtility.info("keeping maximum " + numberOfVersionsToKeep + " out of " + allVersionDescMap.size() + " versions, thus keeping all.");
        }
      }
    }

    //cleanup delta file structure
    for (File f : m_siteDir.listFiles()) {
      if (f.isDirectory() || !f.getName().endsWith(".zip")) {
        continue;
      }
      String fname = f.getName();
      if (!isDeltaName(fname)) {
        continue;
      }
      DeltaDesc deltaDesc = new DeltaDesc(fname, 0, 0);
      if (!validVersionNames.contains(deltaDesc.getOldVersion()) || !validVersionNames.contains(deltaDesc.getNewVersion())) {
        LogUtility.info(fname + " is not referring to a valid version. Deleting it.");
        f.delete();
      }
      else if (!checkIntegrity(desc, f)) {
        LogUtility.info(fname + " is not up to date. Deleting it.");
        f.delete();
      }
    }

    //check if files in existing f2.txt really exist
    if (!changed) {
      for (VersionDesc exDesc : desc.getVersionList()) {
        if (!new File(m_siteDir, exDesc.getFileName()).exists()) {
          changed = true;
          break;
        }
      }
    }
    if (!changed) {
      for (DeltaDesc exDesc : desc.getDeltaList()) {
        if (!new File(m_siteDir, exDesc.getFileName()).exists()) {
          changed = true;
          break;
        }
      }
    }

    //create missing deltas
    try {
      ArrayList<VersionDesc> allVersionDescList = new ArrayList<VersionDesc>(allVersionDescMap.values());
      for (int i = 0; i < allVersionDescList.size() - 1; i++) {
        String v1 = allVersionDescList.get(i).getVersion();
        String v2 = allVersionDescList.get(i + 1).getVersion();
        String nameDelta = "delta$" + v1 + "$" + v2;
        File deltaFile = new File(m_siteDir, nameDelta + ".zip");
        if (deltaFile.exists()) {
          LogUtility.info(nameDelta + " is up-to-date.");
          continue;
        }
        changed = true;
        createDelta(v1, v2, nameDelta);
        testDelta(v1, v2, nameDelta);
      }
    }
    finally {
      FileUtility.rmdir(m_tmpDir);
    }
    //create f2.txt
    if (changed) {
      LogUtility.info("Create f2.txt");
      createF2Txt();
    }
    LogUtility.info("Done");
  }

  private boolean isDeltaName(String name) {
    return name.startsWith("delta$");
  }

  private SiteDesc readExistingSiteDesc() {
    File f = new File(m_siteDir, "f2.txt");
    if (f.exists()) {
      try {
        return SiteDesc.parse(new String(FileUtility.readContent(f.length(), new FileInputStream(f), true), "UTF-8"));
      }
      catch (Throwable t) {
        LogUtility.info("Failed reading existing " + f.getName() + ": " + t);
        //nop
      }
    }
    return null;
  }

  /**
   * The only top level folder is topLevelFolder
   */
  protected boolean checkFullVersionZipStructure(File f, String topLevelFolder) throws IOException {
    ZipFile z = new ZipFile(f);
    try {
      for (Enumeration<?> en = z.entries(); en.hasMoreElements();) {
        ZipEntry e = (ZipEntry) en.nextElement();
        if (e.isDirectory()) {
          String path = e.getName();
          if (!path.startsWith(topLevelFolder)) {
            return false;
          }
        }
      }
      return true;
    }
    finally {
      z.close();
    }
  }

  /**
   * Ensure that the full-version zip contains the required root files (ini and executables)
   * 
   * @param f
   * @param versionFolderName
   *          is something like app_1.0.0
   */
  protected abstract void autoCompleteFullVersionZipStructure(File f, String versionFolderName) throws IOException;

  private boolean checkIntegrity(SiteDesc siteDesc, File f) throws IOException {
    if (siteDesc == null) {
      return false;
    }
    for (VersionDesc desc : siteDesc.getVersionList()) {
      if (desc.getFileName().equals(f.getName())) {
        return FileUtility.isMatchingSizeAndCrc(f, desc.getSize(), desc.getCrc()) && FileUtility.archiveHash(f) == desc.getContentHash();
      }
    }
    for (DeltaDesc desc : siteDesc.getDeltaList()) {
      if (desc.getFileName().equals(f.getName())) {
        return FileUtility.isMatchingSizeAndCrc(f, desc.getSize(), desc.getCrc());
      }
    }
    return false;
  }

  private void createDelta(String nameOld, String nameNew, String nameDelta) throws Exception {
    LogUtility.info("Create delta from " + nameOld + " to " + nameNew);
    File fOld = new File(m_siteDir, nameOld + ".zip");
    File fNew = new File(m_siteDir, nameNew + ".zip");
    File fOut = new File(m_siteDir, nameDelta + ".zip");
    ZipDeltaBuilder builder = new ZipDeltaBuilder(1, 0);
    builder.addRenameMapping(nameOld + "/", nameNew + "/");
    boolean equal = builder.process(fOld, fNew, fOut, m_tmpDir);
    if (equal) {
      throw new IllegalStateException("delta files should never be equal!");
    }
  }

  private void createF2Txt() throws Exception {
    TreeMap<Version3Q, VersionDesc> versions = new TreeMap<Version3Q, VersionDesc>();
    TreeMap<String, DeltaDesc> deltas = new TreeMap<String, DeltaDesc>();
    for (File f : m_siteDir.listFiles()) {
      if (!f.getName().endsWith(".zip")) {
        continue;
      }
      if (isDeltaName(f.getName())) {
        DeltaDesc desc = new DeltaDesc(f.getName(), f.length(), FileUtility.crc32(f));
        deltas.put(f.getName(), desc);
      }
      else {
        VersionDesc desc = new VersionDesc(f.getName(), f.length(), FileUtility.crc32(f), FileUtility.archiveHash(f));
        versions.put(desc.getVersion3Q(), desc);
      }
    }
    String content = new SiteDesc(versions.values(), deltas.values()).getContent();
    FileUtility.writeContent(new FileOutputStream(new File(m_siteDir, "f2.txt")), true, content.getBytes("UTF-8"), null);
  }

  private void testDelta(String nameOld, String nameNew, String nameDelta) throws Exception {
    LogUtility.info("TEST: verify delta " + nameDelta);
    String nameNewUpdated = "test-update-" + nameOld + "-" + nameNew;
    testUpdate(nameOld, nameDelta, nameNewUpdated);
    testCompare(nameNew, nameNewUpdated);
    new File(m_siteDir, nameNewUpdated + ".zip").delete();
  }

  private void testUpdate(String nameOld, String nameDelta, String nameNew) throws Exception {
    LogUtility.info("TEST: simulating update from " + nameOld + " to " + nameNew);
    File fOld = new File(m_siteDir, nameOld + ".zip");
    File fDelta = new File(m_siteDir, nameDelta + ".zip");
    File fOut = new File(m_siteDir, nameNew + ".zip");
    ZipDeltaUpdater builder = new ZipDeltaUpdater();
    builder.process(fOld, fDelta, fOut, m_tmpDir, null);
  }

  private void testCompare(String nameNew, String nameNewUpdated) throws Exception {
    LogUtility.info("TEST: comparing zip hashes");
    File f1 = new File(m_siteDir, nameNew + ".zip");
    File f2 = new File(m_siteDir, nameNewUpdated + ".zip");
    long hash1 = FileUtility.archiveHash(new FileInputStream(f1), true);
    long hash2 = FileUtility.archiveHash(new FileInputStream(f2), true);
    if (hash1 == hash2) {
      LogUtility.info("TEST: OK");
      return;
    }
    Set<String> diff = FileUtility.archiveCompare(new FileInputStream(f1), true, new FileInputStream(f2), true);
    StringBuilder buf = new StringBuilder();
    for (String s : diff) {
      buf.append(" " + s + "\n");
    }
    LogUtility.info("TEST: FAILED\n" + buf);
    throw new Exception("archives " + nameNew + " and " + nameNewUpdated + " are not equal");
  }

}
