/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.util;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.HttpURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.security.MessageDigest;
import java.util.Enumeration;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;
import java.util.TreeSet;
import java.util.concurrent.atomic.AtomicReference;
import java.util.zip.CRC32;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;

import org.eclipse.update.f2.IUserAgent;

/**
 * Utility class for managing directories and files
 * <p>
 * Set the system property 'java.io.tmpdir' in order to define the base temp directory.
 * 
 * @author BSI AG
 * @since 1.0
 */
public class FileUtility {
  public static final long DEFAULT_COMPRESSION_RATIO = 3;

  private FileUtility() {
  }

  public static void mkdirs(File dir) {
    if (dir != null && !dir.exists()) {
      dir.mkdirs();
    }
  }

  public static boolean rmdir(File dir) {
    if (dir == null) {
      return true;
    }
    if (dir.exists()) {
      if (dir.isDirectory()) {
        File[] files = dir.listFiles();
        for (int i = 0; files != null && i < files.length; i++) {
          if (files[i].isDirectory()) {
            rmdir(files[i]);
          }
          else {
            files[i].delete();
          }
        }
      }
    }
    return dir.delete();
  }

  public static File createTempDir(String prefix, String suffix) throws IOException {
    File f = File.createTempFile(prefix, suffix);
    f.delete();
    f.mkdirs();
    return f;
  }

  public static byte[] readContent(long size, InputStream in, boolean closeInputStream) throws IOException {
    ByteArrayOutputStream out = new ByteArrayOutputStream();
    streamContent(size, in, closeInputStream, out, true, null);
    return out.toByteArray();
  }

  public static void writeContent(OutputStream out, boolean closeOutputStream, byte[] content, IStreamingListener listener) throws IOException {
    streamContent(content.length, new ByteArrayInputStream(content), true, out, closeOutputStream, listener);
  }

  public static long streamContent(long size, InputStream in, boolean closeInputStream, OutputStream out, boolean closeOutputStream, IStreamingListener listener) throws IOException {
    try {
      if (size == 0L) {
        return 0;
      }
      if (size > 0) {
        int consumed = 0;
        byte[] buffer = new byte[8192];
        while (consumed < size) {
          int n = in.read(buffer, 0, Math.min(buffer.length, (int) (size - consumed)));
          if (n < 0) {
            break;
          }
          if (n > 0) {
            out.write(buffer, 0, n);
          }
          consumed += n;
          if (listener != null) {
            listener.streamed(n);
          }
        }
        return size;
      }
      //unknown size
      byte[] buffer = new byte[8192];
      size = 0;
      int n;
      while ((n = in.read(buffer)) > 0) {
        out.write(buffer, 0, n);
        size += n;
      }
      return size;
    }
    finally {
      if (closeInputStream) {
        try {
          in.close();
        }
        catch (Throwable t) {
          //nop
        }
      }
      if (closeOutputStream) {
        try {
          out.close();
        }
        catch (Throwable t) {
          //nop
        }
      }
    }
  }

  /**
   * This function copies a file from a remote location to the local filesystem
   * 
   * @param source
   *          location of the remote file to copy
   * @param destination
   *          destination of the file
   * @return <code>File</code> object of the destination file
   */
  public static void download(URL source, OutputStream out, long size, IUserAgent ua, IStreamingListener listener) throws IOException {
    LogUtility.info("Downloading " + source);
    InputStream is = null;
    URLConnection conn = null;
    int tryCount = 3;
    NameCallback cbName = new NameCallback("Username");
    PasswordCallback cbPass = new PasswordCallback("Password", false);
    AtomicReference<String> authMethodRef = new AtomicReference<String>("BASIC");
    ua.preAuthenticate(authMethodRef, cbName, cbPass);
    for (int i = 1; i <= tryCount && is == null; i++) {
      try {
        conn = source.openConnection();
        if (cbName.getName() != null && cbPass.getPassword() != null) {
          conn.setRequestProperty("Authorization", authMethodRef.get() + " " + Base64Utility.encode((cbName.getName() + ":" + new String(cbPass.getPassword())).getBytes("UTF-8")));
        }
        is = new BufferedInputStream(conn.getInputStream());
      }
      catch (IOException e) {
        LogUtility.warn("Downloading " + source + " failed " + i + " of " + tryCount, null);
        if (i == tryCount) {
          throw e;
        }
        //detect auth method
        if (conn != null && (conn instanceof HttpURLConnection) && ((HttpURLConnection) conn).getResponseCode() == HttpURLConnection.HTTP_UNAUTHORIZED) {
          String www = conn.getHeaderField("WWW-Authenticate");
          if (www != null) {
            String[] a = www.split(" ", 2);
            if (a[0].length() > 0) {
              authMethodRef.set(a[0]);
            }
          }
          ua.authenticate(authMethodRef, cbName, cbPass);
        }
        else {
          throw e;
        }
      }
    }
    streamContent(size, is, true, out, true, listener);
  }

  /**
   * @param srcDir
   * @param pathPrefix
   *          is "" if the srcDir is to be archived as / or for example foo/ if the src dir is to be mapped as foo/
   * @param archiveFile
   * @param deflaterLevel
   */
  public static void compressArchive(File srcDir, String pathPrefix, File archiveFile, int deflaterLevel) throws Exception {
    if (pathPrefix.length() > 0 && !pathPrefix.endsWith("/")) {
      throw new IllegalArgumentException("pathPrefix must be empty or end with a /");
    }
    ZipOutputStream zOut = new ZipOutputStream(new FileOutputStream(archiveFile));
    try {
      zOut.setLevel(deflaterLevel);
      archiveFile.delete();
      for (File f : srcDir.listFiles()) {
        addFileToZip(pathPrefix, f, zOut);
      }
    }
    finally {
      try {
        zOut.finish();
      }
      catch (Throwable t) {
      }
      try {
        zOut.close();
      }
      catch (Throwable t) {
      }
    }
  }

  private static void addFileToZip(String pathPrefix, File f, ZipOutputStream zOut) throws Exception {
    if (f.exists() && !f.isHidden() && !f.getName().startsWith(".svn")) {
      if (f.isDirectory()) {
        String path = pathPrefix + f.getName() + "/";
        addZipEntry(zOut, path, null);
        addFolderToZip(path, f, zOut);
      }
      else {
        String path = pathPrefix + f.getName();
        byte[] data = readContent(f.length(), new FileInputStream(f), true);
        addZipEntry(zOut, path, data);
      }
    }
  }

  private static void addFolderToZip(String pathPrefix, File dir, ZipOutputStream zOut) throws Exception {
    if ((!dir.exists()) || (!dir.isDirectory())) {
      throw new IOException("source directory " + dir + " does not exist or is not a folder");
    }
    for (File f : dir.listFiles()) {
      addFileToZip(pathPrefix, f, zOut);
    }
  }

  private static void addZipEntry(ZipOutputStream zOut, String path, byte[] data) throws Exception {
    zOut.putNextEntry(new ZipEntry(path));
    if (data != null) {
      zOut.write(data);
    }
    zOut.closeEntry();
  }

  /**
   * Extract root files but not subfolders
   */
  public static void extractArchiveRoot(File archiveFile, File destinationDir, Set<String> pathsToIgnoreExtractErrors, IStreamingListener listener) throws IOException {
    FileUtility.mkdirs(destinationDir);
    ZipFile zip = new ZipFile(archiveFile);
    try {
      Enumeration<? extends ZipEntry> entries = zip.entries();
      while (entries.hasMoreElements()) {
        ZipEntry ze = entries.nextElement();
        String path = ze.getName();
        if (ze.isDirectory() || path.indexOf("/") >= 0) {
          // if it's a directory or subfolder entry then ignore it
          continue;
        }
        File f = new File(destinationDir, path);
        if (pathsToIgnoreExtractErrors != null && pathsToIgnoreExtractErrors.contains(path)) {
          try {
            streamContent(ze.getSize(), zip.getInputStream(ze), true, new FileOutputStream(f), true, listener);
          }
          catch (Throwable t) {
          }
        }
        else {
          streamContent(ze.getSize(), zip.getInputStream(ze), true, new FileOutputStream(f), true, listener);
        }
      }
    }
    finally {
      try {
        zip.close();
      }
      catch (Throwable t) {
      }
    }
  }

  /**
   * Extract subfolders but not the root files
   */
  public static void extractArchiveSubTree(File archiveFile, File destinationDir, Set<String> pathsToIgnoreExtractErrors, IStreamingListener listener) throws IOException {
    FileUtility.mkdirs(destinationDir);
    ZipFile zip = new ZipFile(archiveFile);
    try {
      Enumeration<? extends ZipEntry> entries = zip.entries();
      while (entries.hasMoreElements()) {
        ZipEntry ze = entries.nextElement();
        String path = ze.getName();
        if (path.indexOf("/") < 0) {
          // if it's a root entry then ignore it
          continue;
        }
        File f = new File(destinationDir, path);
        if (ze.isDirectory()) { // if it's a directory, create it
          FileUtility.mkdirs(f);
          continue;
        }
        FileUtility.mkdirs(f.getParentFile());
        if (pathsToIgnoreExtractErrors != null && pathsToIgnoreExtractErrors.contains(path)) {
          try {
            streamContent(ze.getSize(), zip.getInputStream(ze), true, new FileOutputStream(f), true, listener);
          }
          catch (Throwable t) {
          }
        }
        else {
          streamContent(ze.getSize(), zip.getInputStream(ze), true, new FileOutputStream(f), true, listener);
        }
      }
    }
    finally {
      try {
        zip.close();
      }
      catch (Throwable t) {
      }
    }
  }

  /**
   * Deep structure content compare (also including jars and zips in all depths) of two archives with regard to crc and
   * size. Uses {@link #archiveHash(File, File)}.
   * 
   * @return true if both archives have the same hash
   */
  public static boolean archiveEquals(InputStream archive1, boolean closeArchive1, InputStream archive2, boolean closeArchive2) throws IOException {
    return archiveHash(archive1, closeArchive1) == archiveHash(archive2, closeArchive2);
  }

  /**
   * Deep structure content hash (also including jars and zips in all depths) of an archives with regard to crc and
   * size. Uses {@link #listArchiveContentForHash(File)}.
   * 
   * @return hash of all file entries (including sub archives (zip, jar) recursively).
   *         The hash is built by sorting the records by name ascending.
   *         Then the crc is calculated on name (byte[]), size (8 bytes LE), crc (8 bytes LE).
   *         Directories (foo/) are completely ignored since they appear in some zips but not in others.
   */
  public static long archiveHash(File f) throws IOException {
    return archiveHash(new FileInputStream(f), true);
  }

  /**
   * see {@link #archiveHash(File)}
   */
  public static long archiveHash(InputStream archive, boolean closeArchive) throws IOException {
    TreeMap<String, long[]> sortMap = getSortedContentMap(archive, closeArchive);
    CRC32 crc = new CRC32();
    for (Map.Entry<String, long[]> e : sortMap.entrySet()) {
      crc.update(e.getKey().getBytes("UTF-8"));
      long n = e.getValue()[0];
      for (int i = 0; i < 8; i++) {
        crc.update((int) (n & 0xff));
        n = n >>> 8;
      }
      n = e.getValue()[1];
      for (int i = 0; i < 8; i++) {
        crc.update((int) (n & 0xff));
        n = n >>> 8;
      }
    }
    return crc.getValue();
  }

  /**
   * Content deep structure compare (also including jars and zips in all depths) of two archives with regard to crc and
   * size.
   * 
   * @return a set with description of differences or an empty set when equal
   */
  public static Set<String> archiveCompare(InputStream archive1, boolean closeArchive1, InputStream archive2, boolean closeArchive2) throws IOException {
    TreeMap<String, long[]> map1 = getSortedContentMap(archive1, closeArchive1);
    TreeMap<String, long[]> map2 = getSortedContentMap(archive2, closeArchive2);
    TreeSet<String> deltaSet = new TreeSet<String>();
    for (Map.Entry<String, long[]> e : map1.entrySet()) {
      String name = e.getKey();
      long[] z1 = e.getValue();
      long[] z2 = map2.get(name);
      if (z2 == null) {
        deltaSet.add("Left only: " + name + "\t" + z1[0] + "\t" + z1[1] + "\t0\t0");
        continue;
      }
      if (z1[0] != z2[0] || z1[1] != z2[1]) {
        deltaSet.add("Different: " + name + "\t" + z1[0] + "\t" + z1[1] + "\t" + z2[0] + "\t" + z2[1]);
        continue;
      }
    }
    for (Map.Entry<String, long[]> e : map2.entrySet()) {
      String name = e.getKey();
      long[] z2 = e.getValue();
      long[] z1 = map1.get(name);
      if (z1 == null) {
        deltaSet.add("Right only:" + name + "\t0\t0" + "\t" + z2[0] + "\t" + z2[1]);
      }
    }
    return deltaSet;
  }

  private static TreeMap<String, long[]> getSortedContentMap(InputStream in, boolean closeInputStream) throws IOException {
    final TreeMap<String, long[]> sortMap = new TreeMap<String, long[]>();
    visitArchiveContentRec("", new ZipInputStream(in), closeInputStream, new IZipVisitor() {
      @Override
      public void visit(ZipInputStream zin, String pathPrefix, ZipEntry e) throws IOException {
        if (!e.isDirectory()) {
          sortMap.put(pathPrefix + e.getName(), sizeAndCrc32(zin, e));
        }
      }
    });
    return sortMap;
  }

  private static void visitArchiveContentRec(String pathPrefix, ZipInputStream zin, boolean closeInputStream, IZipVisitor visitor) throws IOException {
    try {
      ZipEntry ze;
      while ((ze = zin.getNextEntry()) != null) {
        if (PathUtility.isArchiveEntry(ze)) {
          ZipInputStream zinInner = new ZipInputStream(zin);
          visitArchiveContentRec(pathPrefix + ze.getName() + "/", zinInner, false, visitor);
        }
        else {
          visitor.visit(zin, pathPrefix, ze);
        }
        zin.closeEntry();
      }
    }
    finally {
      if (closeInputStream) {
        zin.close();
      }
    }
  }

  /**
   * Copies one file to another. source must exist and be readable cannot copy a
   * directory to a file will not copy if timestamps and filesize match, will
   * overwrite otherwise
   * 
   * @param source
   *          the source file
   * @param dest
   *          the destination file
   * @throws IOException
   *           if an error occurs during the copy operation
   */
  public static void copyFile(File source, File dest, IStreamingListener listener) throws IOException {
    if (!source.exists()) {
      throw new FileNotFoundException(source.getAbsolutePath());
    }
    if (!source.canRead()) {
      throw new IOException("cannot read " + source);
    }

    if (dest.exists() && !dest.canWrite()) {
      throw new IOException("cannot write " + dest);
    }

    if (source.isDirectory()) {
      // source can not be a directory
      throw new IOException("source is a directory: " + source);
    }
    // source is a file
    if (dest.isDirectory()) {
      throw new IOException("destination is a directory: " + dest);
    }
    // both source and dest are files
    streamContent(source.length(), new FileInputStream(source), true, new FileOutputStream(dest), true, listener);
  }

  public static String sha1Base64(byte[] bytes) throws Exception {
    MessageDigest md = MessageDigest.getInstance("SHA-1");
    md.update(bytes, 0, bytes.length);
    return Base64Utility.encode(md.digest());
  }

  public static boolean isMatchingSizeAndCrc(File f, long size, long crc) throws IOException {
    return f.exists() && !f.isDirectory() && f.length() == size && crc32(f) == crc;
  }

  public static boolean isMatchingSizeAndCrc(ZipEntry z1, ZipEntry z2) {
    return (z1.getCrc() == z2.getCrc() && z1.getSize() == z2.getSize());
  }

  public static boolean isMatchingSizeAndCrc(File f1, File f2) throws IOException {
    if (!f1.exists()) {
      return false;
    }
    if (!f2.exists()) {
      return false;
    }
    if (f1.length() != f2.length()) {
      return false;
    }
    if (crc32(f1) != crc32(f2)) {
      return false;
    }
    return true;
  }

  public static long crc32(File f) throws IOException {
    if (!f.exists()) {
      return 0L;
    }
    return sizeAndCrc32(f.length(), new FileInputStream(f), true)[1];
  }

  /**
   * In {@link ZipEntry#DEFLATED} method the entry has size/crc=-1 unless the content is fully read, then the values are
   * magically set.
   */
  private static long[] sizeAndCrc32(ZipInputStream zin, ZipEntry e) throws IOException {
    long[] a = new long[]{e.getSize(), e.getCrc()};
    if (a[0] != -1L && a[1] != -1L) {
      return a;
    }
    a = sizeAndCrc32(e.getSize(), zin, false);
    if (e.getSize() != -1L) {
      a[0] = e.getSize();
    }
    if (e.getCrc() != -1L) {
      a[1] = e.getCrc();
    }
    return a;
  }

  public static long[] sizeAndCrc32(long size, InputStream in, boolean closeInputStream) throws IOException {
    final CRC32 crc = new CRC32();
    OutputStream out = new OutputStream() {
      @Override
      public void write(int b) throws IOException {
        crc.update(b);
      }

      @Override
      public void write(byte[] b) throws IOException {
        crc.update(b);
      }

      @Override
      public void write(byte[] b, int off, int len) throws IOException {
        crc.update(b, off, len);
      }
    };
    long sizeStreamed = streamContent(size, in, closeInputStream, out, true, null);
    return new long[]{size >= 0L ? size : sizeStreamed, crc.getValue()};
  }

  private static interface IZipVisitor {
    public void visit(ZipInputStream zin, String pathPrefix, ZipEntry e) throws IOException;
  }
}
