/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.update.win32.elev;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.eclipse.update.f2.internal.update.UpdateContext;
import org.eclipse.update.f2.internal.update.win32.Win32UpdateCommitter;
import org.eclipse.update.f2.internal.util.FileUtility;

/**
 * This is the host process that calls a child process via CMD /C with elevation and delegates to
 * {@link Win32UpdateCommitter}
 */
public class Win32ElevationCaller {

  public void call(UpdateContext ctx, File extractDir) throws Exception {
    List<String> args = createArgs(ctx, extractDir);
    ArrayList<String> cmd = new ArrayList<String>();
    cmd.add("cmd.exe");
    cmd.add("/C");
    cmd.add("update.exe");
    cmd.addAll(args);
    //
    ProcessBuilder pb = new ProcessBuilder(cmd);
    pb.directory(extractDir);
    final Process p = pb.redirectErrorStream(true).start();
    Thread t1 = new Thread() {
      @Override
      public void run() {
        InputStream in = p.getInputStream();
        try {
          while (in.read() >= 0) {
            //nop
          }
        }
        catch (Throwable t) {
          //nop
        }
      }
    };
    t1.start();
    int code = p.waitFor();
    if (code != 0) {
      throw new Exception("Win32 call failed with code " + code);
    }
    final String log = new String(FileUtility.readContent(-1, new FileInputStream(new File(ctx.getTempDir(), Win32ElevationTarget.WIN_ELEV_LOG_FILE_NAME)), true));
    if (!log.contains(Win32ElevationTarget.WIN_ELEV_SUCCESS_TOKEN)) {
      throw new Exception("Win32 call was performed but log contains errors") {
        private static final long serialVersionUID = 1L;

        @Override
        public void printStackTrace(PrintStream s) {
          s.println(log);
        }

        @Override
        public void printStackTrace(PrintWriter s) {
          s.println(log);
        }
      };
    }
  }

  protected String getElevationValue() {
    return Win32ElevationTarget.WIN_ELEV_VALUE_RUN;
  }

  protected List<String> createArgs(UpdateContext ctx, File extractDir) throws Exception {
    File iniFile = new File(extractDir, "update.ini");
    List<String> args = new ArrayList<String>(Arrays.asList(new String(FileUtility.readContent(-1, new FileInputStream(iniFile), true), "UTF-8").split("\\s+")));
    //add vmargs always at END!
    if (!args.contains("-vmargs")) {
      args.add("-vmargs");
    }
    for (int i = 0; i < args.size(); i++) {
      String s = args.get(i);
      if ("-vmargs".equals(s)) {
        i++;
        args.add(i, "-D" + Win32ElevationTarget.UPDATE_TEMP_DIR + "=" + ctx.getTempDir().getAbsolutePath().replace(" ", "%20"));
        args.add(i, "-D" + Win32ElevationTarget.UPDATE_OLD_VERSION + "=" + ctx.getOldVersion());
        args.add(i, "-D" + Win32ElevationTarget.UPDATE_NEW_VERSION + "=" + ctx.getNewVersion());
        args.add(i, "-D" + Win32ElevationTarget.UPDATE_INSTALL_ROOT_DIR + "=" + ctx.getInstallRootDir().getAbsolutePath().replace(" ", "%20"));
        args.add(i, "-D" + Win32ElevationTarget.UPDATE_APP_NAME + "=" + ctx.getAppName());
        args.add(i, "-D" + Win32ElevationTarget.WIN_ELEV_KEY + "=" + getElevationValue());
      }
    }
    return args;
  }
}
