/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.create.win32;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.Deflater;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import java.util.zip.ZipOutputStream;

import org.eclipse.update.f2.F2Parameter;
import org.eclipse.update.f2.internal.create.AbstractCreateProcessor;
import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.LogUtility;
import org.eclipse.update.f2.internal.util.PathUtility;

/**
 * Create F2 Updatesite
 */
public class Win32CreateProcessor extends AbstractCreateProcessor {
  private static final Pattern JVM_DLL_PATH_PATTERN = Pattern.compile("((.*/)?jre.*/bin/client/jvm.dll)", Pattern.CASE_INSENSITIVE);
  private static final Pattern JAVAW_EXE_PATH_PATTERN = Pattern.compile("((.*/)?jre.*/bin/javaw.exe)", Pattern.CASE_INSENSITIVE);

  public Win32CreateProcessor(Map<F2Parameter, String> optionMap) {
    super(optionMap);
  }

  /**
   * Win32
   * <p>
   * Create a top level ini and exe if missing.
   * <p>
   * Create a inner folder update.exe and update.ini
   */
  @Override
  protected void autoCompleteFullVersionZipStructure(File f, String versionFolderName) throws IOException {
    LogUtility.info("verifying content of " + f.getName());
    ZipFile z = new ZipFile(f);
    File t = new File(f.getAbsolutePath() + ".tmp");
    try {
      Pattern innerAppFilePattern = Pattern.compile("([^/]+)/([^/]+)\\.exe");
      boolean containsF2Plugin = false;
      //set of all inner exe app names (without .exe)
      HashSet<String> exeNames = new HashSet<String>();
      HashSet<String> rootEntryNames = new HashSet<String>();
      for (Enumeration<? extends ZipEntry> en = z.entries(); en.hasMoreElements();) {
        ZipEntry ze = en.nextElement();
        String path = ze.getName();
        if (!path.contains("/")) {
          rootEntryNames.add(path);
        }
        if (PathUtility.F2_JAR_PATH_PATTERN.matcher(path).matches()) {
          containsF2Plugin = true;
        }
        Matcher m = innerAppFilePattern.matcher(path);
        if (!m.matches()) {
          continue;
        }
        if (!m.group(1).equals(versionFolderName)) {
          continue;
        }
        exeNames.add(m.group(2));
      }
      exeNames.remove("update");
      if (exeNames.size() == 0) {
        throw new IOException("zip must contain at least one application exe");
      }
      String uacOption = getOptionMap().get(F2Parameter.WindowsUAC);
      boolean useUAC = uacOption != null ? "true".equals(uacOption) : containsF2Plugin;
      boolean complete = true;
      //check if there is an inner update exe/ini
      String innerUpdateExePath = versionFolderName + "/update.exe";
      String innerUpdateIniPath = versionFolderName + "/update.ini";
      if (useUAC) {
        if (z.getEntry(innerUpdateExePath) == null) {
          complete = false;
        }
        if (z.getEntry(innerUpdateIniPath) == null) {
          complete = false;
        }
      }
      //check if there is an app exe/ini on root
      for (String exeName : exeNames) {
        if (!rootEntryNames.contains(exeName + ".exe")) {
          complete = false;
          break;
        }
        if (!rootEntryNames.contains(exeName + ".ini")) {
          complete = false;
          break;
        }
      }
      if (complete) {
        return;
      }
      ZipOutputStream zOut = new ZipOutputStream(new FileOutputStream(t));
      zOut.setLevel(Deflater.BEST_COMPRESSION);
      try {
        //copy existing entries
        for (Enumeration<? extends ZipEntry> en = z.entries(); en.hasMoreElements();) {
          ZipEntry e = en.nextElement();
          zOut.putNextEntry(new ZipEntry(e.getName()));
          if (!e.isDirectory()) {
            FileUtility.streamContent(e.getSize(), z.getInputStream(e), true, zOut, false, null);
          }
          zOut.closeEntry();
        }
        //add missing root exe for app
        for (String exeName : exeNames) {
          String rootPath = exeName + ".exe";
          if (z.getEntry(rootPath) == null) {
            LogUtility.info(" adding " + rootPath);
            ZipEntry innerExe = z.getEntry(versionFolderName + "/" + exeName + ".exe");
            zOut.putNextEntry(new ZipEntry(rootPath));
            FileUtility.streamContent(innerExe.getSize(), z.getInputStream(innerExe), true, zOut, false, null);
            zOut.closeEntry();
          }
        }
        //add missing root ini for app
        for (String exeName : exeNames) {
          String rootPath = exeName + ".ini";
          if (z.getEntry(rootPath) == null) {
            LogUtility.info(" adding " + rootPath);
            ZipEntry innerIni = z.getEntry(versionFolderName + "/" + exeName + ".ini");
            zOut.putNextEntry(new ZipEntry(rootPath));
            byte[] innerIniContent = null;
            if (innerIni != null) {
              innerIniContent = FileUtility.readContent(innerIni.getSize(), z.getInputStream(innerIni), true);
            }
            byte[] data = createRootAppIni(z, versionFolderName, innerIniContent);
            zOut.write(data);
            zOut.closeEntry();
          }
        }
        //add update.exe/ini only if f2 exists inside the product setup
        if (useUAC) {
          //add missing inner update.exe
          if (z.getEntry(innerUpdateExePath) == null) {
            LogUtility.info(" adding " + innerUpdateExePath);
            ZipEntry innerExe = z.getEntry(versionFolderName + "/" + exeNames.iterator().next() + ".exe");
            zOut.putNextEntry(new ZipEntry(innerUpdateExePath));
            FileUtility.streamContent(innerExe.getSize(), z.getInputStream(innerExe), true, zOut, false, null);
            zOut.closeEntry();
          }
          //add missing root update.ini
          if (z.getEntry(innerUpdateIniPath) == null) {
            LogUtility.info(" adding " + innerUpdateIniPath);
            ZipEntry innerIni = z.getEntry(versionFolderName + "/" + exeNames.iterator().next() + ".ini");
            byte[] innerIniContent = null;
            if (innerIni != null) {
              innerIniContent = FileUtility.readContent(innerIni.getSize(), z.getInputStream(innerIni), true);
            }
            zOut.putNextEntry(new ZipEntry(innerUpdateIniPath));
            byte[] data = createInnerUpdateIni(z, innerIniContent);
            zOut.write(data);
            zOut.closeEntry();
          }
        }
      }
      finally {
        zOut.close();
      }
    }
    finally {
      z.close();
    }
    f.delete();
    t.renameTo(f);
  }

  /**
   * Ensure the following entries and qualify directory paths with sub dir name
   * 
   * <pre>
   * --launcher.library
   * 1.0.0/plugins/org.eclipse.equinox.launcher.win32.win32.x86_1.1.2.R36x_v20101222
   * -startup
   * 1.0.0/plugins/org.eclipse.equinox.launcher_1.1.1.R36x_v20101122_1400.jar
   * -vm
   * 1.0.0/jre_1.6.0.26/bin/java.exe
   * </pre>
   */
  protected byte[] createRootAppIni(ZipFile z, String versionFolderName, byte[] innerIni) throws IOException {
    //read ini file
    String startupJar = null;
    String vmExe = null;
    String vmDll = null;
    String launcherLibrary = null;
    List<String> args = new ArrayList<String>();
    if (innerIni != null) {
      args.addAll(Arrays.asList(new String(innerIni, "UTF-8").split("\\s+")));
    }
    //add missing entries (in reverse order and adding to front of list)
    for (Enumeration<?> en = z.entries(); en.hasMoreElements();) {
      ZipEntry ze = (ZipEntry) en.nextElement();
      Matcher m;
      if ((m = JVM_DLL_PATH_PATTERN.matcher(ze.getName())).matches()) {
        vmDll = m.group(1);
      }
      else if ((m = JAVAW_EXE_PATH_PATTERN.matcher(ze.getName())).matches()) {
        vmExe = m.group(1);
      }
      else if ((m = PathUtility.LAUNCHER_JAR_PATH_PATTERN.matcher(ze.getName())).matches()) {
        startupJar = m.group(1);
      }
      else if ((m = PathUtility.LAUNCHER_LIBRARY_PATH_PATTERN.matcher(ze.getName())).matches()) {
        launcherLibrary = m.group(1);
      }
    }
    //adapt paths to launcher specific paths
    for (int i = 0; i < args.size(); i++) {
      String s = args.get(i);
      if ("--launcher.library".equals(s) || "-startup".equals(s) || "-vm".equals(s)) {
        i++;
        args.set(i, versionFolderName + File.separator + args.get(i));
      }
      else if (s != null && s.startsWith("-Djava.util.logging.config.file=")) {
        String[] parts = s.split("=", 2);
        if (parts.length > 1) {
          args.set(i, parts[0] + "=" + versionFolderName + File.separator + parts[1]);
        }
      }
    }
    //add everything else than vmargs BEFORE vmargs, vmargs must be at END!
    if (!args.contains("-vm")) {
      if (vmDll == null && vmExe == null) {
        throw new IOException("could not find a jre in " + z.getName());
      }
      args.add(0, "-vm");
      args.add(1, vmDll != null ? vmDll : vmExe);
    }
    if (!args.contains("-startup")) {
      if (startupJar == null) {
        throw new IOException("could not find a startup jar in " + z.getName());
      }
      args.add(0, "-startup");
      args.add(1, startupJar);
    }
    if (!args.contains("--launcher.library")) {
      if (launcherLibrary == null) {
        throw new IOException("could not find a launcher.library in " + z.getName());
      }
      args.add(0, "--launcher.library");
      args.add(1, launcherLibrary);
    }
    //
    StringBuilder buf = new StringBuilder();
    for (String s : args) {
      buf.append(s);
      buf.append("\n");
    }
    String rootIniContent = buf.toString().trim();
    //create new root ini file
    return rootIniContent.getBytes("UTF-8");
  }

  /**
   * Ensure the following entries
   * 
   * <pre>
   * -startup
   * plugins/org.eclipse.update.f2_1.1.1.R36x_v20101122_1400.jar
   * </pre>
   */
  protected byte[] createInnerUpdateIni(ZipFile z, byte[] innerIni) throws IOException {
    //read ini file
    String startupJar = null;
    String vmExe = null;
    List<String> args = new ArrayList<String>();
    if (innerIni != null) {
      args.addAll(Arrays.asList(new String(innerIni, "UTF-8").split("\\s+")));
    }
    //find f2 plugin
    for (Enumeration<?> en = z.entries(); en.hasMoreElements();) {
      ZipEntry ze = (ZipEntry) en.nextElement();
      Matcher m;
      if ((m = PathUtility.F2_JAR_PATH_PATTERN.matcher(ze.getName())).matches()) {
        startupJar = PathUtility.splitFirstPart(m.group(1))[1];
        break;
      }
      else if ((m = JAVAW_EXE_PATH_PATTERN.matcher(ze.getName())).matches()) {
        vmExe = PathUtility.splitFirstPart(m.group(1))[1];
      }
    }
    if (vmExe == null) {
      LogUtility.warn("There is no java runtime folder in " + z.getName(), null);
    }
    if (startupJar == null) {
      return null;
    }
    //adapt paths to launcher specific paths
    if (vmExe != null && !args.contains("-vm")) {
      args.add(0, "-vm");
      args.add(1, "XXX");
    }
    if (!args.contains("-startup")) {
      args.add(0, "-startup");
      args.add(1, "XXX");
    }
    for (int i = 0; i < args.size(); i++) {
      String s = args.get(i);
      if ("-startup".equals(s)) {
        i++;
        args.set(i, startupJar);
      }
      else if ("-vm".equals(s)) {
        i++;
        args.set(i, vmExe);
      }
    }
    //
    StringBuilder buf = new StringBuilder();
    for (String s : args) {
      buf.append(s);
      buf.append("\n");
    }
    String updateIniContent = buf.toString().trim();
    //create new root ini file
    return updateIniContent.getBytes("UTF-8");
  }

}
