/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;

/**
 * Descriptor for the content of the updatesite defined in the f2.txt file.
 * <p>
 * Full versions are zip with the naming pattern fullVersion := (name)_(version).zip
 * <p>
 * Delta versions are zip with the naming pattern deltaVersion := delta$(name)_(oldVersion)$(name)_(newVersion).zip
 * <p>
 * The name is the name of the application, the version is like the bundle versions 1.2.3[.qualifier]
 * <p>
 * Example:
 * 
 * <pre>
 * app_12.6.0.00.zip (size) (crc) (content-hash)
 * app_12.6.1.00.zip (size) (crc) (content-hash)
 * app_12.6.1.01.zip (size) (crc) (content-hash)
 * app_12.6.2.00.zip (size) (crc) (content-hash)
 * delta$app_12.6.0.00$app_12.6.1.00.zip (size) (crc)
 * delta$app_12.6.1.00$app_12.6.1.01.zip (size) (crc)
 * delta$app_12.6.1.01$app_12.6.2.00.zip (size) (crc)
 * </pre>
 */
public class SiteDesc {
  private final List<VersionDesc> m_versionList;
  private final List<DeltaDesc> m_deltaList;

  /**
   * @param versionList
   *          in the order of versions. The last {@link VersionDesc} in the list is defined as the
   *          {@link #getLatestVersion()}
   * @param deltaList
   */
  public SiteDesc(Collection<VersionDesc> versionList, Collection<DeltaDesc> deltaList) {
    m_versionList = Collections.unmodifiableList(new ArrayList<VersionDesc>(versionList));
    m_deltaList = Collections.unmodifiableList(new ArrayList<DeltaDesc>(deltaList));
  }

  public String getLatestVersion(String maxVersion) {
    String latest = null;
    if (maxVersion == null) {
      if (m_versionList.size() > 0) {
        latest = m_versionList.get(m_versionList.size() - 1).getVersion();
      }
    }
    else {
      for (VersionDesc vd : m_versionList) {
        if (vd.getVersion().equals(maxVersion)) {
          latest = vd.getVersion();
          break;
        }
      }
    }
    return latest;
  }

  public List<VersionDesc> getVersionList() {
    return m_versionList;
  }

  public List<DeltaDesc> getDeltaList() {
    return m_deltaList;
  }

  public String getContent() throws Exception {
    StringBuilder buf = new StringBuilder();
    buf.append("#Filename\tSize\tCRC\tContent-Hash\n");
    for (VersionDesc d : m_versionList) {
      String line = d.getFileName() + "\t" + d.getSize() + "\t" + d.getCrc() + "\t" + d.getContentHash();
      buf.append(line);
      buf.append("\n");
    }
    for (DeltaDesc d : m_deltaList) {
      String line = d.getFileName() + "\t" + d.getSize() + "\t" + d.getCrc();
      buf.append(line);
      buf.append("\n");
    }
    return buf.toString();
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + "[versions=" + m_versionList + ", deltas=" + m_deltaList + "]";
  }

  public static SiteDesc parse(String content) {
    ArrayList<VersionDesc> versionList = new ArrayList<VersionDesc>();
    ArrayList<DeltaDesc> deltaList = new ArrayList<DeltaDesc>();
    for (String line : content.split("[\\n\\r]")) {
      line = line.trim();
      if (line.length() == 0 || line.startsWith("#")) {
        continue;
      }
      String[] a = line.split("[\\s]+");
      if (a[0].startsWith("delta$")) {
        DeltaDesc desc = new DeltaDesc(a[0], Long.parseLong(a[1]), Long.parseLong(a[2]));
        deltaList.add(desc);
      }
      else {
        VersionDesc desc = new VersionDesc(a[0], Long.parseLong(a[1]), Long.parseLong(a[2]), Long.parseLong(a[3]));
        versionList.add(desc);
      }
    }
    return new SiteDesc(versionList, deltaList);
  }

}
