/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.util;

import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.ZipEntry;

import org.eclipse.update.f2.internal.VersionDesc;

/**
 *
 */
public class PathUtility {

  private PathUtility() {
  }

  /**
   * {@link ZipEntry#getComment()} for entries that need to be deleted
   */
  public static final String DELETED_COMMENT = "!deleted!";

  /**
   * {@link ZipEntry#getComment()} for entries that represent an archive delta instead of the archive itself
   */
  public static final String DELTA_ARCHIVE_COMMENT = "!delta-archive!";

  public static final String RENAME_ENTRY_NAME = "rename-map.txt";

  public static final Pattern LAUNCHER_LIBRARY_PATH_PATTERN = Pattern.compile("((.*/)?plugins/org\\.eclipse\\.equinox\\.launcher\\.[^/]+)/", Pattern.CASE_INSENSITIVE);

  public static final Pattern LAUNCHER_JAR_PATH_PATTERN = Pattern.compile("((.*/)?plugins/org\\.eclipse\\.equinox\\.launcher_[^/]+\\.jar)", Pattern.CASE_INSENSITIVE);

  public static final Pattern F2_JAR_PATH_PATTERN = Pattern.compile("((.*/)?plugins/org\\.eclipse\\.update\\.f2_[^/]+\\.jar)", Pattern.CASE_INSENSITIVE);

  protected static final Pattern ZIP_PATH_VERSION_PATTERN = Pattern.compile("(.*)_([0-9]+)\\.([0-9]+)\\.([0-9]+)(\\.([^/\\\\]+))?(/|\\.zip|\\.jar)", Pattern.CASE_INSENSITIVE);

  protected static final Pattern ARCHIVE_PATH_PATTERN = Pattern.compile("(.*)\\.(zip|jar)", Pattern.CASE_INSENSITIVE);

  /**
   * true if the path has at least two parts
   */
  protected static final Pattern FIRST_PATH_PART_PATTERN = Pattern.compile("([^/]+/)([^/]+.*)");

  /**
   * true if the path has at least two parts
   */
  protected static final Pattern LAST_PATH_PART_PATTERN = Pattern.compile("([^/]+.*/)([^/]+[/]?)");

  /**
   * Split the first parts of a path. When there are >=2 parts returns [firstPart, remainingParts] .
   * When there is 1 part returns [firstPart, ""].
   * firstPart ends either with a / (folder) or without a / (file)
   * <p>
   * The path must not start with a /
   */
  public static String[] splitFirstPart(String path) {
    if (path.startsWith("/")) {
      throw new IllegalArgumentException("path must not start with /: " + path);
    }
    Matcher m = FIRST_PATH_PART_PATTERN.matcher(path);
    if (m.matches()) {
      return new String[]{m.group(1), m.group(2)};
    }
    return new String[]{path, ""};
  }

  /**
   * Split the last parts of a path. When there are >=2 parts returns [remainingParts,lastPart] .
   * When there is 1 part returns ["", lastPart].
   * lastPart ends either with a / (folder) or without a / (file)
   * <p>
   * The path must not start with a /
   */
  public static String[] splitLastPart(String path) {
    if (path.startsWith("/")) {
      throw new IllegalArgumentException("path must not start with /: " + path);
    }
    Matcher m = LAST_PATH_PART_PATTERN.matcher(path);
    if (m.matches()) {
      return new String[]{m.group(1), m.group(2)};
    }
    return new String[]{"", path};
  }

  /**
   * When path ends with a versionized pattern _1.2.3[.qualifier] then returns the path without the version suffix,
   * otherwise returns null
   * 
   * <pre>
   * a.b.c_1.2.3.zip -> a.b.c.zip
   * a.b.c_1.0.0.56235423542/ -> a.b.c/
   * a.b.c -> null
   * </pre>
   */
  public static String getPathWithoutVersion(String path) {
    String[] split = PathUtility.splitLastPart(path);
    Matcher m = ZIP_PATH_VERSION_PATTERN.matcher(split[1]);
    if (!m.matches()) {
      return null;
    }
    return split[0] + m.replaceFirst("$1$7");
  }

  public static boolean isArchiveEntry(ZipEntry z) {
    return !z.isDirectory() && ARCHIVE_PATH_PATTERN.matcher(z.getName()).matches();
  }

  public static String getPathWithoutArchiveSuffix(String path) {
    Matcher m = ARCHIVE_PATH_PATTERN.matcher(path);
    if (m.matches()) {
      return m.group(1);
    }
    return path;
  }

  /**
   * Returns true if the provided directory name is the one of an F2 installation.
   * 
   * @param directoryName
   * @param appName
   * @return
   */
  public static boolean isValidF2DirectoryName(String directoryName, String appName) {
    try {
      VersionDesc versionDesc = new VersionDesc(directoryName, 0, 0, 0);
      // directory name must contain the application name as prefix for a valid F2 installation
      return versionDesc.getVersion().startsWith(appName + "_");
    }
    catch (Exception e) {
      // nop (wrong format)
    }
    return false;
  }
}
