/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2.internal.update;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.eclipse.core.runtime.Platform;
import org.eclipse.update.f2.F2Parameter;
import org.eclipse.update.f2.IUserAgent;
import org.eclipse.update.f2.UpdateResult;
import org.eclipse.update.f2.UpdateStrategy;
import org.eclipse.update.f2.internal.DeltaDesc;
import org.eclipse.update.f2.internal.SiteDesc;
import org.eclipse.update.f2.internal.Version3Q;
import org.eclipse.update.f2.internal.VersionDesc;
import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.IStreamingListener;
import org.eclipse.update.f2.internal.util.LogUtility;

/**
 * apply a delta to migrate an old app to the new app
 */
public abstract class AbstractUpdateProcessor implements IUpdateProcessor {
  private final Map<F2Parameter, String> m_optionMap;
  private final IUserAgent m_ua;
  private final File m_installRootDir;
  private final File m_tmpDir;
  private final URL m_rawUrl;
  private URL m_targetUrl;

  /**
   * @param updatesite
   *          is a url to the f2.txt file of the updatesite
   * @param appName
   */
  public AbstractUpdateProcessor(IUserAgent ua, Map<F2Parameter, String> argMap) {
    m_optionMap = argMap != null ? argMap : new HashMap<F2Parameter, String>();
    if (ua == null) {
      throw new IllegalArgumentException("user agent is null");
    }
    m_ua = ua;
    //
    String siteUrl = m_optionMap.get(F2Parameter.SiteUrl);
    try {
      m_rawUrl = new URL(siteUrl);
    }
    catch (Exception e) {
      throw new IllegalArgumentException(F2Parameter.SiteUrl + " <" + siteUrl + "> is not valid: " + e);
    }
    //
    String appName = m_optionMap.get(F2Parameter.Name);
    if (appName == null) {
      throw new IllegalArgumentException(F2Parameter.Name + " is null");
    }
    //detect install root
    String installArg = m_optionMap.get(F2Parameter.InstallDirectory);
    if (installArg == null) {
      throw new IllegalArgumentException(F2Parameter.InstallDirectory + " is null");
    }
    m_installRootDir = new File(installArg);
    if (!m_installRootDir.exists()) {
      throw new IllegalArgumentException(m_installRootDir + " does not exist");
    }
    //
    String tmpArg = m_optionMap.get(F2Parameter.TempDirectory);
    m_tmpDir = (tmpArg != null ? new File(tmpArg) : new File(m_installRootDir, "_tmp"));
  }

  protected final IUserAgent getUserAgent() {
    return m_ua;
  }

  protected final Map<F2Parameter, String> getOptionMap() {
    return m_optionMap;
  }

  protected final File getInstallRootDir() {
    return m_installRootDir;
  }

  protected final File getTmpDir() {
    return m_tmpDir;
  }

  @Override
  public final UpdateResult update(UpdateStrategy strategy) {
    try {
      FileUtility.rmdir(getTmpDir());
      getTmpDir().mkdirs();
      //
      UpdateResult res;
      try {
        res = updateInternal(strategy);
      }
      catch (Throwable t) {
        LogUtility.error("Update failed", t);
        res = UpdateResult.UpdateFailed;
      }
      try {
        getUserAgent().setProgressValue(1.0);
        getUserAgent().setProgressText(res.toString());
      }
      catch (Throwable t) {
        //nop
      }
      return res;
    }
    finally {
      FileUtility.rmdir(getTmpDir());
    }
  }

  protected UpdateResult updateInternal(UpdateStrategy strategy) throws Throwable {
    //get site desc
    SiteDesc siteDesc = downloadSiteDesc();
    String limitVersion = getOptionMap().get(F2Parameter.LimitedVersion);
    String newVersion = siteDesc.getLatestVersion(limitVersion);
    LogUtility.info("Latest version is " + newVersion);
    if (newVersion == null) {
      //nothing to do
      if (limitVersion != null) {
        LogUtility.info("UnexpectedVersion since version " + limitVersion + " does not exist");
        return UpdateResult.UnexpectedVersion;
      }
      else {
        LogUtility.info("NothingToDo since there is no latest version");
        return UpdateResult.NothingToDo;
      }
    }
    VersionDesc newVersionDesc = findVersionDesc(siteDesc, newVersion);
    if (newVersionDesc == null) {
      //there is no valid delta update constellation available (misconfiguration?)
      LogUtility.warn("There is no desc available for the new version " + newVersion + " (bad configuration?)", null);
      return UpdateResult.UpdateFailed;
    }
    //get latest valid installation folder and version
    File installRootDir = getInstallRootDir();
    LogUtility.info("Installation root directory is " + installRootDir);
    TreeMap<Version3Q, VersionDesc> installedVersions = new TreeMap<Version3Q, VersionDesc>();
    for (File dir : m_installRootDir.listFiles()) {
      if (dir.isHidden() || !dir.isDirectory()) {
        continue;
      }
      VersionDesc vd = findVersionDesc(siteDesc, dir.getName());
      if (vd != null && new File(m_installRootDir, vd.getVersion() + ".zip").exists()) {
        installedVersions.put(vd.getVersion3Q(), vd);
      }
    }
    VersionDesc oldVersionDesc;
    if (installedVersions.size() == 0) {
      LogUtility.warn(m_installRootDir + " does not contain an existing installation; Assuming it is empty or contains a legacy application.", null);
      oldVersionDesc = null;
    }
    else {
      oldVersionDesc = installedVersions.lastEntry().getValue();
      LogUtility.info("Old version is " + oldVersionDesc.getVersion());
    }
    String oldVersion = (oldVersionDesc != null ? oldVersionDesc.getVersion() : null);
    if (newVersion.equalsIgnoreCase(oldVersion)) {
      //nothing to do
      LogUtility.info("NothingToDo since version is up to date");
      return UpdateResult.NothingToDo;
    }
    File alreadyInstalledNewVersionZip = new File(installRootDir, newVersion + ".zip");
    if (alreadyInstalledNewVersionZip.exists()) {
      //nothing to do
      LogUtility.info("NothingToDo since another process updated already");
      return UpdateResult.NothingToDo;
    }

    // handle the case the zip file was deleted manually but the current installation is the newest one
    if (isVersionInstalled(newVersionDesc)) {
      LogUtility.info("No old version description found, but install directory seems to be a valid F2 installation with the newest version.");
      return UpdateResult.NothingToDo;
    }

    if (strategy == UpdateStrategy.CheckForUpdate) {
      LogUtility.info("UpdateRequired");
      return UpdateResult.UpdateRequired;
    }
    //do update
    UpdateContext ctx = new UpdateContext(m_optionMap.get(F2Parameter.Name), installRootDir, oldVersion, m_tmpDir, newVersion);
    //try delta update
    try {
      if (oldVersion != null) {
        File oldInstalledZip = new File(ctx.getInstallRootDir(), ctx.getOldVersion() + ".zip");
        LogUtility.info("Does old zip " + oldInstalledZip + " exist? " + oldInstalledZip.exists());
        if (oldInstalledZip.exists()) {
          //calculate which deltas would be needed
          List<DeltaDesc> deltaList = calculateRequiredDeltas(siteDesc, ctx.getOldVersion(), ctx.getNewVersion());
          if (oldVersionDesc != null && !deltaList.isEmpty() && getDownloadSizeOfDeltaList(deltaList) < newVersionDesc.getSize() * 8 / 10) {
            doDeltaUpdate(siteDesc, deltaList, oldVersionDesc, newVersionDesc, ctx);
            LogUtility.info("UpdateSuccessful");
            return UpdateResult.UpdateSuccessful;
          }
        }
      }
    }
    catch (Throwable t) {
      LogUtility.warn("Delta update failed", t);
    }
    doFullUpdate(siteDesc, newVersionDesc, ctx);
    LogUtility.info("UpdateSuccessful");
    return UpdateResult.UpdateSuccessful;
  }

  /**
   * Checks whether the provided version is the currently executed one.
   * 
   * @param versionDesc
   * @return
   */
  protected boolean isVersionInstalled(VersionDesc versionDesc) {
    try {
      // get real install dir, and not the one provided from the option map
      File installDir = new File(Platform.getInstallLocation().getURL().getPath());
      VersionDesc installedVersionDesc = new VersionDesc(installDir.getName(), 0, 0, 0);
      return versionDesc.getVersion().equals(installedVersionDesc.getVersion());
    }
    catch (Exception e) {
      // failed to parse version desc
    }
    return false;
  }

  /**
   * download f2.txt and detect which subfolder folder to use. See {@link IUserAgent#getUpdateSiteUrl()} for details.
   */
  protected SiteDesc downloadSiteDesc() throws IOException {
    //2 variations: /os/arch and /os
    IStreamingListener strListener = new ProgressPhaseStreamingListener(m_ua, ProgressPhase.DownloadSiteDesc, 1000);
    String os = getOptionMap().get(F2Parameter.OS);
    String arch = getOptionMap().get(F2Parameter.Arch);
    if (arch != null) {
      m_targetUrl = new URL(m_rawUrl, m_rawUrl.getPath() + "/" + os + "/" + arch + "/");
      try {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        downloadImpl("f2.txt", bos, -1, strListener);
        return SiteDesc.parse(new String(bos.toByteArray(), "UTF-8"));
      }
      catch (Throwable t) {
        //nop
      }
    }
    m_targetUrl = new URL(m_rawUrl, m_rawUrl.getPath() + "/" + os + "/");
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    downloadImpl("f2.txt", bos, -1, strListener);
    return SiteDesc.parse(new String(bos.toByteArray(), "UTF-8"));
  }

  /**
   * @param siteDesc
   * @param deltaList
   * @param oldVersionDesc
   * @param newVersionDesc
   * @param ctx
   */
  protected void doDeltaUpdate(SiteDesc siteDesc, final List<DeltaDesc> deltaList, VersionDesc oldVersionDesc, VersionDesc newVersionDesc, UpdateContext ctx) throws Throwable {
    LogUtility.info("Delta update");
    //copy old version to tmp
    File oldInstalledZip = new File(ctx.getInstallRootDir(), ctx.getOldVersion() + ".zip");
    if (!oldInstalledZip.exists()) {
      throw new IOException("old zip " + oldInstalledZip + " does not exist");
    }
    boolean checkContentHash = ("true".equals(getOptionMap().get(F2Parameter.CheckContentHash)));
    if (checkContentHash) {
      LogUtility.info("check content hash of old zip " + oldInstalledZip.getName());
      long h = FileUtility.archiveHash(oldInstalledZip);
      if (h != oldVersionDesc.getContentHash()) {
        throw new IOException("content hash mismatch on zipped old zip: expected " + oldVersionDesc.getContentHash() + " got " + h);
      }
    }
    //download deltas
    IStreamingListener downloadListener = new ProgressPhaseStreamingListener(m_ua, ProgressPhase.DownloadZipFiles, getDownloadSizeOfDeltaList(deltaList));
    for (DeltaDesc desc : deltaList) {
      File deltaFile = new File(ctx.getTempDir(), desc.getFileName());
      downloadImpl(desc.getFileName(), new FileOutputStream(deltaFile), desc.getSize(), downloadListener);
      checkCrc(deltaFile, desc.getCrc());
    }
    //apply deltas
    File inFile = oldInstalledZip;
    File outFile = null;
    IStreamingListener applyListener = new ProgressPhaseStreamingListener(m_ua, ProgressPhase.BuildUpdate, newVersionDesc.getSize());
    for (DeltaDesc desc : deltaList) {
      File deltaFile = new File(ctx.getTempDir(), desc.getFileName());
      outFile = new File(ctx.getTempDir(), desc.getNewVersion() + ".zip");
      LogUtility.info("Delta update from " + desc.getOldVersion() + " to " + desc.getNewVersion() + " with " + desc.getSize() + " bytes");
      new ZipDeltaUpdater().process(inFile, deltaFile, outFile, ctx.getTempDir(), applyListener);
      inFile = outFile;
    }
    //verify new app
    if (checkContentHash) {
      LogUtility.info("check content hash of new zip " + (outFile != null ? outFile.getName() : null));
      long expectedHash = newVersionDesc.getContentHash();
      long actualHash = FileUtility.archiveHash(outFile);
      if (expectedHash != actualHash) {
        throw new IOException("content hash mismatch on " + outFile + ": expected " + expectedHash + " got " + actualHash);
      }
    }
    //commit new app
    LogUtility.info("Committing");
    commit(ctx);
  }

  /**
   * @param siteDesc
   * @param desc
   * @param ctx
   */
  protected void doFullUpdate(SiteDesc siteDesc, VersionDesc desc, UpdateContext ctx) throws Throwable {
    LogUtility.info("Full update");
    //download full zip
    File vFile = new File(ctx.getTempDir(), desc.getFileName());
    IStreamingListener downloadListener = new ProgressPhaseStreamingListener(m_ua, ProgressPhase.DownloadZipFiles, desc.getSize());
    downloadImpl(desc.getFileName(), new FileOutputStream(vFile), desc.getSize(), downloadListener);
    m_ua.setProgressValue(ProgressPhase.BuildUpdate.getProgress(1, 1));
    checkCrc(vFile, desc.getCrc());
    //commit new app
    LogUtility.info("Committing");
    commit(ctx);
  }

  protected void commit(UpdateContext ctx) throws Throwable {
    new DefaultUpdateCommitter(m_ua, ctx).commit();
  }

  protected void downloadImpl(String context, OutputStream out, long size, IStreamingListener listener) throws IOException {
    URL url = new URL(m_targetUrl, context);
    FileUtility.download(url, out, size, m_ua, listener);
  }

  protected void checkCrc(File f, long expectedCrc) throws IOException {
    LogUtility.info("Check CRC of " + f.getName());
    long actualCrc = FileUtility.crc32(f);
    if (actualCrc != expectedCrc) {
      throw new IOException("crc mismatch on " + f.getName() + ": expected " + Long.toHexString(expectedCrc) + " got " + Long.toHexString(actualCrc));
    }
  }

  protected List<DeltaDesc> calculateRequiredDeltas(SiteDesc siteDesc, String oldVersion, String newVersion) {
    TreeMap<Long, List<DeltaDesc>> candidates = new TreeMap<Long, List<DeltaDesc>>();
    ArrayList<DeltaDesc> currentList = new ArrayList<DeltaDesc>();
    for (DeltaDesc desc : siteDesc.getDeltaList()) {
      if (desc.getOldVersion().equals(oldVersion)) {
        currentList.add(desc);
        visitDeltaList(siteDesc, currentList, candidates, newVersion);
        currentList.remove(desc);
      }
    }
    if (candidates.size() == 0) {
      return Collections.emptyList();
    }
    return candidates.get(candidates.firstKey());
  }

  protected void visitDeltaList(SiteDesc siteDesc, List<DeltaDesc> currentList, Map<Long, List<DeltaDesc>> candidates, String maxVersion) {
    String curVersion = currentList.get(currentList.size() - 1).getNewVersion();
    if (curVersion.equals(maxVersion)) {
      candidates.put(getDownloadSizeOfDeltaList(currentList), new ArrayList<DeltaDesc>(currentList));
      return;
    }
    for (DeltaDesc desc : siteDesc.getDeltaList()) {
      if (desc.getOldVersion().equals(curVersion)) {
        currentList.add(desc);
        visitDeltaList(siteDesc, currentList, candidates, maxVersion);
        currentList.remove(desc);
      }
    }
  }

  protected long getDownloadSizeOfDeltaList(List<DeltaDesc> list) {
    long size = 0L;
    for (DeltaDesc desc : list) {
      size += desc.getSize();
    }
    return size;
  }

  protected VersionDesc findVersionDesc(SiteDesc siteDesc, String version) {
    for (VersionDesc desc : siteDesc.getVersionList()) {
      if (desc.getVersion().equals(version)) {
        return desc;
      }
    }
    return null;
  }
}
