/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2;

import java.io.File;
import java.util.HashMap;
import java.util.Map;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.Platform;
import org.eclipse.core.runtime.Plugin;
import org.eclipse.update.f2.internal.update.UpdateProcessorFactory;
import org.eclipse.update.f2.internal.util.FileUtility;
import org.eclipse.update.f2.internal.util.LogUtility;
import org.eclipse.update.f2.internal.util.PathUtility;
import org.osgi.framework.BundleContext;

/**
 * Call {@link #update(UpdateStrategy, IUserAgent)} to update the install location
 */
public class F2Updater extends Plugin {

  public static final String PLUGIN_ID = "org.eclipse.update.f2";

  private static BundleContext bundleContext;

  public F2Updater() {
  }

  @Override
  public void start(BundleContext context) throws Exception {
    super.start(context);
    bundleContext = context;
  }

  @Override
  public void stop(BundleContext context) throws Exception {
    bundleContext = null;
    super.stop(context);
  }

  public static UpdateResult update() {
    return update(null, null, null);
  }

  /**
   * see {@link #update(UpdateStrategy, Map, IUserAgent, IProgressMonitor)}
   */
  public static UpdateResult update(UpdateStrategy strategy, Map<F2Parameter, String> optionMap, IUserAgent ua) {
    return update(strategy, optionMap, ua, null);
  }

  /**
   * See the {@link IUserAgent#getUpdateSiteUrl()} updateSite parameter for more details.
   * <p>
   * This method never throws an exception.
   * 
   * @param strategy
   *          is default {@link UpdateStrategy#DoUpdate}
   * @param optionMap
   *          may be null. Overrides the options read from the config.ini, launch.ini and system properties
   * @param ua
   *          is default {@link EclipseUserAgent}
   * @param monitor
   *          only used if ua is null
   */
  public static UpdateResult update(UpdateStrategy strategy, Map<F2Parameter, String> optionMap, IUserAgent ua, IProgressMonitor monitor) {
    if (strategy == null) {
      strategy = UpdateStrategy.DoUpdate;
    }

    Map<F2Parameter, String> completeMap = new HashMap<F2Parameter, String>();
    completeMap.put(F2Parameter.OS, Platform.getOS());
    completeMap.put(F2Parameter.Arch, Platform.getOSArch());
    // override with config.ini
    completeMap.putAll(parseOptions());
    // override with call parameter
    if (optionMap != null) {
      completeMap.putAll(optionMap);
    }

    // add install directory if not set
    if (!completeMap.containsKey(F2Parameter.InstallDirectory)) {
      File installDir = getInstallDir(completeMap.get(F2Parameter.Name));
      completeMap.put(F2Parameter.InstallDirectory, installDir.getAbsolutePath());
    }

    // add temp directory if not set
    if (completeMap.get(F2Parameter.TempDirectory) == null) {
      try {
        completeMap.put(F2Parameter.TempDirectory, FileUtility.createTempDir("oeuf2", "").getAbsolutePath());
      }
      catch (Throwable t) {
        LogUtility.error("Create temp directory", t);
        return UpdateResult.UpdateFailed;
      }
    }

    // progress monitor
    if (ua == null) {
      if (monitor == null) {
        monitor = new NullProgressMonitor();
      }
      ua = new EclipseUserAgent(monitor, completeMap);
    }

    // start update process
    return UpdateProcessorFactory.getProcessor(ua, completeMap).update(strategy);
  }

  public static BundleContext getContext() {
    return bundleContext;
  }

  /**
   * Parse all arguments from the config.ini
   */
  public static Map<F2Parameter, String> parseOptions() {
    BundleContext ctx = F2Updater.getContext();
    HashMap<F2Parameter, String> argMap = new HashMap<F2Parameter, String>();
    for (F2Parameter key : F2Parameter.values()) {
      String value = ctx.getProperty("f2." + key.getKey());
      if (value != null) {
        // a value of length 0 is okay (used to override prefilled parameters)
        // but put null in map, because only == null is checked
        argMap.put(key, value.trim().length() == 0 ? null : value);
      }
    }
    return argMap;
  }

  /**
   * Returns the installation directory of the application.
   * Special handling is required in case of an existing F2 application.
   * 
   * @param appName
   * @return
   */
  private static File getInstallDir(String appName) {
    File installDir = new File(Platform.getInstallLocation().getURL().getPath());
    if (installDir.exists()) {
      // in case of a legacy application, the above install dir points to the main directory (contains the executable file)
      // in case of an F2 application, the above install dir points to the sub directory (i.e. [name]_[version incl. qualifier])
      if (new File(installDir.getParentFile(), installDir.getName() + ".zip").exists()) {
        // the zip file (with the same name as the directory) exists in the parent directory
        // i.e. it is an F2 installation -> change the install directory to the parent directory
        installDir = installDir.getParentFile();
      }
      else {
        // zip file does not exists. 2 options:
        // 1. legacy application
        // 2. user deleted the zip files (leaving only the executables and other folders)
        // Thus checking the name of the above install folder. If it matches the expected signature of an F2 application
        // change the install directory to the parent directory (as if the zip file would exist).
        // This will return the wrong directory in the case that the legacy application was installed in a directory
        // having the same name format as an F2 application. In such a case the parent directory of the installation folder is returned
        // which would be incorrect.
        if (PathUtility.isValidF2DirectoryName(installDir.getName(), appName)) {
          // F2 application -> use parent directory as installation directory
          installDir = installDir.getParentFile();
        }
        else {
          LogUtility.info("Installation directory does not conform to expected F2 application structure. Legacy application?");
        }
      }
    }
    return installDir;
  }
}
