/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2;

public enum F2Parameter {

  /**
   * Perform create/rebuild process
   * <p>
   * Command line: -create
   */
  Create("create"),

  /**
   * Perform update process
   * <p>
   * Command line: -update
   */
  Update("update"),

  /**
   * Operation system
   * <p>
   * Command line: -os win32
   * <p>
   * config.ini: f2.os=win32
   */
  OS("os"),

  /**
   * Operation system architecture
   * <p>
   * Command line: -arch x86
   * <p>
   * config.ini: f2.arch=x86
   */
  Arch("arch"),

  /**
   * Simple name of the application
   * <p>
   * Command line: -name example
   * <p>
   * config.ini: f2.name=example
   */
  Name("name"),

  /**
   * Site root directory when creating / rebuilding an updatesite
   * <p>
   * Command line: -site D:/tomcat6/webapps/updatesite
   * <p>
   * config.ini: f2.site=D:/tomcat6/webapps/updatesite
   */
  SiteDirectory("site"),

  /**
   * Site root url when updating from an updatesite
   * <p>
   * Command line: -url http://.../updatesite
   * <p>
   * config.ini: f2.url=http://.../updatesite
   * <p>
   * The updatesite points to the root folder under which there are sub folders for every supported os.arch combination.
   * 
   * <pre>
   * updateSiteUrl = .../updatesite
   * </pre>
   * 
   * The updater will first check to find
   * 
   * <pre>
   * ${updateSiteUrl}/${OS}/${ARCH}/f2.txt
   * </pre>
   * 
   * and - if not found - as a convenience also
   * 
   * <pre>
   * ${updateSiteUrl}/${OS}/${ARCH}/f2.txt
   * </pre>
   * 
   * This is to support defining a single application version for one os and any architecture. Thus valid updatesite
   * subfolders are
   * 
   * <pre>
   * .../updatesite/win32/x86
   * .../updatesite/win32/x86_64
   * .../updatesite/macosx/ppc
   * .../updatesite/macosx
   * </pre>
   * 
   * Each such subfolder contains a f2.txt file.
   * <p>
   * The following tables show values for OS and ARCH:
   * <p>
   * <b>os</b>
   * <ul>
   * <li>aix</li>
   * <li>hpux</li>
   * <li>linux</li>
   * <li>macosx</li>
   * <li>solaris</li>
   * <li>win32</li>
   * </ul>
   * <b>arch</b>
   * <ul>
   * <li>ia64_32</li>
   * <li>ppc</li>
   * <li>ppc64</li>
   * <li>s390</li>
   * <li>s390x</li>
   * <li>sparc</li>
   * <li>x86</li>
   * <li>x86_64</li>
   * </ul>
   */
  SiteUrl("url"),

  /**
   * Local installation root folder
   * <p>
   * Command line: -install "C:/Program Files/example"
   * <p>
   * config.ini: f2.install=C:/Program Files/example
   * <p>
   * More specific this is the directory where the current (old) application is installed in. This is not the version
   * subdirectory inside the root dir (not the app-root/app_1.0.0/ folder).
   */
  InstallDirectory("install"),

  /**
   * Temporary directory to use
   * <p>
   * Command line: -temp D:/TEMP
   * <p>
   * config.ini: f2.temp=D:/TEMP
   */
  TempDirectory("temp"),

  /**
   * Option for detailed logging
   * <p>
   * Command line: -verbose
   * <p>
   * config.ini: f2.verbose=true
   */
  LogVerbose("verbose"),
  /**
   * Option for sparsed logging
   * <p>
   * Command line: -silent
   * <p>
   * config.ini: f2.silent=true
   */
  LogSilent("silent"),

  /**
   * HTTP Basic auth username
   * <p>
   * Command line: -http.user xyz
   * <p>
   * config.ini: f2.http.user=xyz
   */
  HttpUser("http.user"),
  /**
   * HTTP Basic auth password
   * <p>
   * Command line: -http.pass xyz
   * <p>
   * config.ini: f2.http.user=xyz
   */
  HttpPass("http.pass"),

  /**
   * Windows 7 UAC (user access control) enable/disable
   * <p>
   * Command line: -uac true
   * <p>
   * config.ini: f2.uac=true
   */
  WindowsUAC("uac"),

  /**
   * Only update up to a limited version. Value is a version text such as <code>example_1.2.0</code>
   * <p>
   * Command line: -version example_1.2.0
   * <p>
   * config.ini: f2.version=example_1.2.0
   */
  LimitedVersion("version"),

  /**
   * When update performs a delta update, this parameter determines if the zip content hashes should be checked or not.
   * Default is true.
   * <p>
   * Command line: -checkhash true
   * <p>
   * config.ini: f2.checkhash=true
   */
  CheckContentHash("checkhash"),

  /**
   * The number of application versions to keep when creating / rebuilding an updatesite. Due to required space on disk,
   * it might be useful to keep only a specific number of old application versions. By default, all versions are kept
   * (value <= 0). A value of 1 would mean to only keep the newest version.
   * <p>
   * Command line: -versionsToKeep 5
   * <p>
   * config.ini: f2.versionsToKeep=5
   */
  NumberOfVersionsToKeep("versionsToKeep");

  private String m_key;

  F2Parameter(String key) {
    m_key = key;
  }

  public String getKey() {
    return m_key;
  }

  @Override
  public String toString() {
    return getKey();
  }

  public static F2Parameter parse(String key) {
    for (F2Parameter arg : F2Parameter.values()) {
      if (arg.getKey().equalsIgnoreCase(key)) {
        return arg;
      }
    }
    throw new IllegalArgumentException("unknown argument '" + key + "'");
  }
}
