/*******************************************************************************
 * Copyright (c) 2010 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the BSI AG Software License v1.0
 * which accompanies this distribution as bsi-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.update.f2;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicReference;

import javax.security.auth.callback.NameCallback;
import javax.security.auth.callback.PasswordCallback;

import org.eclipse.update.f2.internal.util.LogUtility;
import org.eclipse.update.f2.resource.Texts;

public class BasicUserAgent implements IUserAgent {
  private final Map<F2Parameter, String> m_optionMap;
  private String m_progressText;
  private int m_percent;

  public BasicUserAgent(Map<F2Parameter, String> optionMap) {
    m_optionMap = Collections.unmodifiableMap(optionMap != null ? optionMap : new HashMap<F2Parameter, String>());
    m_progressText = Texts.Updating;
    m_percent = 0;
  }

  protected final Map<F2Parameter, String> getOptionMap() {
    return m_optionMap;
  }

  @Override
  public void preAuthenticate(AtomicReference<String> authMethod, NameCallback cbName, PasswordCallback cbPass) {
    authenticate(authMethod, cbName, cbPass);
  }

  @Override
  public void authenticate(AtomicReference<String> authMethod, NameCallback cbName, PasswordCallback cbPass) {
    String httpUser = getOptionMap().get(F2Parameter.HttpUser);
    String httpPass = getOptionMap().get(F2Parameter.HttpPass);
    if (httpUser != null) {
      cbName.setName(httpUser);
    }
    if (httpPass != null) {
      cbPass.setPassword(httpPass.toCharArray());
    }
  }

  @Override
  public void setProgressText(String name) {
    m_progressText = name;
    LogUtility.info(m_progressText + " " + getProgressPercent() + "%");
  }

  @Override
  public void setProgressValue(double value) {
    // T126012: Synchronize, since this is called from multiple threads (visibility issue).
    //          Sanity: ensure that progress never declines, otherwise we can get > 100% values.
    int newPercent = (int) (value * 100);
    int worked = newPercent - m_percent;
    if (worked > 0) {
      m_percent = newPercent;
      updateProgress(worked);
    }
  }

  /**
   * @return the progress in percent 0...100
   */
  protected int getProgressPercent() {
    return m_percent;
  }

  /**
   * @param worked
   *          the new additional amount in percent that is done. The current percentage is obtained from
   *          {@link #getProgressPercent()}
   */
  protected void updateProgress(int worked) {
    LogUtility.info(m_progressText + " " + getProgressPercent() + "%");
  }
}
