package org.eclipse.scout.sdk.docx4j.internal.operation;

import java.io.File;
import java.util.Collection;
import java.util.LinkedList;
import java.util.TreeMap;

import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.jdt.core.ICompilationUnit;
import org.eclipse.jdt.core.IJavaElement;
import org.eclipse.jdt.core.JavaCore;
import org.eclipse.jdt.core.JavaModelException;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.exception.ProcessingException;
import org.eclipse.scout.docx4j.XlsxAdapter;
import org.eclipse.scout.docx4j.XlsxAdapter.PredefinedStyle;
import org.eclipse.scout.nls.sdk.internal.search.NlsFindKeysJob;
import org.eclipse.scout.nls.sdk.model.INlsEntry;
import org.eclipse.scout.nls.sdk.model.util.Language;
import org.eclipse.scout.nls.sdk.model.workspace.project.INlsProject;
import org.eclipse.scout.sdk.docx4j.internal.Docx4jSdkActivator;
import org.eclipse.scout.sdk.docx4j.internal.model.NlsLine;
import org.eclipse.scout.sdk.operation.IOperation;
import org.eclipse.scout.sdk.util.log.ScoutStatus;
import org.eclipse.scout.sdk.util.type.TypeUtility;
import org.eclipse.scout.sdk.util.typecache.IWorkingCopyManager;
import org.eclipse.search.ui.text.Match;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Row;

public class ExportNlsToExcelOperation implements IOperation {

  static final String USAGE_HEADER_TEXT = "Usage";

  private boolean m_exportUsage;
  private boolean m_exportInheritedEntries;
  private String m_fileUrl;
  private INlsProject m_projectToExport;

  @Override
  public String getOperationName() {
    return "Export NLS Entries to Excel.";
  }

  @Override
  public void validate() {
    if (!StringUtility.hasText(getFileUrl())) {
      throw new IllegalArgumentException("No destination file specified.");
    }
    if (getProjectToExport() == null) {
      throw new IllegalArgumentException("NLS Project may not be null.");
    }
  }

  @Override
  public void run(IProgressMonitor monitor, IWorkingCopyManager workingCopyManager) throws CoreException {
    try {
      Collection<NlsLine> entriesToExport = getEntriesToExport(monitor);

      writeXlsx(entriesToExport);
    }
    catch (ProcessingException e) {
      throw new CoreException(new ScoutStatus(e));
    }
  }

  protected void writeXlsx(Collection<NlsLine> lines) throws ProcessingException {
    XlsxAdapter xlsx = new XlsxAdapter();
    WorksheetPart sheet = xlsx.createSheet("NLS Entries");

    writeHeader(xlsx, sheet);
    long row = 1;
    for (NlsLine l : lines) {
      writeLine(xlsx, sheet, l, row++);
    }

    String fileUrl = getFileUrl();
    String xlsxExtension = ".xlsx";
    if (!fileUrl.toLowerCase().endsWith(xlsxExtension)) {
      fileUrl += xlsxExtension;
    }
    File file = new File(fileUrl);
    if (file.exists()) {
      file.delete();
    }
    else {
      File folder = file.getParentFile();
      if (!folder.exists() && !folder.mkdirs()) {
        throw new ProcessingException("Cannot create folder '" + folder.getAbsolutePath() + "'.");
      }
    }
    xlsx.save(file);
  }

  private void writeLine(XlsxAdapter xlsx, WorksheetPart sheet, NlsLine line, long rowIndex) throws ProcessingException {
    Row row = xlsx.getRow(sheet, rowIndex);
    int col = 0;
    xlsx.setCellValue(row, col++, line.getKey());
    for (Language lang : getProjectToExport().getAllLanguages()) {
      xlsx.setCellValue(row, col++, line.getTranslation(lang));
    }
    if (isExportUsage()) {
      Cell cell = xlsx.getCell(row, col);
      cell.setS(xlsx.getPredefinedStyleNumber(XlsxAdapter.PredefinedStyle.WRAP_TEXT));
      xlsx.setCellValue(row, col, line.getUsagesAsString());
    }
  }

  private void writeHeader(XlsxAdapter xlsx, WorksheetPart sheet) throws ProcessingException {
    Row headerRow = xlsx.getRow(sheet, 0);
    int col = 0;

    xlsx.setCellStyle(xlsx.getCell(headerRow, col), PredefinedStyle.BOLD);
    xlsx.setCellValue(headerRow, col++, Language.LANGUAGE_KEY.getIsoCode());
    for (Language l : getProjectToExport().getAllLanguages()) {
      xlsx.setCellStyle(xlsx.getCell(headerRow, col), PredefinedStyle.BOLD);
      xlsx.setCellValue(headerRow, col++, l.getIsoCode());
    }

    if (isExportUsage()) {
      xlsx.setCellStyle(xlsx.getCell(headerRow, col), PredefinedStyle.BOLD);
      xlsx.setCellValue(headerRow, col++, USAGE_HEADER_TEXT);
    }
  }

  protected Collection<NlsLine> getEntriesToExport(IProgressMonitor monitor) {
    Language[] languages = getProjectToExport().getAllLanguages();
    INlsEntry[] entries = getProjectToExport().getAllEntries();
    TreeMap<String, NlsLine> lines = new TreeMap<String, NlsLine>();
    for (INlsEntry entry : entries) {
      if (entry.getType() == INlsEntry.TYPE_LOCAL || isExportInheritedEntries()) {
        String key = entry.getKey();
        NlsLine line = new NlsLine(key);
        for (Language lang : languages) {
          line.addTranslation(lang, entry.getTranslation(lang));
        }
        lines.put(key, line);
      }
    }
    if (isExportUsage()) {
      NlsFindKeysJob job = new NlsFindKeysJob(getProjectToExport(), "Calculate NLS Entry Usage");
      job.run(monitor);
      for (NlsLine l : lines.values()) {
        Match[] matches = job.getMatches(l.getKey());
        for (Match m : matches) {
          try {
            String path = findPathOfMatch(m);
            if (path != null) {
              l.addUsage(path);
            }
          }
          catch (JavaModelException e) {
            Docx4jSdkActivator.logWarning("could not resolve path of match in '" + m.getElement() + "'.", e);
          }
        }
      }
    }
    return lines.values();
  }

  private String findPathOfMatch(Match match) throws JavaModelException {
    if (match.getElement() instanceof IFile) {
      ICompilationUnit compilationUnit = (ICompilationUnit) JavaCore.create((IFile) match.getElement());
      IJavaElement element = compilationUnit.getElementAt(match.getOffset());
      if (TypeUtility.exists(element)) {
        LinkedList<String> pathTokens = new LinkedList<String>();
        createPath(element, pathTokens);
        if (pathTokens.size() > 0) {
          StringBuilder builder = new StringBuilder();
          for (int i = 0; i < pathTokens.size(); i++) {
            builder.append(pathTokens.get(i));
            if (i < pathTokens.size() - 1) {
              builder.append(" - ");
            }
          }
          return builder.toString();
        }
      }
    }
    return null;
  }

  private void createPath(IJavaElement element, LinkedList<String> pathTokens) {
    switch (element.getElementType()) {
      case IJavaElement.COMPILATION_UNIT:
        return;
      case IJavaElement.METHOD:
      case IJavaElement.TYPE:
        pathTokens.addFirst(element.getElementName());
        IJavaElement parent = element.getParent();
        if (TypeUtility.exists(parent)) {
          createPath(parent, pathTokens);
        }
        break;
    }
  }

  public boolean isExportUsage() {
    return m_exportUsage;
  }

  public void setExportUsage(boolean m_exportUsage) {
    this.m_exportUsage = m_exportUsage;
  }

  public String getFileUrl() {
    return m_fileUrl;
  }

  public void setFileUrl(String m_fileUrl) {
    this.m_fileUrl = m_fileUrl;
  }

  public INlsProject getProjectToExport() {
    return m_projectToExport;
  }

  public void setProjectToExport(INlsProject m_projectToExport) {
    this.m_projectToExport = m_projectToExport;
  }

  public boolean isExportInheritedEntries() {
    return m_exportInheritedEntries;
  }

  public void setExportInheritedEntries(boolean m_exportInheritedEntries) {
    this.m_exportInheritedEntries = m_exportInheritedEntries;
  }
}
