package org.eclipse.scout.sdk.docx4j.internal.operation;

import java.io.File;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;

import org.docx4j.openpackaging.parts.SpreadsheetML.WorksheetPart;
import org.eclipse.core.resources.IFolder;
import org.eclipse.core.runtime.CoreException;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.docx4j.XlsxAdapter;
import org.eclipse.scout.nls.sdk.internal.jdt.INlsFolder;
import org.eclipse.scout.nls.sdk.internal.jdt.NlsFolder;
import org.eclipse.scout.nls.sdk.model.util.Language;
import org.eclipse.scout.nls.sdk.model.workspace.NlsEntry;
import org.eclipse.scout.nls.sdk.model.workspace.project.INlsProject;
import org.eclipse.scout.nls.sdk.simple.model.ws.NlsType;
import org.eclipse.scout.nls.sdk.simple.model.ws.project.SimpleNlsProject;
import org.eclipse.scout.nls.sdk.simple.ui.dialog.language.TranslationFileNewModel;
import org.eclipse.scout.nls.sdk.ui.action.INewLanguageContext;
import org.eclipse.scout.sdk.docx4j.internal.Docx4jSdkActivator;
import org.eclipse.scout.sdk.docx4j.internal.model.NlsLine;
import org.eclipse.scout.sdk.operation.IOperation;
import org.eclipse.scout.sdk.util.log.ScoutStatus;
import org.eclipse.scout.sdk.util.typecache.IWorkingCopyManager;
import org.xlsx4j.sml.Cell;
import org.xlsx4j.sml.Row;
import org.xlsx4j.sml.SheetData;

public class ImportNlsFromExcelOperation implements IOperation {

  private String m_fileUrl;
  private INlsProject m_projectToImportTo;
  private HashSet<String> m_messages;

  @Override
  public String getOperationName() {
    return "Import NLS Entries from Excel.";
  }

  @Override
  public void validate() throws IllegalArgumentException {
    if (!StringUtility.hasText(getFileUrl())) {
      throw new IllegalArgumentException("No source file specified.");
    }
    if (getProjectToImportTo() == null) {
      throw new IllegalArgumentException("NLS Project may not be null.");
    }
  }

  @Override
  public void run(IProgressMonitor monitor, IWorkingCopyManager workingCopyManager) throws CoreException, IllegalArgumentException {
    try {
      m_messages = new HashSet<String>();
      String fileUrl = getFileUrl();
      String xlsxExtension = ".xlsx";
      if (!fileUrl.toLowerCase().endsWith(xlsxExtension)) {
        fileUrl += xlsxExtension;
      }

      XlsxAdapter xlsx = new XlsxAdapter(new File(fileUrl));
      List<WorksheetPart> sheets = xlsx.getSheets();
      if (sheets.size() > 0) {
        if (sheets.size() > 1) {
          m_messages.add("Multiple sheets have been found in the Excel file. The import used the first sheet only.");
        }
        List<NlsLine> lines = readLinesFromSheet(xlsx, sheets.get(0), monitor);
        if (monitor.isCanceled()) {
          return;
        }

        storeEntries(lines, monitor);
      }
      else {
        m_messages.add("No sheet has been found in the Excel file.");
      }
    }
    catch (Exception e) {
      throw new CoreException(new ScoutStatus(e));
    }
  }

  private void storeEntries(List<NlsLine> lines, IProgressMonitor monitor) {
    INlsProject p = getProjectToImportTo();
    for (NlsLine l : lines) {
      NlsEntry newEntry = new NlsEntry(l.getKey(), p);
      for (Entry<Language, String> e : l.getTranslations().entrySet()) {
        if (monitor.isCanceled()) {
          return;
        }

        if (!p.containsLanguage(e.getKey()) && p instanceof SimpleNlsProject) {
          // new language: add to the project
          NlsType txtProvSvc = ((SimpleNlsProject) p).getNlsType();
          IFolder fld = txtProvSvc.getType().getJavaProject().getProject().getFolder(txtProvSvc.getTranslationsFolderName());

          INewLanguageContext translationCreationContext = p.getTranslationCreationContext();
          TranslationFileNewModel model = (TranslationFileNewModel) translationCreationContext.getModel();
          model.setLanguageIso(e.getKey().getIsoCode());
          model.setFolder(new NlsFolder(fld, INlsFolder.TYPE_SIMPLE_FOLDER));
          translationCreationContext.execute(monitor); // create the new language
        }
        newEntry.addTranslation(e.getKey(), e.getValue());
      }
      p.updateRow(newEntry, monitor);
    }
  }

  private List<NlsLine> readLinesFromSheet(XlsxAdapter xlsx, WorksheetPart sheet, IProgressMonitor monitor) throws CoreException {
    ArrayList<NlsLine> lines = new ArrayList<NlsLine>();
    @SuppressWarnings("restriction")
    SheetData data = sheet.getJaxbElement().getSheetData();
    Map<Integer, Language> langIndexes = null;
    for (Row row : data.getRow()) {
      if (monitor.isCanceled()) {
        return null;
      }

      if (langIndexes == null || langIndexes.size() < 1) {
        // no languages found yet: parse header
        langIndexes = parseHeaderRow(xlsx, row);
      }
      else {
        // languages are defined: start reading entries
        NlsLine parsedLine = readRow(xlsx, row, langIndexes);
        if (parsedLine != null) {
          lines.add(parsedLine);
        }
      }
    }
    return lines;
  }

  private NlsLine readRow(XlsxAdapter xlsx, Row row, Map<Integer, Language> langIndexes) {
    NlsLine nlsLine = new NlsLine();
    for (Cell c : row.getC()) {
      Language l = langIndexes.get(XlsxAdapter.getColumnIndex(c));
      String cellContent = StringUtility.trim(xlsx.getCellValue(c));
      if (l != null && cellContent != null) {
        if (Language.LANGUAGE_KEY.equals(l)) {
          nlsLine.setKey(cellContent);
        }
        else {
          nlsLine.addTranslation(l, cellContent);
        }
      }
    }

    // validate
    if (!StringUtility.hasText(nlsLine.getKey())) {
      // no key was found
      return null;
    }
    if (!StringUtility.hasText(nlsLine.getTranslation(Language.LANGUAGE_DEFAULT))) {
      return null;
    }
    return nlsLine;
  }

  private Map<Integer, Language> parseHeaderRow(XlsxAdapter xlsx, Row headerRow) throws CoreException {
    HashMap<Integer, Language> languages = new HashMap<Integer, Language>();
    for (Cell cell : headerRow.getC()) {
      String cellContent = StringUtility.trim(xlsx.getCellValue(cell));
      int index = XlsxAdapter.getColumnIndex(cell);
      if (StringUtility.hasText(cellContent)) {
        if (Language.LANGUAGE_KEY.getIsoCode().equals(cellContent)) {
          languages.put(index, Language.LANGUAGE_KEY);
        }
        else if (Language.LANGUAGE_DEFAULT.getIsoCode().equals(cellContent)) {
          languages.put(index, Language.LANGUAGE_DEFAULT);
        }
        else if (!ExportNlsToExcelOperation.USAGE_HEADER_TEXT.equals(cellContent)) {
          Locale locale = null;
          String[] parts = cellContent.split("_");
          switch (parts.length) {
            case 1:
              locale = new Locale(parts[0]);
              break;
            case 2:
              locale = new Locale(parts[0], parts[1]);
              break;
            case 3:
              locale = new Locale(parts[0], parts[1], parts[3]);
              break;
          }
          if (locale == null) {
            throw new CoreException(new Status(IStatus.ERROR, Docx4jSdkActivator.PLUGIN_ID, "could not parse Iso language '" + cellContent + "'."));
          }
          else {
            languages.put(index, new Language(locale));
          }
        }
      }
    }
    return languages;
  }

  /**
   * @return the m_fileUrl
   */
  public String getFileUrl() {
    return m_fileUrl;
  }

  /**
   * @param m_fileUrl
   *          the m_fileUrl to set
   */
  public void setFileUrl(String m_fileUrl) {
    this.m_fileUrl = m_fileUrl;
  }

  /**
   * @return the m_projectToImportTo
   */
  public INlsProject getProjectToImportTo() {
    return m_projectToImportTo;
  }

  /**
   * @param m_projectToImportTo
   *          the m_projectToImportTo to set
   */
  public void setProjectToImportTo(INlsProject m_projectToImportTo) {
    this.m_projectToImportTo = m_projectToImportTo;
  }

}
