/*******************************************************************************
 * Copyright (c) 2010, 2014 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.rt.server.redis;

import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.exception.ProcessingException;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.rt.server.commons.cache.AbstractGlobalServerCacheService;
import org.osgi.framework.ServiceRegistration;

import redis.clients.jedis.Jedis;

public class RedisCacheService extends AbstractGlobalServerCacheService {
  private static final IScoutLogger LOG = ScoutLogManager.getLogger(RedisCacheService.class);
  private static final int MILLISECONDS_IN_SECONDS = 1000;
  private static final int DEFAULT_PORT = 6379;

  private Jedis m_jedis;
  private String m_host;
  private int m_port = DEFAULT_PORT;
  private String m_password;
  private boolean m_authEnabled = false;

  @Override
  public void initializeService(ServiceRegistration registration) {
    super.initializeService(registration);
    setClient();
  }

  public void setHost(String host) {
    m_host = host;
  }

  public void setPort(int port) {
    m_port = port;
  }

  public void setPassword(String password) {
    this.m_password = password;
  }

  public void setAuth(boolean enabled) {
    this.m_authEnabled = enabled;
  }

  @Override
  public void put(String key, Object value, Long expiration) {
    checkConnection();
    try {
      m_jedis.set(key, serializedString(value));
    }
    catch (ProcessingException e) {
      LOG.error("Unable to serialize value", e);
    }
    m_jedis.expire(key, toSeconds(expiration));
  }

  @Override
  public Object get(String key) {
    checkConnection();
    return deserialize((String) m_jedis.get(key));
  }

  @Override
  public Object getAndTouch(String key, Long expiration) {
    checkConnection();
    String str = (String) m_jedis.get(key);

    if (expiration != null) {
      m_jedis.expire(key, toSeconds(expiration));
    }

    if (str != null) {
      return deserialize(str);
    }
    return null;
  }

  @Override
  public void remove(String key) {
    checkConnection();
    m_jedis.del(key);
  }

  @Override
  public void touch(String key, Long expiration) {
    checkConnection();
    m_jedis.expire(key, toSeconds(expiration));
  }

  private Integer toSeconds(Long milliseconds) {
    Long ms = milliseconds / MILLISECONDS_IN_SECONDS;
    return ms.intValue();
  }

  protected void setClient() {
    try {
      m_jedis = new Jedis(m_host, m_port);
      if (m_authEnabled && !StringUtility.isNullOrEmpty(m_password)) {
        m_jedis.auth(m_password);
      }
    }
    catch (Exception e) {
      LOG.error("Unable to connet to Redis", e);
    }
  }

  protected void checkConnection() {
    if (!m_jedis.isConnected()) {
      setClient();
    }
  }

}
