/*******************************************************************************
 * Copyright (c) 2010, 2014 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.rt.server.rabbitmq;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.jobs.Job;
import org.eclipse.scout.commons.StringUtility;
import org.eclipse.scout.commons.job.JobEx;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.commons.serialization.IObjectSerializer;
import org.eclipse.scout.commons.serialization.SerializationUtility;
import org.eclipse.scout.rt.server.services.common.clustersync.IClusterNotificationMessage;
import org.eclipse.scout.rt.server.services.common.clustersync.IPubSubMessageListener;
import org.eclipse.scout.rt.server.services.common.clustersync.IPubSubMessageService;
import org.eclipse.scout.service.AbstractService;
import org.osgi.framework.ServiceRegistration;

import com.rabbitmq.client.Channel;
import com.rabbitmq.client.Connection;
import com.rabbitmq.client.ConnectionFactory;
import com.rabbitmq.client.QueueingConsumer;

public class RabbitMQMessageService extends AbstractService implements IPubSubMessageService {

  private static final IScoutLogger LOG = ScoutLogManager.getLogger(RabbitMQMessageService.class);
  public static final String NOTIFICATION_QUEUE_NAME = "scoutNotificationQueue";

  private IObjectSerializer m_objectSerializer;

  private String m_host;
  private String m_user;
  private String m_password;
  private String m_uri;
  private boolean m_authEnabled;
  private IPubSubMessageListener m_listener;
  private Job m_job;

  @Override
  public boolean subscribe(String queueName) {
    m_job = createMessageListenerJob();
    m_job.schedule();
    return true;
  }

  @Override
  public boolean unsubsribe(String queueName) {
    if (m_job != null) {
      m_job.cancel();
    }
    return true;
  }

  @Override
  public boolean publishNotification(IClusterNotificationMessage notification) {
    try {
      ConnectionFactory factory = new ConnectionFactory();
      factory.setUri(m_uri);
      Connection connection = factory.newConnection();
      Channel channel = connection.createChannel();
      channel.exchangeDeclare(NOTIFICATION_QUEUE_NAME, "fanout");
      channel.basicPublish(NOTIFICATION_QUEUE_NAME, "", null, m_objectSerializer.serialize(notification));
      channel.close();
      connection.close();
      return true;
    }
    catch (Exception e) {
      LOG.error("Unable to send message", e);
      return false;
    }
  }

  @SuppressWarnings("rawtypes")
  @Override
  public void initializeService(ServiceRegistration registration) {
    super.initializeService(registration);
    m_objectSerializer = SerializationUtility.createObjectSerializer();
    setUri();
  }

  protected void setUri() {
    if (!StringUtility.hasText(m_host) && !StringUtility.hasText(m_uri)) {
      throw new IllegalArgumentException("Host or UIR for RabbitMQ has to be set in config.ini");
    }

    if (!StringUtility.hasText(m_uri)) {
      if (m_authEnabled) {
        if (!StringUtility.hasText(m_user)) {
          throw new IllegalArgumentException("Host for RabbitMQ has to be set in config.ini");
        }
        if (!StringUtility.hasText(m_password)) {
          throw new IllegalArgumentException("Host for RabbitMQ has to be set in config.ini");
        }
        m_uri = "amqp://" + m_user + ":" + m_password + "@" + m_host;
      }
      else {
        m_uri = "amqp://" + m_host;
      }
    }
  }

  protected Job createMessageListenerJob() {
    return new JobEx("RabbitMQMessageListener Job") {
      @Override
      protected IStatus run(IProgressMonitor monitor) {
        try {
          ConnectionFactory factory = new ConnectionFactory();
          factory.setUri(m_uri);
          Connection connection = factory.newConnection();
          Channel incomingChannel = connection.createChannel();

          incomingChannel.exchangeDeclare(RabbitMQMessageService.NOTIFICATION_QUEUE_NAME, "fanout");
          String queueName = incomingChannel.queueDeclare().getQueue();
          incomingChannel.queueBind(queueName, RabbitMQMessageService.NOTIFICATION_QUEUE_NAME, "");

          QueueingConsumer consumer = new QueueingConsumer(incomingChannel);
          incomingChannel.basicConsume(queueName, true, consumer);

          while (true) {
            tryHandleMessage(consumer);
          }

        }
        catch (Exception e) {
          LOG.error("Unable to set up message listener", e);
        }
        return Status.OK_STATUS;
      }

      private void tryHandleMessage(QueueingConsumer consumer) {
        try {
          QueueingConsumer.Delivery delivery = consumer.nextDelivery();
          byte[] message = delivery.getBody();
          IClusterNotificationMessage notificationMessage = m_objectSerializer.deserialize(message, IClusterNotificationMessage.class);
          if (m_listener != null) {
            m_listener.onMessage(notificationMessage);
          }
        }
        catch (Exception e) {
          LOG.error("Unable to read incoming message", e);
        }
      }
    };
  }

  public void setHost(String host) {
    this.m_host = host;
  }

  public void setUser(String user) {
    this.m_user = user;
  }

  public void setPassword(String password) {
    this.m_password = password;
  }

  public void setUri(String uri) {
    this.m_uri = uri;
  }

  public void setAuth(boolean enabled) {
    this.m_authEnabled = enabled;
  }

  @Override
  public void setListener(IPubSubMessageListener listener) {
    m_listener = listener;
  }

  @Override
  public IPubSubMessageListener getListener() {
    return m_listener;
  }
}
