/*******************************************************************************
 * Copyright (c) 2010, 2014 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.rt.server.memcached;

import net.spy.memcached.AddrUtil;
import net.spy.memcached.ConnectionFactory;
import net.spy.memcached.ConnectionFactoryBuilder;
import net.spy.memcached.MemcachedClient;
import net.spy.memcached.MemcachedClientIF;
import net.spy.memcached.auth.AuthDescriptor;
import net.spy.memcached.auth.PlainCallbackHandler;

import org.eclipse.scout.commons.exception.ProcessingException;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.rt.server.commons.cache.AbstractGlobalServerCacheService;
import org.osgi.framework.ServiceRegistration;

/**
 * This class stores and loads data inside a memcached instance
 */
public class MemcachedCacheService extends AbstractGlobalServerCacheService {

  private static final IScoutLogger LOG = ScoutLogManager.getLogger(MemcachedCacheService.class);
  private static final int DEFAULT_PORT = 11211;
  private static final int MILLISECONDS_IN_SECONDS = 1000;

  private MemcachedClientIF m_client;

  private String m_url;
  private Integer m_port = DEFAULT_PORT;
  private String m_username;
  private String m_password;
  private boolean m_authEnabled;

  @Override
  public void initializeService(ServiceRegistration registration) {
    super.initializeService(registration);
    m_client = createMemcachedClient();
  }

  public void setPort(Integer port) {
    m_port = port;
  }

  public void setUrl(String url) {
    m_url = url;
  }

  public void setUsername(String username) {
    this.m_username = username;
  }

  public void setPassword(String password) {
    this.m_password = password;
  }

  public void setAuth(boolean enabled) {
    this.m_authEnabled = enabled;
  }

  @Override
  public void put(String key, Object value, Long expiration) {
    try {
      m_client.set(key, toSeconds(expiration), serializedString(value));
    }
    catch (ProcessingException e) {
      LOG.error("Unable to serialize vaue", e);
    }
  }

  @Override
  public Object getAndTouch(String key, Long expiration) {
    Object v = m_client.get(key);
    if (v != null) {
      m_client.touch(key, toSeconds(expiration));
      return deserialize((String) v);
    }
    return null;
  }

  @Override
  public Object get(String key) {
    return deserialize((String) m_client.get(key));
  }

  @Override
  public void remove(String key) {
    m_client.delete(key);
  }

  @Override
  public void touch(String key, Long expiration) {
    m_client.touch(key, toSeconds(expiration));
  }

  private Integer toSeconds(Long milliseconds) {
    Long ms = milliseconds / MILLISECONDS_IN_SECONDS;
    return ms.intValue();
  }

  protected MemcachedClientIF createMemcachedClient() {
    try {
      ConnectionFactory c;
      // allow auth to be disabled for local development
      if (m_authEnabled && m_username != null && m_password != null) {
        LOG.info("Using authentication with memcache");
        AuthDescriptor ad = new AuthDescriptor(new String[]{"PLAIN"}, new PlainCallbackHandler(m_username, m_password));
        c = new ConnectionFactoryBuilder().setProtocol(ConnectionFactoryBuilder.Protocol.BINARY).setAuthDescriptor(ad).build();
      }
      else {
        LOG.info("Using no authentication with memcache");
        c = new ConnectionFactoryBuilder().setProtocol(ConnectionFactoryBuilder.Protocol.BINARY).build();
      }
      if (m_url == null) {
        String configIniSuggestion = "Add \norg.eclipse.scout.rt.server.memcached.MemcachedCacheService#url=<URL>\norg.eclipse.scout.rt.server.memcached.MemcachedCacheService#port=<PORT>\nto your config.ini file!";
        LOG.error("url not defined in MemcachedCacheService.\n" + configIniSuggestion);
      }
      else {
        return new MemcachedClient(c, AddrUtil.getAddresses(m_url + ":" + m_port));
      }
    }
    catch (Exception e) {
      LOG.error("Couldn't create a connection to memcached", e);
    }
    return null;
  }
}
