/*******************************************************************************
 * Copyright (c) 2010, 2014 BSI Business Systems Integration AG.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     BSI Business Systems Integration AG - initial API and implementation
 ******************************************************************************/
package org.eclipse.scout.rt.server.activemq;

import java.io.IOException;
import java.util.Properties;

import javax.jms.BytesMessage;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageListener;
import javax.jms.Session;
import javax.jms.Topic;
import javax.jms.TopicConnection;
import javax.jms.TopicConnectionFactory;
import javax.jms.TopicPublisher;
import javax.jms.TopicSession;
import javax.jms.TopicSubscriber;
import javax.naming.Context;
import javax.naming.NamingException;
import javax.naming.directory.InitialDirContext;

import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;
import org.eclipse.scout.commons.serialization.IObjectSerializer;
import org.eclipse.scout.commons.serialization.SerializationUtility;
import org.eclipse.scout.rt.server.services.common.clustersync.IClusterNotificationMessage;
import org.eclipse.scout.rt.server.services.common.clustersync.IPubSubMessageListener;
import org.eclipse.scout.rt.server.services.common.clustersync.IPubSubMessageService;
import org.eclipse.scout.service.AbstractService;

public class ActiveMQMessageService extends AbstractService implements IPubSubMessageService {

  private static final IScoutLogger LOG = ScoutLogManager.getLogger(ActiveMQMessageService.class);

  private IObjectSerializer m_objectSerializer;
  private static final int DEFAULT_PORT = 61616;

  private String m_topicName = "scout.messages";
  private String m_initialContextFactory = "org.apache.activemq.jndi.ActiveMQInitialContextFactory";
  private String m_host;
  private Integer m_port = DEFAULT_PORT;
  private String m_username;
  private String m_password;
  private boolean m_auth;

  private TopicConnectionFactory m_fact = null;
  private Topic m_topic = null;
  private TopicConnection m_connection;
  private TopicSession m_session;
  private TopicSubscriber m_subscriber;
  private TopicPublisher m_publisher;

  private boolean m_checkForMessages;
  private IPubSubMessageListener m_listener;

  public void setHost(String host) {
    m_host = host;
  }

  public void setPort(Integer port) {
    m_port = port;
  }

  public void setUsername(String username) {
    this.m_username = username;
  }

  public void setPassword(String password) {
    this.m_password = password;
  }

  public void setAuth(boolean auth) {
    this.m_auth = auth;
  }

  private class NotificationListener implements MessageListener {

    @Override
    public void onMessage(Message message) {
      try {
        if (message instanceof BytesMessage) {
          BytesMessage bm = (BytesMessage) message;
          byte[] data = new byte[(int) bm.getBodyLength()];
          bm.readBytes(data);
          if (m_listener != null) {
            m_listener.onMessage(m_objectSerializer.deserialize(data, IClusterNotificationMessage.class));
          }
        }
      }
      catch (JMSException e) {
        LOG.error("Error receiving message", e);
      }
      catch (ClassNotFoundException e) {
        LOG.error("Error receiving message", e);
      }
      catch (IOException e) {
        LOG.error("IO Error receiving message", e);
      }
    }

  }

  @Override
  public boolean subscribe(String queueName) {

    m_objectSerializer = SerializationUtility.createObjectSerializer();

    // Initialise JNDI properties
    Properties properties = new Properties();
    properties.put("java.naming.factory.initial", m_initialContextFactory);
    properties.put("connectionfactory.QueueConnectionFactory", "tcp://" + m_host + ":" + m_port); // "tcp://localhost:61616"
    properties.put("topic." + m_topicName, m_topicName);

    try {
      LOG.info("Initialising JNDI... ");
      Context ctx = new InitialDirContext(properties);
      LOG.info("Obtaining TCF from JNDI... ");
      m_fact = (TopicConnectionFactory) ctx.lookup("QueueConnectionFactory");
      LOG.info("Obtaining topic T from JNDI... ");
      m_topic = (Topic) ctx.lookup(m_topicName);

      // Create connection
      m_connection = createConnection();

      // Start connection
      m_connection.start();
      // Session
      m_session = m_connection.createTopicSession(false, Session.AUTO_ACKNOWLEDGE);

      // Create a topic dynamically
      Topic t = m_session.createTopic(m_topicName);
      // Publisher
      m_publisher = m_session.createPublisher(t);
      // Subscriber
      m_subscriber = m_session.createSubscriber(t);
      m_subscriber.setMessageListener(new NotificationListener());
    }
    catch (NamingException e) {
      LOG.error("Unable to initialise context", e);
      return false;
    }
    catch (JMSException e) {
      LOG.error("JMS Error", e);
      return false;
    }
    return true;
  }

  private TopicConnection createConnection() throws JMSException {
    if (m_auth) {
      return m_fact.createTopicConnection(m_username, m_password);
    }
    return m_fact.createTopicConnection();
  }

  @Override
  public boolean unsubsribe(String queueName) {
    m_checkForMessages = false;
    try {
      // Close publishers and subscribers
      m_publisher.close();
      m_subscriber.close();

      // Close session and connection
      m_session.close();
      m_connection.close();
    }
    catch (JMSException e) {
      LOG.error("Unable to close jms connection", e);
      return false;
    }

    return true;
  }

  @Override
  public boolean publishNotification(IClusterNotificationMessage notification) {
    try {
      BytesMessage message = m_session.createBytesMessage();
      message.writeBytes(m_objectSerializer.serialize(notification));
      m_publisher.publish(message);
    }
    catch (JMSException e) {
      LOG.error("Unable to send message", e);
    }
    catch (IOException e) {
      LOG.error("Unable to send message", e);
    }
    return false;
  }

  @Override
  public void setListener(IPubSubMessageListener listener) {
    m_listener = listener;
  }

  @Override
  public IPubSubMessageListener getListener() {
    return m_listener;
  }

}
