/*
 * Copyright 2001-2004 The Apache Software Foundation.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.log4j;

import java.util.Enumeration;

import org.apache.log4j.helpers.NullEnumeration;
import org.eclipse.scout.commons.logger.IScoutLogger;
import org.eclipse.scout.commons.logger.ScoutLogManager;

/**
 * <p>
 * This class is a minimal implementation of the original <code>org.apache.log4j.Category</code> class (as found in
 * log4j 1.2) by delegation of all calls to a {@link org.eclipse.scout.commons.logger.ScoutLogManager} instance.
 * </p>
 * 
 * @author S&eacute;bastien Pennec
 * @author Ceki G&uuml;lc&uuml;
 * @author Matthias Villiger
 */
@SuppressWarnings("rawtypes")
public class Category {

  protected final IScoutLogger logger;

  Category(String name) {
    logger = ScoutLogManager.getLogger(name);
  }

  public static Category getInstance(Class clazz) {
    return Log4jLoggerFactory.getLogger(clazz.getName());
  }

  public static Category getInstance(String name) {
    return Log4jLoggerFactory.getLogger(name);
  }

  public String getName() {
    return logger.getName();
  }

  public Enumeration getAllAppenders() {
    return NullEnumeration.getInstance();
  }

  /**
   * Return the level in effect for this category/logger.
   * <p>
   * The result is computed by simulation.
   * 
   * @return
   */
  public Level getEffectiveLevel() {
    return scoutLevelToLog4jLevel(logger.getLevel());
  }

  /**
   * Returns the assigned {@link Level}, if any, for this Category.
   * 
   * @return Level - the assigned Level, can be <code>null</code>.
   */
  final public Level getLevel() {
    return scoutLevelToLog4jLevel(logger.getLevel());
  }

  public void setLevel(Level level) {
    logger.setLevel(log4jPriorityToScoutLevel(level));
  }

  /**
   * @deprecated Please use {@link #getLevel} instead.
   */
  @Deprecated
  final public Level getPriority() {
    return getLevel();
  }

  /**
   * Delegates to Scout Logger
   */
  public boolean isDebugEnabled() {
    return logger.isDebugEnabled();
  }

  /**
   * Delegates to Scout Logger
   */
  public boolean isInfoEnabled() {
    return logger.isInfoEnabled();
  }

  /**
   * Delegates to Scout Logger
   */
  public boolean isWarnEnabled() {
    return logger.isWarnEnabled();
  }

  /**
   * Delegates to Scout Logger
   */
  public boolean isErrorEnabled() {
    return logger.isErrorEnabled();
  }

  /**
   * Determines whether the priority passed as parameter is enabled in the
   * underlying Scout Logger.
   * 
   * @param p
   *          the priority to check against
   * @return true if this logger is enabled for the given level, false
   *         otherwise.
   */
  public boolean isEnabledFor(Priority p) {
    return logger.isLoggable(log4jPriorityToScoutLevel(p));
  }

  /**
   * Delegates to Scout Logger
   */
  public void debug(Object message) {
    log(IScoutLogger.LEVEL_DEBUG, convertToString(message), null);
  }

  /**
   * Delegates to Scout Logger
   */
  public void debug(Object message, Throwable t) {
    log(IScoutLogger.LEVEL_DEBUG, convertToString(message), t);
  }

  /**
   * Delegates to Scout Logger
   */
  public void info(Object message) {
    log(IScoutLogger.LEVEL_INFO, convertToString(message), null);
  }

  /**
   * Delegates to Scout Logger
   */
  public void info(Object message, Throwable t) {
    log(IScoutLogger.LEVEL_INFO, convertToString(message), t);
  }

  /**
   * Delegates to Scout Logger
   */
  public void warn(Object message) {
    log(IScoutLogger.LEVEL_WARN, convertToString(message), null);
  }

  /**
   * Delegates to Scout Logger
   */
  public void warn(Object message, Throwable t) {
    log(IScoutLogger.LEVEL_WARN, convertToString(message), t);
  }

  /**
   * Delegates to Scout Logger
   */
  public void error(Object message) {
    log(IScoutLogger.LEVEL_ERROR, convertToString(message), null);
  }

  /**
   * Delegates to Scout Logger
   */
  public void error(Object message, Throwable t) {
    log(IScoutLogger.LEVEL_ERROR, convertToString(message), t);
  }

  /**
   * Delegates to Scout Logger
   */
  public void fatal(Object message) {
    log(IScoutLogger.LEVEL_ERROR, convertToString(message), null);
  }

  /**
   * Delegates to Scout Logger
   */
  public void fatal(Object message, Throwable t) {
    log(IScoutLogger.LEVEL_ERROR, convertToString(message), t);
  }

  public void log(String FQCN, Priority p, Object message, Throwable t) {
    log(log4jPriorityToScoutLevel(p), convertToString(message), t);
  }

  public void log(Priority p, Object message, Throwable t) {
    log(log4jPriorityToScoutLevel(p), convertToString(message), t);
  }

  public void log(Priority p, Object message) {
    log(log4jPriorityToScoutLevel(p), convertToString(message), null);
  }

  protected String convertToString(Object message) {
    return message == null ? null : message.toString();
  }

  protected Level scoutLevelToLog4jLevel(int scoutLogLevel) {
    switch (scoutLogLevel) {
      case IScoutLogger.LEVEL_OFF:
        return Level.OFF;
      case IScoutLogger.LEVEL_WARN:
        return Level.WARN;
      case IScoutLogger.LEVEL_INFO:
        return Level.INFO;
      case IScoutLogger.LEVEL_DEBUG:
        return Level.DEBUG;
      case IScoutLogger.LEVEL_TRACE:
        return Level.TRACE;
      default:
        return Level.ERROR;
    }
  }

  protected int log4jPriorityToScoutLevel(Priority p) {
    if (p == null) {
      return IScoutLogger.LEVEL_ERROR;
    }

    switch (p.level) {
      case Priority.OFF_INT:
        return IScoutLogger.LEVEL_OFF;
      case Priority.WARN_INT:
        return IScoutLogger.LEVEL_WARN;
      case Priority.INFO_INT:
        return IScoutLogger.LEVEL_INFO;
      case Priority.DEBUG_INT:
        return IScoutLogger.LEVEL_DEBUG;
      case Priority.ALL_INT:
        return IScoutLogger.LEVEL_TRACE;
      default:
        return IScoutLogger.LEVEL_ERROR;
    }
  }

  protected void log(int scoutLevel, String message, Throwable t) {
    if (!logger.isLoggable(scoutLevel) || scoutLevel == IScoutLogger.LEVEL_OFF || (message == null && t == null)) {
      return;
    }

    switch (scoutLevel) {
      case IScoutLogger.LEVEL_WARN:
        logger.warn(message, t);
      case IScoutLogger.LEVEL_INFO:
        logger.info(message, t);
      case IScoutLogger.LEVEL_DEBUG:
        logger.debug(message, t);
      case IScoutLogger.LEVEL_TRACE:
        logger.trace(message, t);
      default:
        logger.error(message, t);
    }
  }
}
